//
// Copy example for OpenPTC 1.0 C++ Implementation
// Copyright (c) Glenn Fiedler (ptc@gaffer.org)
// This source code is in the public domain
//

// include files
#include "ptc.h"
#include <math.h>
#include <stdlib.h>
#include <fstream.h>


inline int random(int max)
{
    // random number
    return rand() % max;
}


void load(Surface &surface,char filename[])
{
    // open image file
    ifstream is(filename,ios::in|ios::binary|ios::nocreate);

    // check image file stream
    if (!is) throw Error("could not load image");

    // skip header
    is.seekg(18);

    // get surface dimensions
    const int width  = surface.width();
    const int height = surface.height();
    
    // allocate image pixels
    char8 *pixels = new char8[width*height*3];

    // read image pixels one line at a time
    for (int y=height-1; y>=0; y--) is.read((char*)&pixels[width*y*3],width*3);

    // load pixels to surface
    surface.load(pixels,width,height,width*3,Format(24,0x00FF0000,0x0000FF00,0x000000FF),Palette());

    // free image pixels
    delete[] pixels;
}


// image tile
struct Tile
{
    float x;               // current x position
    float y;               // current y position
    float cx;              // central x position
    float cy;              // central y position
    float dx;              // x movement phase shift
    float dy;              // y movement phase shift
    float sx;              // x movement amplitude multiplier
    float sy;              // y movement applitude multiplier
    float tx;              // x movement time multiplier
    float ty;              // y movement time multiplier
    Surface *surface;      // tile surface
};



int APIENTRY WinMain(HINSTANCE hInst,HINSTANCE hPrevInst,LPSTR lpCmdLine,int nCmdShow)
{
    try
    {
        // create console
        Console console;

        // create format
        Format format(32,0x00FF0000,0x0000FF00,0x000000FF);
 
        // open the console matching the image resolution
        console.open("Copy example",320,200,format);

        // create drawing surface
        Surface surface(320,200,format);

        // create image surface
        Surface image(320,200,format);

        // load image to surface
        load(image,"Image.tga");

        // locals
        int i=0;
        int x=0;
        int y=0;

        // initialize surface tiles
        Tile tile[40];
        for (i=0,y=0; y<200; y+=40)
        {
            for (x=0; x<320; x+=40)
            {
                // create tile surface
                tile[i].surface = new Surface(40,40,format);

                // copy section of image to tile surface
                image.copy(*tile[i].surface,Area(x,y,x+40,y+40),Area(0,0,40,40));
        
                // setup tile parameters
                tile[i].cx = (float) x;
                tile[i].cy = (float) y;
                tile[i].dx = (float) random(1000) / 200.0f + 50.0f;
                tile[i].dy = (float) random(1000) / 200.0f + 50.0f;
                tile[i].sx = (float) random(1000) / 500.0f + 0.5f;
                tile[i].sy = (float) random(1000) / 700.0f + 0.5f;
                tile[i].tx = (float) random(1000) / 500.0f + 1.0f;
                tile[i].ty = (float) random(1000) / 500.0f + 1.0f;

                // next tile
                i++;
            }
        }

        // time
        float t  = 0.0f;
        float dt = 0.01f;

        // loop until a key is pressed
        while (!console.key())
        {
            // clear surface
            surface.clear();

            // update tile positions
            for (i=0; i<40; i++)
            {
                // tile energy
                float energy = 50.0;
        
                // calculate current tile positions
                tile[i].x = tile[i].cx + ( energy * tile[i].sx * (float) sin( t*tile[i].tx + tile[i].dx ) ); 
                tile[i].y = tile[i].cy + ( energy * tile[i].sy * (float) sin( t*tile[i].ty + tile[i].dy ) ); 

                // draw tile on surface
                tile[i].surface->copy(surface,tile[i].surface->area(),Area((int)tile[i].x,(int)tile[i].y,(int)tile[i].x+40,(int)tile[i].y+40));
            }

            // copy surface to console
            surface.copy(console);
    
            // update console
            console.update();
        
            // time
            t += dt;
        }
    }
    catch (Error &error)
    {
        // report error
        error.report();
    }

    // exit
    return 0;
}
