// PNG file handler
//
// Copyright (c) 1999 Dan Brown (_danbrown_@yahoo.com)
//
// Part of the Titan 1.1.x image handling library for PTC
// (http://now.at/Titan)
//
// This source code is licensed under the GNU GPL
//
// This is a wrapper for libpng
//

#include "../titan.h"

#ifdef USE_PNG

#include <memory.h>
extern "C" {
#include <png.h>     // include official libpng header
}

#include "png.h"     // include our PNGLoader header

PNGHandler::PNGHandler()
{
    // defaults
    Defaults();    
}


PNGHandler::PNGHandler(char *filename)
{
    // defaults
    Defaults();

    m_filename = filename;

    // assign
    m_imagefile=fopen(m_filename, "rb+");
    if (m_imagefile == 0)
    {
      throw Error("Titan error - File does not exist");
    }

    // read PNG header
    fread(m_header, 8, 1, m_imagefile);

    // Is file a PNG?
    if (!valid())
    {
      fclose(m_imagefile);
      return;
    }

    // Create libpng structures
    // FIXME check return values
    png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    info_ptr = png_create_info_struct(png_ptr);
    end_ptr = png_create_info_struct(png_ptr);

    // Did they get created properly?
    if ((png_ptr == NULL) || (info_ptr == NULL) || (end_ptr == NULL))
    {
      throw Error("Titan error - Could not allocate PNG structures");
    }

    // Pass file handle to libpng
    png_init_io(png_ptr, m_imagefile);

    // Tell libpng we have already read the 8 byte header
    png_set_sig_bytes(png_ptr, 8);

    // Get image details (height, width, bpp, etc)
    png_read_info(png_ptr, info_ptr);

    // Setup width and height
    m_width = png_get_image_width(png_ptr, info_ptr);
    m_height = png_get_image_height(png_ptr, info_ptr);

    // Bits per plane
    int32 bpplane = png_get_bit_depth(png_ptr, info_ptr);

    // image type
    int32 itype = png_get_color_type(png_ptr, info_ptr);

    // number of planes
    int32 planes = 0;

    switch (itype)
    {
      case PNG_COLOR_TYPE_GRAY       : planes = 0; break;  // don't support
      case PNG_COLOR_TYPE_GRAY_ALPHA : planes = 0; break;  // gray scale yet
      case PNG_COLOR_TYPE_PALETTE    : planes = 1; break;
      case PNG_COLOR_TYPE_RGB        : planes = 3; break;
      case PNG_COLOR_TYPE_RGB_ALPHA  : planes = 4; break;
    }

    // PNG files store data in BGR order, we want RGB
    if ((planes == 3) || (planes == 4))
    {
      png_set_bgr(png_ptr);
    }

    if (planes == 0)
    {
      // grayscale
      throw Error("Titan error - Gray scale PNG files not supported yet");
    }

    // Get bits per pixel
    int bpp = bpplane * planes;

    // setup format using bits per pixel
    switch (bpp)
    {
        case 8:  m_format = Format(8); break;
        case 16: m_format = Format(16, 16<<10, 32<<5, 16);       break;
        case 24: m_format = Format(24, 0xff<<16, 0xff<<8, 0xff); break;
        case 32: m_format = Format(32, 0xff<<16, 0xff<<8, 0xff); break;
    }

    // setup palette flag
    if (planes == 1)
    {
      m_paletteflag = true;
    }

    // If we changed any image settings, let libpng update itself
    png_read_update_info(png_ptr, info_ptr);
}

PNGHandler::~PNGHandler()
{
    // avoid warnings
    if (m_imagefile);
}

int PNGHandler::info(int32 &width,int32 &height,Format &format,int32 &palette)
{
    // setup info
    width=m_width;
    height=m_height;
    format=m_format;
    palette=m_paletteflag;
    return 1;
}


int PNGHandler::load(void *image, Palette *palette)
{
    // read palette
    if (palette && m_paletteflag)
    {
        // setup temp palette buffer
        png_color *temp=new png_color[256];
        if (!temp) throw(Error("Titan error - Could not allocate a palette"));

        // How many palette entries are there?
        int num_colours;

        // read palette into temp buffer
        png_get_PLTE(png_ptr, info_ptr, &temp, &num_colours);

        // setup another temporary palette buffer
        int32 *pal=new(int32[256]);

        // convert colors to PTC palette format
        for (int32 loop = 0; loop < (int32)num_colours; loop++)
        {
          char8 r = temp[loop].red;
          char8 g = temp[loop].green;
          char8 b = temp[loop].blue;
          pal[loop] = r<<16 | g<<8 | b;
        }

        // copy palette to PTC Surface palette
        palette->load(pal);

        // deallocate memory
//        delete temp;
	delete pal;
    }

    // read image data
    if (image)
    {
      // Set up a table of row pointers
      int32 *row_pointers = new int32 [m_height*4];

      // How many bytes in a row?
      int32 rowbytes = (m_width*m_format.bytes());

      // make a table of pointers to each row of the Surface
      for (int32 loop=0; loop < (m_height); loop++)
      {
        row_pointers[loop] = (int32)image+(loop*rowbytes);
      }

      // read the image
      png_read_image(png_ptr, (char8 **)row_pointers);
      // That was hard :)

      // deallocate memory
      delete row_pointers;
    }

    // Finished reading the file
    png_read_end(png_ptr, info_ptr);

    // Remove the png structures
    png_destroy_read_struct(&png_ptr, &info_ptr, &end_ptr);

    // Close the file
    fclose(m_imagefile);

    // success
    return 1;
}


int PNGHandler::valid()
{
  return !(png_sig_cmp(m_header, 0, 7));
}

int PNGHandler::save(char *filename, int32 width, int32 height, 
					 Format *format, Palette *palette, void *pixels, 
					 void *params)
{
  PNGParams userparams;
  if (params == NULL)
  {
    PNGParamDefaults(userparams);          // if no parameters given, use
  } else                                   // defaults
  {
    userparams = *(PNGParams *)params;     // else use given parameters
  }

  // Check for unsupported options
  if (userparams.interlaced == TRUE)
  {
    throw Error("Titan error - Cannot currently save interlaced PNG files");
  }

  // options are OK
  m_filename = filename;
  m_width = width;
  m_height = height;
  m_format = *format;
  m_paletteflag = (m_format.bytes() == 1);

  // create file
  m_imagefile = fopen(m_filename, "wb");
  if (!m_imagefile)
  {
    throw Error("Titan error - Cannot create PNG file");
  }

  // Create save structure
  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  if (!png_ptr)
  {
    fclose(m_imagefile);
    throw Error("Titan error - Cannot create PNG save structure");
  }

  // Create info struct
  info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr)
  {
    png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
    fclose(m_imagefile);
    throw Error("Titan error - Cannot create PNG save info structure");
  }

  // Set up libpng error handler
  if (setjmp(png_ptr->jmpbuf))
  {
    png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
    fclose(m_imagefile);
    throw Error("Titan error - Cannot create PNG save error handler");
  }

  // Tell libpng to write to our file
  png_init_io(png_ptr, m_imagefile);

  // Set up libpng to write header
  int palettetype;
  if (m_paletteflag == TRUE)
  {
    palettetype = PNG_COLOR_TYPE_PALETTE;
  } else if (format->bytes() == 3)
  {
    palettetype = PNG_COLOR_TYPE_RGB;
  } else
  {
    palettetype = PNG_COLOR_TYPE_RGB_ALPHA;
  }
  png_set_IHDR(png_ptr,
               info_ptr,
               m_width,
               m_height,
               8,                       // Bit depth per channel (24-bit = 8 per channel)
               palettetype,
               PNG_INTERLACE_NONE,
               PNG_COMPRESSION_TYPE_DEFAULT,
               PNG_FILTER_TYPE_DEFAULT);

// TODO Set palette to save
  // Save palette
  if (m_paletteflag == TRUE)
  {
    //Convert palette to libpng format
    char8 *pal = (char8 *)palette->lock();
    png_color newpalette[256];
    
    for (int loop=0; loop < (256*4); loop+=4)
    {
      newpalette[loop/4].red = pal[loop+2];
      newpalette[loop/4].green = pal[loop+1];
      newpalette[loop/4].blue = pal[loop];
    }
    palette->unlock();
    
    // Tell libpng to use this palette
    png_set_PLTE(png_ptr, info_ptr, newpalette, 256);
  }

  // TODO Add comment
//  png_set_text();

  // Write header
  png_write_info(png_ptr, info_ptr);

  // PNG is stores data in network byte order, PCs don't, so swap formats.
  // TODO check which format the pixels are in
  png_set_bgr(png_ptr);

  // Output the header so far
  png_write_flush(png_ptr);

  // Set up a table of row pointers
  int32 *row_pointers = new int32 [m_height*4];

  // How many bytes in a row?
  int32 rowbytes = (m_width*m_format.bytes());

  // make a table of pointers to each row of the Surface
  for (int32 loop=0; loop < (m_height); loop++)
  {
    row_pointers[loop] = (int32)pixels+(loop*rowbytes);
  }

  // write the image
  png_write_image(png_ptr, (char8 **)row_pointers);
  // That was hard :)

  // deallocate memory
  delete row_pointers;

  // finish writing file
  png_write_end(png_ptr, info_ptr);

  // Free memory
  png_destroy_write_struct(&png_ptr, &info_ptr);

  fclose(m_imagefile);

  return 0;
}

void PNGHandler::Defaults()
{
    // defaults
    m_width=0;
    m_height=0;
    m_paletteflag=0;
    m_imagefile=NULL;
    memset(m_header,0,8);
}

#endif
