// PCX file handler
//
// Copyright (c) Dan Brown 1999 (_danbrown_@yahoo.com)
//
// Part of the Titan 1.1.x image handling library for PTC
// (http://now.at/Titan)
//
// This source code is licensed under the GNU GPL
//

#include "../titan.h"

#ifdef USE_PCX

#include <memory.h>
#include "pcx.h"

PCXHandler::PCXHandler()
{
    // defaults
    Defaults();    
}


PCXHandler::PCXHandler(char *filename)
{
    // defaults
    Defaults();

    m_filename = filename;

    // assign
    m_imagefile=fopen(m_filename, "rb+");
    if (m_imagefile == 0)
    {
      throw Error("Titan error - File does not exist");
    }

    // read pcx header
    fread(m_header, 128, 1, m_imagefile);

    // close file
    fclose(m_imagefile);

    // setup width
    m_width=m_header[9];
    m_width<<=8;
    m_width+=m_header[8];
    m_temp=m_header[5];
    m_temp<<=8;
    m_temp+=m_header[4];
    m_width-=m_temp;
    m_width+=1;

    // setup height
    m_height=m_header[11];
    m_height<<=8;
    m_height+=m_header[10];
    m_temp=m_header[7];
    m_temp<<=8;
    m_temp+=m_header[6];
    m_height-=m_temp;
    m_height+=1;

    // setup format depending on planes - Should be bits per pixel as well
    switch (m_header[65])
    {
        case 3:  m_format = Format(24, 0xff<<16, 0xff<<8, 0xff);   break;    // 3 planes - 24 bit
                 // no 16-bit variation of .PCX
        case 1:  m_format = Format(8);
                 m_paletteflag = 1; break;
    }
}


PCXHandler::~PCXHandler()
{
}


int PCXHandler::info(int32 &width,int32 &height,Format &format,int32 &palette)
{
    // setup info
    width=m_width;
    height=m_height;
    format=m_format;
    palette=m_paletteflag;
    return 1;
}


int PCXHandler::load(void *image, Palette *palette)
{
    // check there is an image buffer to write to
    if (!image)
    {
      throw Error("Titan error - no pixels to load image onto");
    }

    // open file
    m_imagefile=fopen(m_filename, "rb+");

    // try to load palette for 8bpp picture
    if (m_paletteflag)
    {
      // setup temp palette buffer
      int32 temp[256];
  
      fseek(m_imagefile, -769, SEEK_END);
      // check there is a palette there
      char8 palcheck;
      fread(&palcheck, 1,1, m_imagefile);
      if (palcheck != 0x0c)
      {
        throw Error("Titan error - no palette in 8-bit .PCX file");
      }
      // read into temp buffer
      for (int paletteloop=0; paletteloop<=255; paletteloop++)
      {
        char8 r;
	char8 g;
	char8 b;
	fread(&r, 1,1, m_imagefile);
        fread(&g, 1,1, m_imagefile);
        fread(&b, 1,1, m_imagefile);
	temp[paletteloop] = r<<16 | g<<8 | b;
      }
      palette->load(temp);
    }
  
    // read image data
    if (image)
    {
      // skip header
      fseek(m_imagefile, 128, SEEK_SET);
          
      // read image data
      int line_size=m_width*m_format.bytes();
      int pcx_line_size=m_header[67];
      pcx_line_size<<=8;
      pcx_line_size += m_header[66];
      pcx_line_size *= m_header[65];    // how many bytes per PCX scan line

      char8 *scanbuffer = new char8[pcx_line_size];
      memset(scanbuffer, 0, pcx_line_size);   // allocate a temporary buffer
	
      char8 inbyte;
      char8 length;
      char8 *bufptr = scanbuffer;
      int readin = 0;
	
      for (int32 y=0; y<m_height; y++)
      {
        while (readin < pcx_line_size)
        {
          if (fread(&inbyte, 1,1, m_imagefile)==0)
          {
            throw Error("Titan error - Unexpected end of file");
          }
          if (inbyte > 192)
          {
            length = inbyte - 192;
            if (fread(&inbyte, 1,1, m_imagefile)==0)
            {
              throw Error("Titan error - Unexpected end of file");
            }
            memset(bufptr, inbyte, length);
            bufptr+=length;
            readin+=length;
          } else
          {
            *bufptr = inbyte;
            bufptr++;
            readin++;
          }
        }
        int32 pitch = m_width * m_format.bytes();
        char8 *tempimageline = (char8 *)image+(pitch*y);
        // if 24 bit picture
        if (m_header[65] == 3)
        {
          for (int32 loop=0; loop<m_width; loop++)
          {
            *tempimageline++ = scanbuffer[loop+(m_width*2)];
            *tempimageline++ = scanbuffer[loop+m_width];
            *tempimageline++ = scanbuffer[loop];
    	    }
        } else
        {
          memcpy((char*)image+m_width*y, scanbuffer, line_size);
        }
        bufptr = scanbuffer;
        readin = 0;
      }
    }

    // close file
    fclose(m_imagefile);

    // success
    return 1;
}


int PCXHandler::valid()
{
  return (m_header[0] == 0x0a);
}


int PCXHandler::save(char *filename, int32 width, int32 height, 
					 Format *format, Palette *palette, void *pixels, 
					 void *params)
{
  throw Error("Titan error - PCX save not available");
  return 0;
}


void PCXHandler::Defaults()
{
    // defaults
    m_width=0;
    m_height=0;
    m_paletteflag=0;
    m_imagefile=NULL;
    memset(m_header,0,128);
}

#endif
