// JPG file handler
//
// Copyright (c) 1999 Dan Brown (_danbrown_@yahoo.com)
//
// Part of the Titan 1.1.x image handling library for PTC
// (http://now.at/Titan)
//
// This source code is licensed under the GNU GPL
//
// This is a wrapper for Independant JPEG groups code (libjpg)
//

#include "../titan.h"

#ifdef USE_JPG

#include <memory.h>
#include "jpg.h"

JPGHandler::JPGHandler()
{
    // defaults
    Defaults();    
}


JPGHandler::JPGHandler(char *filename)
{
    // defaults
    Defaults();

    m_filename = filename;

    // assign
    m_imagefile=fopen(m_filename, "rb+");
    if (m_imagefile == 0)
    {
      throw Error("Titan error - File does not exist");
    }

    // read signature (bad way of doing it)
    fseek(m_imagefile, 6, SEEK_SET);
    fread(sig, 4, 1, m_imagefile);
    fseek(m_imagefile, 0, SEEK_SET);

    if (!valid())
    {
      return;
    }

    // Set libjpg up to decompress
    cdinfo.err = jpeg_std_error(&jerr);
    jpeg_create_decompress(&cdinfo);

    // tell libjpeg to read from our file
    jpeg_stdio_src(&cdinfo, m_imagefile);

    // read JPG details
    jpeg_read_header(&cdinfo, true);

    // setup width
    m_width= cdinfo.image_width;

    // setup height
    m_height= cdinfo.image_height;

    // Set up format
    if (cdinfo.num_components == 3)     // 24-bit
    {
      m_format = Format(24, 0xff, 0xff<<8, 0xff<<16);  // reverse order!
    } else
    {
      throw Error("Titan error - Can't do grayscale jpegs yet, author too lazy");
    }
}

JPGHandler::~JPGHandler()
{
    // avoid warnings
    if (m_imagefile);
}

int JPGHandler::info(int32 &width,int32 &height,Format &format,int32 &palette)
{
    // setup info
    width=m_width;
    height=m_height;
    format=m_format;
    palette=false;
    return 1;
}


int JPGHandler::load(void *image, Palette *palette)
{
    // read image data
    if (image)
    {
      // tell libjpg we are ready to start decompressing
      jpeg_start_decompress(&cdinfo);

      char *i = (char *)image;

      // read the image
      for (int32 loop=0; loop < m_height; loop++)
      {
        jpeg_read_scanlines(&cdinfo, (char8 **)&i, 1);
        i += m_width*m_format.bytes();
      }

      // we are done
      jpeg_finish_decompress(&cdinfo);

      // free memory
      jpeg_destroy_decompress(&cdinfo);
    }

    fclose(m_imagefile);

    // success
    return 1;
}


int JPGHandler::valid()
{
  if ((sig[0] == 'J') &&
      (sig[1] == 'F') &&
      (sig[2] == 'I') &&
      (sig[3] == 'F'))
  {
    return true;
  }
  return false;
}

int JPGHandler::save(char *filename, int32 width, int32 height, 
					 Format *format, Palette *palette, void *pixels, 
					 void *params)
{
  JPGParams userparams;
  if (params == NULL)
  {
    JPGParamDefaults(userparams);          // if no parameters given, use
  } else                                   // defaults
  {
    userparams = *(JPGParams *)params;     // else use given parameters
  }

  // Store options
  m_filename = filename;
  m_width = width;
  m_height = height;
  m_format = format;

  // Check for invalid formats
  if (m_format.bytes() != 3)
  {
    throw Error("Titan error - Jpegs can only be saved from 24-bit Surfaces");
  }

  // Create file
  m_imagefile = fopen(m_filename, "wb");
  if (!m_imagefile)
  {
    throw Error("Titan error - Cannot create JPG file");
  }

  // Create compression structure
  ccinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&ccinfo);

  // Tell libjpeg to write to our file
  jpeg_stdio_dest(&ccinfo, m_imagefile);

  // Pass image details to libjpeg
  ccinfo.image_width = m_width;             // Set width + height
  ccinfo.image_height = m_height;
  ccinfo.input_components = 3;              // Set bits per pixel
  ccinfo.in_color_space = JCS_RGB;          // RGB format

  // Set all options to the default
  jpeg_set_defaults(&ccinfo);

  // Change the settings depending on our defaults (or user parameters)
  jpeg_set_quality(&ccinfo, userparams.quality, true);

  // Write header to file
  jpeg_start_compress(&ccinfo, true);

  // Temporary pointer to current scanline
  char8 *image = (char8 *)pixels;

  // If pixel format needs to be flipped
  char8 *data;
  data = new char8[m_width*3];

  // Write image
  for (int32 loop=0; loop < m_height; loop++)
  {
    // JPG stores pixels in GBR format, so if we have RGB...
    if (m_format.r() > m_format.b())
    {
      // flip data
      for (int32 innerloop=0; innerloop < (m_width*3); innerloop+=3)
      {
        data[innerloop+2] = image[innerloop];
        data[innerloop+1] = image[innerloop+1];
        data[innerloop] = image[innerloop+2];
      }

      // Write scanline
      jpeg_write_scanlines(&ccinfo, &data, 1);
    } else  // data is already BGR
    {
      // Write scanline
      jpeg_write_scanlines(&ccinfo, &image, 1);
    }

    // Make pixels point to next scanline
    image += (m_width*3);
  }

  // release memory for flip buffer
  delete data;

  // Finished compressing
  jpeg_finish_compress(&ccinfo);

  // Close our file
  fclose(m_imagefile);

  // Destroy compression structure
  jpeg_destroy_compress(&ccinfo);

//  throw Error("Titan error - JPG save not available");
  return 0;
}


void JPGHandler::Defaults()
{
    // defaults
    m_width=0;
    m_height=0;
    m_imagefile=NULL;
}

#endif