//
// X11 Windowed mode driver class for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#include <X11/Xlib.h>

#include "ptconfig.h"
#include "X11Display.h"
#include "X11Image.h"
#include "Error.h"
#include "Console.h"


X11WindowDisplay::~X11WindowDisplay()
{ close();
}

X11WindowDisplay::X11WindowDisplay()
{ m_primary=0;
  m_window=0;
  m_colours=0;
  m_id=PTC_WINDOW;
}


void X11WindowDisplay::open(const char title[],int width,int height,
  const Format& format,Display *disp,int screen) 
{ 
  m_disp=disp;
  m_screen=DefaultScreen(disp);
  m_height=height;
  m_width=width;
  m_destx=0;
  m_desty=0;

  // Check if we have that colour depth available.. Easy as there is no 
  // format conversion yet

  m_format=getFormat(format);

  // Create a window

  m_window=XCreateSimpleWindow(m_disp,DefaultRootWindow(m_disp),0,0,width,
			       height,0,0,0);
  

  // Register the delete atom
  
  m_atom_close=XInternAtom(m_disp,"WM_DELETE_WINDOW",False);
  XSetWMProtocols(m_disp,m_window,&m_atom_close,1);

  // Get graphics context
  XGCValues xgcv;
  xgcv.graphics_exposures=false;
  m_gc=XCreateGC(m_disp,m_window,GCGraphicsExposures,&xgcv);
  if(!m_gc)
  throw Error("can't create graphics context");

  m_atom_close=XInternAtom(m_disp,"WM_DELETE_WINDOW",1);
  XSetWMProtocols(m_disp,m_window,&m_atom_close,1);

  // Set window title
  XTextProperty textprop;
  char *title_ptr[]={const_cast<char*>(title)};

  XStringListToTextProperty(title_ptr,1,&textprop);
  XSetWMName(m_disp,m_window,&textprop);
  XFlush(m_disp);
  XFree(textprop.value);


  // Map the window and wait for success
  XSelectInput(m_disp,m_window,StructureNotifyMask);

  XMapRaised(m_disp,m_window);

  for(;;)
  { XEvent e;

    XNextEvent(disp,&e);
    if(e.type==MapNotify) break;
  }


  // Get keyboard input and sync
  XSelectInput(m_disp,m_window,KeyPress|KeyRelease);
  XSync(m_disp,0);

 
  // Create XImage using factory method
  m_primary=createImage(m_disp,m_screen,m_width,m_height,m_format);

  bool found;                                      // Stupid loop. The key
  do                                               // events were causing
  { XEvent e;                                      // problems.. 
     
    found=XCheckMaskEvent(m_disp,KeyPress|KeyRelease,&e);
  } while(found);

  // If we are in indexed mode, create the colour map 
  if(m_format.bits()==8)
  { 
    m_colours=new XColor[256];
    if(!m_colours) throw Error("Cannot allocated colour map cells");

    m_cmap=XCreateColormap(m_disp,RootWindow(m_disp,m_screen),
		           DefaultVisual(m_disp,m_screen),AllocAll);
    if(!m_cmap) throw Error("Cannot create colour map");

    XInstallColormap(m_disp,m_cmap);
    XSetWindowColormap(m_disp,m_window,m_cmap);
  }

  // Set 332 palette, for now
  if(m_format.bits()==8 && m_format.direct())
  { 
    // Taken from PTC 0.72, i hope it's fine
    for (int i=0;i<256;i++)
    {
      float r = (float)((i&0xE0)>>5) * (float)255.0 / (float)7.0;
      float g = (float)((i&0x1C)>>2) * (float)255.0 / (float)7.0;
      float b = (float) (i&0x03)     * (float)255.0 / (float)3.0;
	
      m_colours[i].pixel=i;
	
      m_colours[i].red=((int)r)<<8;
      m_colours[i].green=((int)g)<<8;
      m_colours[i].blue=((int)b)<<8;

      m_colours[i].flags=DoRed|DoGreen|DoBlue;
    }
   
    XStoreColors(m_disp,m_cmap,m_colours,256);
  }

#ifdef HAVE_PTHREADS
#endif
}


void X11WindowDisplay::open(Display *disp,int screen,Window w,
  const Format& format)
{ m_disp=disp;
  m_screen=screen;
  m_window=w;
  m_destx=0;
  m_desty=0;

  // Get the window's size (in an artificial scope)
  { Window wtmp;
    unsigned int i;
    int j;

    XGetGeometry(m_disp,m_window,&wtmp,&j,&j,&m_width,&m_height,
	         &i,&i);
  }

  m_format=getFormat(format); 

  // Create the graphics context
  XGCValues xgcv;
  xgcv.graphics_exposures=false;
  m_gc=XCreateGC(m_disp,m_window,GCGraphicsExposures,&xgcv);
  if(!m_gc)
  throw Error("can't create graphics context");

  // Create XImage using factory method
  m_primary=createImage(m_disp,m_screen,m_width,m_height,m_format);
}


void X11WindowDisplay::open(Display *disp,int screen,Window window,
  const Format& format,int x,int y,int w,int h)
{ m_disp=disp;
  m_screen=screen;
  m_window=window;
  m_destx=x;
  m_desty=y;
  m_width=w;
  m_height=h;

  m_format=getFormat(format); 

  // Create the graphics context
  XGCValues xgcv;
  xgcv.graphics_exposures=false;
  m_gc=XCreateGC(m_disp,m_window,GCGraphicsExposures,&xgcv);
  if(!m_gc)
  throw Error("can't create graphics context");

  // Create XImage using factory method
  m_primary=createImage(m_disp,m_screen,m_width,m_height,m_format);
}


void X11WindowDisplay::close()
{
  // Destroy XImage and buffer
  if(m_primary)
  delete m_primary;
  m_primary=0;

  if(m_colours)
  delete [] m_colours;
  m_colours=0;
 
  // Hide and destroy window
  if(m_window && !(m_flags&PTC_X11_LEAVE_WINDOW))
  { XUnmapWindow(m_disp,m_window);
    XDestroyWindow(m_disp,m_window);
  }
}


void X11WindowDisplay::update() 
{ m_primary->put(m_window,m_gc,m_destx,m_desty);

  // If we don't have pthreads, see if the close icon has been pressed
#ifndef HAVE_PTHREADS
  XEvent e;
  
  if(XCheckTypedEvent(m_disp,ClientMessage,&e))
  { if(e.xclient.format==32 && (Atom)e.xclient.data.l[0]==m_atom_close)
    exit(0);  
  }
#endif

}


void X11WindowDisplay::update(const Area &area) 
{ m_primary->put(m_window,m_gc,area.left(),area.top(),m_destx+area.left(),
		 m_desty+area.top(),area.width(),area.height());

  // If we don't have pthreads, see if the close icon has been pressed
#ifndef HAVE_PTHREADS
  XEvent e;
  
  if(XCheckTypedEvent(m_disp,ClientMessage,&e))
  { if(e.xclient.format==32 && (Atom)e.xclient.data.l[0]==m_atom_close)
    exit(0);  
  }
#endif

}

 
void* X11WindowDisplay::lock()
{ return m_primary->lock();
}


void X11WindowDisplay::unlock()
{ return;
}


void X11WindowDisplay::palette(int32 palette[])
{ if(!m_format.indexed()) return;

  for(int i=0;i<256;i++)
  { m_colours[i].pixel=i;

    m_colours[i].red=((palette[i]>>16)&0xff)<<8;
    m_colours[i].green=((palette[i]>>8)&0xff)<<8;
    m_colours[i].blue=(palette[i]&0xff)<<8;

    m_colours[i].flags=DoRed|DoGreen|DoBlue;
  }
  XStoreColors(m_disp,m_cmap,m_colours,256);

}


int X11WindowDisplay::pitch() const
{ return m_primary->pitch();
}


X11Image* X11WindowDisplay::createImage(Display *display,int screen, int width,
  int height,Format &format)
{
  
#ifdef HAVE_X11_EXTENSIONS_XSHM_H
  if(XShmQueryExtension(display))      // Look for SHM extension
  { X11Image *tmp;

    try {
      tmp=new X11SHMImage(display,screen,width,height,format);
    }
    catch(Error e)                     // Fall back to normal ximage routines
    { tmp=new X11NormalImage(display,screen,width,height,format);
    }

    return tmp;
  }
  else
#endif
  return new X11NormalImage(display,screen,width,height,format);
}


// Static event handler thread function
void X11WindowDisplay::eventHandler(void *data)
{ 
#ifdef HAVE_PTHREADS
  for(;;)
  {
    static XEvent e;
  
    if(XCheckTypedEvent(m_disp,ClientMessage,&e))
    { if(e.xclient.format==32 && e.xclient.data.l[0]==m_atom_close)
      exit(0);  
    }  
  }
#endif
}






