//
// X11 Display abstraction classes for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#ifndef __PTC_X11DISPLAY_H
#define __PTC_X11DISPLAY_H

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "ptconfig.h"
#include "Area.h"
#include "Format.h"
#include "Copy.h"
#include "X11Image.h"


#ifdef HAVE_PTHREADS
#include <pthread.h>
#endif 


enum x11_id{PTC_WINDOW,PTC_DGA};

class X11Display {

public:
  X11Display();
  virtual ~X11Display();

  // Check if DGA can be initialised
  static bool checkDGA(int width,int height,Display *disp,int screen);
  

  virtual void open(const char title[],int width,int height,
		    const Format& format,Display *disp,int screen) = 0;
  // This will always return a windowed console (obviously). The first version
  // fills the whole window, the second one has a custom size
  virtual void open(Display *disp,int screen,Window w,const Format& format)=0;
  virtual void open(Display *disp,int screen,Window window,
		    const Format& format,int x,int y,int w,int h) = 0;

  virtual void close() = 0;


  // Set the X11 specific flags

  void flags(long mask) { m_flags=mask; }
  void addflags(long mask) { m_flags|=mask; }


  // Update the console
  virtual void update() = 0;
  virtual void update(const Area &area) = 0;

  // Keyboard I/O
  virtual bool key();
  virtual int read();

  // Lock / unlock the display memory
  virtual void* lock() = 0;
  virtual void unlock() = 0;

  
  // load pixels to console
  virtual void load(const void *pixels,int width,int height,
		    const Format &format,const int32 palette[]);
  virtual void load(const void *pixels,int width,int height,
		    const Area &source,const Area &destination,
		    const Format &format,const int32 palette[]);

  // save console pixels
  virtual void save(void *pixels,int width,int height,
		    const Format &format,const int32 palette[]);
  virtual void save(void *pixels,int width,int height,
		    const Area &source,const Area &destination,
		    const Format &format,const int32 palette[]);

  // Console palette
  virtual void palette(int32 palette[]) = 0;
  const int32* palette() {}
 
  // Data access
  int width() const;
  int height() const;
  virtual int pitch() const = 0;
  const Format& format() const;


  // X11 helper functions for your enjoyment

  // return the display we are using
  Display* getX11Display() const { return m_disp; }

  // return the screen we are using
  int getX11Screen() const { return m_screen; }

  // return our window (0 if DGA)
  virtual Window getX11Window() const = 0;

  int id() const { return m_id; }

protected:
  // Generate internal format from requested format
  Format getFormat(const Format &format);          

  // Conversion object
  Copy m_copy;

  long m_flags;
  unsigned int m_width;
  unsigned int m_height;
  Format m_format;

  Display *m_disp;                                // X11 Display
  int m_screen;                                   // X11 Screen

  Colormap m_cmap;                                // Color map
  XColor *m_colours;                              // Color cells

  x11_id m_id;

#ifdef HAVE_PTHREADS
  pthread_t m_thread;
#endif
};



class X11WindowDisplay : public X11Display {
public:
  X11WindowDisplay();
  ~X11WindowDisplay();

  virtual void open(const char title[],int width,int height,
		    const Format& format,Display *disp,int screen);
  virtual void open(Display *disp,int screen,Window w,const Format &format);
  virtual void open(Display *disp,int screen,Window window,
		    const Format& format,int x,int y,int w,int h);

  virtual void close();

  virtual void update();
  virtual void update(const Area &area);

  virtual void* lock();
  virtual void unlock();

  virtual void palette(int32 palette[]);

  virtual int pitch() const;


  // X11 convenience functions 
  virtual Window getX11Window() const { return m_window; }
  virtual GC getX11GC() const { return m_gc; }

private:
  X11Image *createImage(Display *disp,int screen,int width,int height,
			Format &format);          // Factory method

  void eventHandler(void *);               // Event handler

  Window m_window;

  X11Image *m_primary;
  GC m_gc;
  int m_destx,m_desty;

  Atom m_atom_close;
};



#ifdef HAVE_DGA

#include <X11/extensions/xf86vmode.h>

class X11DGADisplay : public X11Display {
public:
  X11DGADisplay();

  ~X11DGADisplay();

  virtual void open(const char title[],int width,int height,
		    const Format& format,Display *disp,int screen);
  // Not in DGA mode
  virtual void open(Display *disp,int screen,Window w,const Format& format) 
  { if(disp==0); // Prevent warnings
    if(screen==0);
    if(w==0);
    if(format==Format(8));
  }
  virtual void open(Display *disp,int screen,Window window,
		    const Format& format,int x,int y,int w,int h) 
  { if(disp==0 || screen==0 || window==0 || format==Format(8) || x==0 ||
       y==0 || w==0 || h==0);
  }

  virtual void close();

  virtual void update();
  virtual void update(const Area &area) { update(); }

  virtual void* lock();
  virtual void unlock();

  virtual void palette(int32 palette[]);

  virtual int pitch() const;

  // check if DGA is available
  static bool check(int width,int height,Display *disp,int screen,long flags);

  // return 0 for window
  virtual Window getX11Window() const { return 0; }
  
private:
  XF86VidModeModeInfo **modeinfo;
  int num_modeinfo;
  int previousmode;
  
  char *dga_addr;
  int dga_linewidth;
  int dga_banksize;
  int dga_memsize;
  int dga_width,dga_height;

  // Coordinates of upper left frame corner
  int m_destx,m_desty;
};



#else


class X11DGADisplay : public X11Display {
public:
  static bool check(int width,int height,Display *disp,int screen,long flags)
  { return false;
  }
};

#endif

#endif



