//
// X11 Direct Graphics Access driver class for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#include "ptconfig.h"

#ifdef HAVE_DGA

#include <X11/Xlib.h>
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>
#include <unistd.h>
#include <string.h>

#include "X11Display.h"
#include "Console.h"
#include "Error.h"




X11DGADisplay::X11DGADisplay()
{ m_id=PTC_DGA;
}



X11DGADisplay::~X11DGADisplay()
{ close();
}


void X11DGADisplay::open(const char title[],int width,int height,
			 const Format& format,Display *disp,int screen)
{ 
  m_disp=disp;
  m_screen=screen;

  // Get all availabe video modes
  XF86VidModeGetAllModeLines(m_disp,m_screen,&num_modeinfo,&modeinfo);
 

  // Save previous mode
  XF86VidModeModeLine *vml=new XF86VidModeModeLine;
  int dotclock;
  XF86VidModeGetModeLine(m_disp,m_screen,&dotclock,vml);

  for(int i=0;i<num_modeinfo;i++)
    { if(vml->hdisplay==modeinfo[i]->hdisplay &&
	 vml->vdisplay==modeinfo[i]->vdisplay)
      { previousmode=i;
        break;
      }
    }
  delete vml;




  // Find a video mode to set
  
  // Normal modesetting first, find exactly matching mode
  if(!m_flags&PTC_X11_PEDANTIC_DGA)
  for(int i=0;i<num_modeinfo;i++)
  { 
    if(modeinfo[i]->hdisplay==width && modeinfo[i]->vdisplay==height)
    {  
      if(!XF86VidModeSwitchToMode(m_disp,m_screen,modeinfo[i]))
      throw Error("Error switching to request video mode");

      m_destx=0;
      m_desty=0;

      break;
 
    }
  }
  else
  { int found_mode=0xffff;

    // Try to find a mode that matches the width first
    for(int i=0;i<num_modeinfo;i++)
    { if(modeinfo[i]->hdisplay==width && modeinfo[i]->vdisplay>=height)
      { found_mode=i;
        break;
      }
    }

    // Next try to match the height
    if(found_mode==0xffff)
    for(int i=0;i<num_modeinfo;i++)
    { if(modeinfo[i]->hdisplay>=width && modeinfo[i]->vdisplay==height)
      { found_mode=i;
        break;
      }
    }

    // Finally, find the mode that is bigger than the requested one and makes
    // the least difference
    int min_diff=987654321;
   
    for(int i=0;i<num_modeinfo;i++)
    { if(modeinfo[i]->hdisplay>=width && modeinfo[i]->vdisplay>=height)
      { int d_x=modeinfo[i]->hdisplay-width;
        d_x*=d_x;

        int d_y=modeinfo[i]->vdisplay-height;
	d_y*=d_y;

        if(d_x+d_y<min_diff)
	{ min_diff=d_x+d_y;
	  found_mode=i;
	}
      } 
    }

    if(found_mode!=0xffff)
    { if(!XF86VidModeSwitchToMode(m_disp,m_screen,modeinfo[found_mode]))
      throw Error("Error switching to requested video mode");

      m_destx=modeinfo[found_mode]->hdisplay/2-width/2;
      m_desty=modeinfo[found_mode]->vdisplay/2-height/2;
      
    }
    else throw Error("Cannot find a video mode to use");
  }

  XFlush(m_disp);

  m_width=width;
  m_height=height;

  // Check if the requested colour mode is available
  m_format=getFormat(format);

  // Grab exclusive control over the keyboard and mouse
  Window root=XRootWindow(m_disp,m_screen);
  XGrabKeyboard(m_disp,root,True,GrabModeAsync,GrabModeAsync,CurrentTime);
  XGrabPointer(m_disp,root,True,PointerMotionMask|ButtonPressMask|
	       ButtonReleaseMask,GrabModeAsync,GrabModeAsync,None,None,
	       CurrentTime);
  XFlush(m_disp);

  // Get Display information
  XF86DGAGetVideo(m_disp,m_screen,&dga_addr,&dga_linewidth,
		  &dga_banksize,&dga_memsize);

  // Don't have to be root anymore
  setuid(getuid());                             

  XF86DGAGetViewPortSize(m_disp,m_screen,(int *)(&dga_width),
			 (int *)(&dga_height));

  if(XF86DGAForkApp(m_screen) != 0)
  throw Error("cannot do safety fork");
  else
  { if(!XF86DGADirectVideo(m_disp,m_screen,XF86DGADirectGraphics|
        XF86DGADirectKeyb|XF86DGADirectMouse))
    throw Error("cannot switch to DGA mode");
  }
  
  memset(dga_addr,0,dga_linewidth*dga_height*(m_format.bits()>>3));

  XSelectInput(m_disp,DefaultRootWindow(m_disp),KeyPress|KeyRelease);
  XF86DGASetViewPort(m_disp,m_screen,0,0);         // Important.. sort of =)

  bool found;                                      // Stupid loop. The key
  do                                               // events were causing
  { XEvent e;                                      // problems.. 
     
    found=XCheckMaskEvent(m_disp,KeyPress|KeyRelease,&e);
  } while(found);

  // Create colour map in 8 bit mode
  if(m_format.bits()==8)
  {
    m_colours=new XColor[256];
    if(!m_colours) throw Error("Cannot allocated colour map cells");

    m_cmap=XCreateColormap(m_disp,RootWindow(m_disp,m_screen),
                           DefaultVisual(m_disp,m_screen),AllocAll);
    if(!m_cmap) throw Error("Cannot create colour map");
  }

  // Set 332 palette 
  if(m_format.bits()==8 && m_format.direct())
  { 
    // Taken from PTC 0.72, i hope it's fine
    for (int i=0;i<256;i++)
    {
      float r = (float)((i&0xE0)>>5) * (float)255.0 / (float)7.0;
      float g = (float)((i&0x1C)>>2) * (float)255.0 / (float)7.0;
      float b = (float) (i&0x03)     * (float)255.0 / (float)3.0;
	
      m_colours[i].pixel=i;
	
      m_colours[i].red=((int)r)<<8;
      m_colours[i].green=((int)g)<<8;
      m_colours[i].blue=((int)b)<<8;

      m_colours[i].flags=DoRed|DoGreen|DoBlue;
    }
   
    XStoreColors(m_disp,m_cmap,m_colours,256);
    XF86DGAInstallColormap(m_disp,m_screen,m_cmap);
  }
}



void X11DGADisplay::close()
{
  XF86DGADirectVideo(m_disp,m_screen,0);

  XF86VidModeSwitchToMode(m_disp,m_screen,modeinfo[previousmode]);
  XFlush(m_disp);

  XFree(modeinfo);
}



void X11DGADisplay::update()
{ return;
}



void* X11DGADisplay::lock()
{ return dga_addr+dga_linewidth*m_desty*(m_format.bits()>>3)+
                 m_destx*(m_format.bits()>>3);
}



void X11DGADisplay::unlock()
{ return;
}



void X11DGADisplay::palette(int32 palette[])
{ if(!m_format.indexed()) return;

  for(int i=0;i<256;i++)
  { m_colours[i].pixel=i;

    m_colours[i].red=(palette[i]>>8)&0xff00;
    m_colours[i].green=palette[i]&0xff00;
    m_colours[i].blue=(palette[i]&0xff)<<8;

    m_colours[i].flags=DoRed|DoGreen|DoBlue;
  }
  XStoreColors(m_disp,m_cmap,m_colours,256);
  XF86DGAInstallColormap(m_disp,m_screen,m_cmap);
}



int X11DGADisplay::pitch() const
{ return dga_linewidth*(m_format.bits()/8);
}


// Error handlers for check routine
static int (*m_dga_oldhandler)(Display *,XErrorEvent *);
static bool m_dga_handler_error=false;

static int m_dga_errorhandler(Display *disp,XErrorEvent *xev)
{ if(xev->request_code==145 && xev->minor_code==8)    // XF86DGAQueryDirectV.
  { m_dga_handler_error=true;                         // catch error
    return 0;
  }
  else
  return((*m_dga_oldhandler)(disp,xev));
}


#include <stdio.h>
bool X11DGADisplay::check(int width,int height,Display *disp,int screen,
  long flags)
{ 
  // User doesn't want DGA
  if(flags&PTC_X11_NODGA) return false;


  // We aren't root 
  if(geteuid()) return false;


  // Check if the DGA extension and VidMode extension can be used
  int dummy1,dummy2;
  if(!XF86DGAQueryExtension(disp,&dummy1,&dummy2)) return false;
  if(!XF86VidModeQueryExtension(disp,&dummy1,&dummy2)) return false;


  // Check if we can switch to DGA mode
  m_dga_oldhandler=XSetErrorHandler(m_dga_errorhandler);

  int dga_flags;

  XF86DGAQueryDirectVideo(disp,screen,&dga_flags);
  if(!dga_flags&XF86DGADirectPresent)
  return false;

  XSetErrorHandler(m_dga_oldhandler);

  // Get a list of all video modes
  int num_modeinfo;
  XF86VidModeModeInfo **modeinfo=0;
  
  XF86VidModeGetAllModeLines(disp,screen,&num_modeinfo,&modeinfo);
  if(num_modeinfo==0) return false;

  bool found=false;

  // Try harder to initialise DGA mode, allow any resolution greater than
  // the requested one.
  if(flags&PTC_X11_PEDANTIC_DGA)
  { for(int i=0;i<num_modeinfo;i++)
    { if(modeinfo[i]->hdisplay>=width && modeinfo[i]->vdisplay>=height)
      { found=true;
        break;
      }
    }
  }
  else
  { // Otherwise, look for exact matches

    for(int i=0;i<num_modeinfo;i++)
    {
      if(modeinfo[i]->hdisplay==width && modeinfo[i]->vdisplay==height)
      { found=true;
        break;
      }
    }
  }

  XFree(modeinfo);

  if(!found) return false;

  return true;
}

#endif





