//
// DirectX Display class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include "Core/Error.h"
#include "DirectX/Check.h"
#include "DirectX/Display.h"




DirectXDisplay::DirectXDisplay()
{
    // setup data
    m_open = false;
    m_fullscreen = false;
    m_ddraw = 0;
    m_window = 0;
    m_foreground = 0;
}


DirectXDisplay::~DirectXDisplay()
{
    // close
    close();
}




void DirectXDisplay::setup(LPDIRECTDRAW2 ddraw)
{
    // setup ddraw
    m_ddraw = ddraw;
}


void DirectXDisplay::save()
{
    // save foreground window handle
    m_foreground = GetForegroundWindow();

    // save foreground window rect
    GetWindowRect(m_foreground,&m_foreground_rect);

    // save foreground window placement
    m_foreground_placement.length = sizeof WINDOWPLACEMENT;
    GetWindowPlacement(m_foreground,&m_foreground_placement);
}


bool DirectXDisplay::test(int width,int height,const Format &format)
{
    // todo: test display mode exists
    
    // mode exists
    return true;
}


void DirectXDisplay::cooperative(HWND window,bool fullscreen)
{
    // set cooperative level
    if (fullscreen)
    {
        // enter exclusive mode
        DirectXCheck(m_ddraw->SetCooperativeLevel(window,DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWMODEX));
    }
    else
    {
        // set normal cooperative level
        DirectXCheck(m_ddraw->SetCooperativeLevel(window,DDSCL_NORMAL));
    }

    // set window
    m_window = window;

    // set fullscreen flag
    m_fullscreen = fullscreen;
}


void DirectXDisplay::open()
{
    // update flags
    m_open = true;
}


void DirectXDisplay::open(int width,int height,const Format &format)
{
    // display mode frequency
    const int frequency = 0;

    // set display mode
    switch (format.bits())
    {
        case 32: 
        {
            // set virtual 32 bit mode
            if (m_ddraw->SetDisplayMode(width,height,32,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,24,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,16,frequency,0)==DD_OK) break;
            
            // error message
            throw Error("could not set display mode (32)");
        }
        break;

        case 24:
        {
            // set virtual 24 bit mode
            if (m_ddraw->SetDisplayMode(width,height,24,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,32,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,16,frequency,0)==DD_OK) break;
            
            // error message
            throw Error("could not set display mode (24)");
        }
        break;

        case 16: 
        {
            // set virtual 16 bit mode
            if (m_ddraw->SetDisplayMode(width,height,16,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,24,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,32,frequency,0)==DD_OK) break;

            // error message
            throw Error("could not set display mode (16)");
        }
        break;

        case 8: 
        {
            // set virtual 8 bit mode
            if (m_ddraw->SetDisplayMode(width,height,8,frequency,0)==DD_OK)  break;
            if (m_ddraw->SetDisplayMode(width,height,32,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,24,frequency,0)==DD_OK) break;
            if (m_ddraw->SetDisplayMode(width,height,16,frequency,0)==DD_OK) break;

            // error message
            throw Error("could not set display mode (8)");
        }
        break;

        default:
        {
            // error message
            throw Error("unsupported pixel format");
        }
    }

    // update flags
    m_open = true;
}


void DirectXDisplay::close()
{
    // check if open
    if (m_open && m_ddraw)
    {
        // check if fullscreen
        if (m_fullscreen)
        {
            // restore display mode
            m_ddraw->RestoreDisplayMode();
        
            // leave exclusive mode
            m_ddraw->SetCooperativeLevel(m_window,DDSCL_NORMAL);
        }
    }

    // clear flags
    m_open = false;
    m_fullscreen = false;
}


void DirectXDisplay::restore()
{
    // check foreground window is valid and not the console window
    if (m_foreground && IsWindow(m_foreground) && m_foreground!=m_window)
    {
        // check console window and fullscreen
        if (IsWindow(m_window) && m_fullscreen)
        {
            // minimize console window
            SetWindowPos(m_window,HWND_BOTTOM,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE|SWP_NOACTIVATE);
        }

        // restore foreground window
        SetForegroundWindow(m_foreground);

        // restore foreground window placement
        SetWindowPlacement(m_foreground,&m_foreground_placement);

        // restore old foreground window position
        SetWindowPos(m_foreground,HWND_TOP,m_foreground_rect.left,m_foreground_rect.top,m_foreground_rect.right-m_foreground_rect.left,m_foreground_rect.bottom-m_foreground_rect.top,SWP_FRAMECHANGED|SWP_NOCOPYBITS);
    
        // clear foreground
        m_foreground = 0;
    }
}




bool DirectXDisplay::fullscreen() const
{
    // get fullscreen
    return m_fullscreen;
}
