// TGA file handler
//
// Copyright (c) Glenn Fiedler 1998 (ptc@gaffer.org)
//
// Updated/bugfixed for Titan by Dan Brown (_danbrown_@yahoo.com)
//    please contact Dan Brown for any queries about this class
//
// Part of the Titan 1.1.x image handling library for PTC
// (http://now.at/Titan)
//
// This source code is licensed under the GNU GPL
//

#include "../titan.h"

#ifdef USE_TGA

#include <memory.h>
#include <string.h>
#include "tga.h"

TGAHandler::TGAHandler()
{
    // defaults
    Defaults();    
}


TGAHandler::TGAHandler(char *filename)
{
    // defaults
    Defaults();

    m_filename = filename;

    // open file
    m_imagefile=fopen(m_filename, "rb+");
    if (m_imagefile == 0)
    {
      throw Error("Titan error - File does not exist");
    }

    // read tga header
    fread(m_header, 18, 1, m_imagefile);

    // close file again
    fclose(m_imagefile);

    // setup width
    m_width=m_header[13];
    m_width<<=8;
    m_width+=m_header[12];

    // setup height
    m_height=m_header[15];
    m_height<<=8;
    m_height+=m_header[14];

    // setup format
    switch (m_header[16])
    {
                 // todo: handle indexed formats > 8bits?
        case 8:  m_format = Format(8); break;
        case 15: m_format = Format(16, 0x1f<<10, 0x1f<<5, 0x1f);       break;
                 // tga 16-bit is actually 15-bit - ARRRRRGGGGGBBBBB
        case 16: m_format = Format(16, 0x1f<<10, 0x1f<<5, 0x1f);       break;
        case 24: m_format = Format(24, 0xff<<16, 0xff<<8, 0xff); break;
        case 32: m_format = Format(32, 0xff<<16, 0xff<<8, 0xff); break;
    }

    // setup palette flag
    m_paletteflag=m_header[1];
}


TGAHandler::~TGAHandler()
{
    // avoid warnings
    if (m_imagefile);
}


int TGAHandler::info(int32 &width,int32 &height,Format &format,int32 &palette)
{
    // setup info
    width=m_width;
    height=m_height;
    format=m_format;
    palette=m_paletteflag;
    return 1;
}


int TGAHandler::load(void *image, Palette *palette)
{
    m_imagefile = fopen(m_filename, "rb+");

    // setup palette data
    int32 palette_offset=m_header[4];
    palette_offset<<=8;
    palette_offset+=m_header[3];
    int32 palette_length=m_header[6];
    palette_length<<=8;
    palette_length+=m_header[5];
    int32 palette_bits=m_header[7];

    // read palette
    if (palette && m_paletteflag && palette_length)
    {
        // reject anything > 256 palette entries
        if (palette_length>256) return 0;

        // setup temp palette buffer
        char8 *temp=new char8[palette_length*palette_bits/8];
        if (!temp) throw(Error("Titan error - Could not allocate a palette"));

        // read into temp buffer
//        fseek(m_imagefile, 18+ m_header[0]+ palette_offset, SEEK_SET);
        fseek(m_imagefile, 18+palette_offset, SEEK_SET);
        int32 result = fread(temp,palette_length*palette_bits/8, 1, m_imagefile);

        int32 tempindex = 0;
        int32 paletteindex = 0;

        // copy temporary palette to proper palette
        int32 pal[256];
        char8 r;
        char8 g;
        char8 b;
        for (int32 loop=0; loop<palette_length; loop++)
        {
          b = temp[tempindex++];
          g = temp[tempindex++];
          r = temp[tempindex++];
          pal[loop] = r<<16 | g<<8 | b;
        }
        palette->load(pal);

        // deallocate memory
        delete temp;
    }

    // read image data
    if (image)
    {
        int32 image_offset = 18+m_header[0];    // m_header[0] is size of id string
        if (m_paletteflag) image_offset = 18 + palette_offset + palette_length*palette_bits/8;
        fseek(m_imagefile, image_offset, SEEK_SET);
        
        // image orientation
        char8 tga_orientation=(m_header[17] & 48) >> 4;  // get orientation bits
        if (tga_orientation==0) tga_orientation=BOTTOMUP;
        else if (tga_orientation==2) tga_orientation=TOPDOWN;
        else throw Error("Titan error - Unknown orientation");

        // read image data (todo: check all these!)
        if (tga_orientation==TOPDOWN)
        {
            // topdown -> topdown
            int32 line_size = m_width*m_format.bytes();
            for (int32 y=0; y<m_height; y++)
            {
              if (fread((char*)image+line_size*y, line_size, 1, m_imagefile)!=1)
                throw Error("Titan error - Unexpected end of file");
            }
        } else if (tga_orientation==BOTTOMUP)
        {
            // bottomup -> topdown
            int line_size=m_width*m_format.bytes();
            for (int y=m_height-1; y>=0; y--)
            {
              if (fread((char*)image+(line_size*y), line_size, 1, m_imagefile)!=1)
                throw Error("Titan error - Unexpected end of file");
            }
        }   
    } else
    {
      throw Error("Titan error - no pixels to load image onto");
    }

    fclose(m_imagefile);

    // success
    return 1;
}

int TGAHandler::save(char *filename, int32 width, int32 height, 
					 Format *format, Palette *palette, void *pixels, 
					 void *params)
{
  TGAParams userparams;
  if (params == NULL)
  {
    TGAParamDefaults(userparams);          // if no parameters given, use
  } else                                   // defaults
  {
    userparams = *(TGAParams *)params;     // else use given parameters
  }

  // Check for unsupported options
  if (userparams.compressed == 1)
  {
    throw Error("Titan error - Cannot save compressed TGA files");
  }

  // options are OK
  m_filename = filename;
  m_width = width;
  m_height = height;
  m_format = *format;
  m_paletteflag = (m_format.bytes() == 1);

  // create file
  m_imagefile = fopen(m_filename, "wb");
  if (!m_imagefile)
  {
    throw Error("Titan error - Cannot create TGA file");
  }

  // write header
  char id[100] = "Created by Titan ";
  strcat(id, TITANVERSIONSTRING);
  char8 len = strlen(id);

  char8 val;                  // Used for temporary storage
  short16 val2;               // Same

  // FIXME, DOESN'T WORK WITH PAINTSHOP PRO
  len = 0;

  fwrite(&len, 1, 1, m_imagefile);            // Write size of id string

  val = m_paletteflag;
  fwrite(&val, 1, 1, m_imagefile);  // Write if there is a palette

  if (m_paletteflag == 0)
  {
    val = 2;
    fwrite(&val, 1, 1, m_imagefile); // Image is truecolour
  } else
  {
    val = 1;
    fwrite(&val, 1, 1, m_imagefile); // Image is 8-bit paletted
  }

  // write palette info
  if (m_paletteflag == 1)
  {
    val2 = len;                     // Palette starts at header + id string
    fwrite(&val2, 2, 1, m_imagefile);

    val2 = 256;                    // Number of palette entries
    fwrite(&val2, 2, 1, m_imagefile);

    val = 24;                      // 24 bits per palette entry
    fwrite(&val, 1, 1, m_imagefile);
  } else
  {
    val = 0;
    fwrite(&val, 1, 1, m_imagefile);  // Write blank colour map section
    fwrite(&val, 1, 1, m_imagefile);
    fwrite(&val, 1, 1, m_imagefile);
    fwrite(&val, 1, 1, m_imagefile);
    fwrite(&val, 1, 1, m_imagefile);  // fwrite (val, 1, 6, file) doesn't work?
  }

  // Image starts at 0,0
  val = 0;
  fwrite(&val, 1, 1, m_imagefile);
  fwrite(&val, 1, 1, m_imagefile);
  fwrite(&val, 1, 1, m_imagefile);
  fwrite(&val, 1, 1, m_imagefile);

  // Write width and height
  val2 = (short16) m_width;
  fwrite(&val2, 2, 1, m_imagefile);
  val2 = (short16) m_height;
  fwrite(&val2, 2, 1, m_imagefile);

  // Write bits per pixel
  val = m_format.bits();
  fwrite(&val, 1, 1, m_imagefile);

  // Write image descriptor byte
  val = 32;                   // No alpha channel, top to bottom, no interleave
  fwrite(&val, 1, 1, m_imagefile);

  // Write id string
  // FIXME, DOESN'T WORK WITH PAINTSHOP PRO
//  fwrite(id, len, 1, m_imagefile);

  // Write actual palette
  if (m_paletteflag == 1)
  {
    int32 *pal = palette->lock();
    char8 newpal[768];
	int32 temppal;
    for (int loop=0; loop < 768; loop+=3)
    {
      temppal  = *(pal++);
	  newpal[loop]   = (char8) (temppal & 0x000000ff);
      newpal[loop+1] = (char8) ((int32)(temppal & 0x0000ff00) >> 8);
	  newpal[loop+2] = (char8) ((int32)(temppal & 0x00ff0000) >> 16);
    }
    fwrite(newpal, 768, 1, m_imagefile);
    palette->unlock();
  }

  // Finished writing header

  // Write pixels
  fwrite(pixels, (m_width*m_height*m_format.bytes()), 1, m_imagefile);

  // Close file again
  fclose(m_imagefile);

  return 0;
}


int TGAHandler::valid()
{
  // bad way of detecting .TGA files
  return ((m_header[2] == 1) || (m_header[2] == 2));
}


void TGAHandler::Defaults()
{
    // defaults
    m_width=0;
    m_height=0;
    m_paletteflag=0;
    m_imagefile=NULL;
    memset(m_header,0,18);
}

#endif
