// Main Titan class for still images
//
// Copyright (c) Dan Brown 1999 (_danbrown_@yahoo.com)
//
// Part of the Titan 1.1.x image handling library for PTC
// (http://now.at/Titan)
//
// This source code is licensed under the GNU GPL
//

// contents.h decides which files will be linked to the application
#include "../contents.h"

#include "image.h"

#include <string.h>

// Include relevant file formats
#ifdef USE_TGA
#include "tga.h"
#endif

#ifdef USE_PCX
#include "pcx.h"
#endif

#ifdef USE_BMP
#include "bmp.h"
#endif

#ifdef USE_PNG
#include "png.h"
#endif

#ifdef USE_JPG
#include "jpg.h"
#endif

#ifdef USE_LBM
#include "lbm.h"
#endif

Image::Image(void)
{
  m_pixels = (void *)NULL;
  m_handler = (ImageHandler *)NULL;
}


Image::Image(char *filename)
{
  m_pixels = (void *)NULL;
  m_handler = (ImageHandler *)NULL;

  load(filename);
}


Image::~Image(void)
{
  if (m_pixels != NULL)
  {
    delete m_pixels;
  }
  if (m_handler != NULL)
  {
    delete m_handler;
  }
}


void Image::load(char *filename)
{
  open(filename);

  m_handler->info(m_width, m_height, m_format, m_paletteflag);

  m_pixels = (void *)new char8[((m_width)*m_height)*m_format.bytes()];

  m_handler->load(m_pixels, &m_palette);
}


void Image::open(char *filename)
{

#ifdef USE_LBM
    // try to load it as a .LBM file
  m_handler=new LBMHandler(filename);
  if (m_handler != NULL && m_handler->valid())
  {
    // file is .LBM
    return;
  }
  delete m_handler;
#endif

#ifdef USE_JPG
  // try to load it as a .JPG file
  m_handler=new JPGHandler(filename);
  if (m_handler != NULL && m_handler->valid())
  {
    // file is .JPG
    return;
  }
  delete m_handler;
#endif

#ifdef USE_PNG
  // try to load it as a .PNG file
  m_handler=new PNGHandler(filename);
  if (m_handler != NULL && m_handler->valid())
  {
    // file is .PNG
    return;
  }
  delete m_handler;
#endif

#ifdef USE_BMP
  // try to load it as a .BMP file
  m_handler=new BMPHandler(filename);
  if (m_handler != NULL && m_handler->valid())
  {
    // file is .BMP
    return;
  }
  delete m_handler;
#endif

#ifdef USE_PCX
  // try to load it as a .PCX file
  m_handler=new PCXHandler(filename);
  if (m_handler != NULL && m_handler->valid())
  {
    // file is .PCX
    return;
  }
  delete m_handler;
#endif

#ifdef USE_TGA
    // try to load it as a .TGA file
  m_handler=new TGAHandler(filename);
  if (m_handler != NULL && m_handler->valid())
  {
    // file is .TGA
    return;
  }
  delete m_handler;
#endif

  m_handler = (ImageHandler *)NULL;
  throw Error("Titan - Unsupported file format");
}


void Image::save(char *filename, int32 width, int32 height, Format *format, Palette *palette, void *pixels, int32 filetype, void *params)
{
  if (m_handler != NULL)
  {
    delete m_handler;
  }
  switch (filetype)
  {
#ifdef USE_TGA
    case TGA : {
                 m_handler = new TGAHandler();
                 m_handler->save(filename, width, height, format, palette, pixels, params);
                 break;
               }
#endif
#ifdef USE_PCX
    case PCX : {
                 m_handler = new PCXHandler();
                 m_handler->save(filename, width, height, format, palette, pixels, params);
                 break;
               }
#endif
#ifdef USE_BMP
    case BMP : {
                 m_handler = new BMPHandler();
                 m_handler->save(filename, width, height, format, palette, pixels, params);
                 break;
               }
#endif
#ifdef USE_PNG
    case PNG : {
                 m_handler = new PNGHandler();
                 m_handler->save(filename, width, height, format, palette, pixels, params);
                 break;
               }
#endif
#ifdef USE_JPG
    case JPG : {
                 m_handler = new JPGHandler();
                 m_handler->save(filename, width, height, format, palette, pixels, params);
                 break;
               }
#endif
    default :  {
                 throw Error("Titan error - Cannot save to the specified file format");
                 break;
               }
  }
}

void Image::save(char *filename, BaseSurface &surf, int32 filetype, void *params)
{
  int32 width = surf.width();
  int32 height = surf.height();
  Format format = surf.format();
  Palette palette = surf.palette();
  save(filename, width, height, &format, &palette, surf.lock(), filetype, params);
  surf.unlock();
}

void Image::copy(BaseSurface &surf)
{
  // copy picture to the surface
  surf.load(m_pixels, m_width, m_height, (m_width*m_format.bytes()), m_format, m_palette);

  // on my system, the above line doesn't set the palette, so extra
  // insurance is needed
  if (m_paletteflag)
  {
    surf.palette(m_palette);
  }
}


void Image::copy(BaseSurface &surf, const Area &src, const Area &dest)
{
  // copy picture to the surface
  surf.load(m_pixels, m_width, m_height, (m_width*m_format.bytes()), m_format, m_palette, src, dest);
  // on my system, the above line doesn't set the palette, so extra
  // insurance is needed
  if (m_paletteflag)
  {
    surf.palette(m_palette);
  }
}


int32 Image::width(void)
{
  return m_width;
}


int32 Image::height(void)
{
  return m_height;
}


Format Image::format(void)
{
  return m_format;
}

Palette Image::palette(void)
{
  return m_palette;
}
