// BMP file handler
//
// Copyright (c) Dan Brown 1999 (_danbrown_@yahoo.com)
//
// Part of the Titan 1.1.x image handling library for PTC
// (http://now.at/Titan)
//
// This source code is licensed under the GNU GPL
//

#include "../titan.h"

#ifdef USE_BMP

#include <memory.h>
#include "bmp.h"

BMPHandler::BMPHandler()
{
    // defaults
    Defaults();    
}


BMPHandler::BMPHandler(char *filename)
{
  // defaults
  Defaults();

  m_filename = filename;

  // open file
  m_imagefile = fopen(m_filename, "rb+");
  if(m_imagefile == NULL)
  {
    throw Error("Titan error - File does not exist");
  }

  // read bmp header
  fread(m_header, 54,1, m_imagefile);

  // close file
  fclose(m_imagefile);

  // setup width - Should really create macro to do this dirty work
  m_temp=m_header[21];
  m_temp<<=24;
  m_width=m_temp;
  m_temp=m_header[20];
  m_temp<<=16;
  m_width+=m_temp;
  m_temp=m_header[19];
  m_temp<<=8;
  m_width+=m_temp;
  m_temp=m_header[18];
  m_width+=m_temp;
    
  // setup height
  m_temp=m_header[25];
  m_temp<<=24;
  m_height=m_temp;
  m_temp=m_header[24];
  m_temp<<=16;
  m_height+=m_temp;
  m_temp=m_header[23];
  m_temp<<=8;
  m_height+=m_temp;
  m_temp=m_header[22];
  m_height+=m_temp;

  // setup format depending on bits per pixel
  switch (m_header[28])
  {
    case 24:  m_format = Format(24, 0xff<<16, 0xff<<8, 0xff);
              m_paletteflag = 0;
              break;     // 24 bit
    case 8:   m_format = Format(8);              // 8 bit
	      m_paletteflag = 1;
	      m_paletteoffset = 54;       // Should calculate this,
     	 	                          // but should be standard
	      m_coloursused = m_header[46];
	      if (m_coloursused == 0)
	      {
                m_coloursused = 256;
              }
              break;
  }
}


BMPHandler::~BMPHandler()
{
}


int BMPHandler::info(int32 &width,int32 &height,Format &format,int32 &palette)
{
    // setup info
    width=m_width;
    height=m_height;
    format=m_format;
    palette=m_paletteflag;
    return 1;
}


int BMPHandler::load(void *image, Palette *palette)
{
  // open file
  m_imagefile = fopen(m_filename, "rb+");

  // try to load palette for 8bpp picture
  if (m_paletteflag)
  {
    // setup temp palette buffer
    int32 temp[256];
    memset(temp, 0, 256*4);

    // goto start of palette
    fseek(m_imagefile, m_paletteoffset, SEEK_SET);

    // read into temp buffer
    char8 r;
    char8 g;
    char8 b;
    char8 filler;
    int32 col;
    for (int32 paletteloop=0; paletteloop <= (m_coloursused-1); paletteloop++)
    {
      fread(&b, 1,1, m_imagefile);
      fread(&g, 1,1, m_imagefile);
      fread(&r, 1,1, m_imagefile);
      fread(&filler, 1,1, m_imagefile);
      col = r<<16 | g<<8 | b;
      temp[paletteloop] = col;
    }
    palette->load(temp);
  }

    // read image data
  if (!image)
  {
    throw Error("Titan error - no pixels to load image onto");
  }

  // where does image start?
  m_temp=m_header[13];
  m_temp<<=24;
  int32 imagestart=m_temp;
  m_temp=m_header[12];
  m_temp<<=16;
  imagestart+=m_temp;
  m_temp=m_header[11];
  m_temp<<=8;
  imagestart+=m_temp;
  m_temp=m_header[10];
  imagestart+=m_temp;

  // goto start of image
  fseek(m_imagefile, imagestart, SEEK_SET);
    
  // work out line size (padded to 4 bytes) by image size / height
  m_temp=m_header[5];
  m_temp<<=24;
  int32 bmp_line_size=m_temp;
  m_temp=m_header[4];
  m_temp<<=16;
  bmp_line_size+=m_temp;
  m_temp=m_header[3];
  m_temp<<=8;
  bmp_line_size+=m_temp;
  m_temp=m_header[2];
  bmp_line_size+=m_temp;            // got file size

  bmp_line_size -= imagestart;    // got image size

  bmp_line_size /= m_height;        // got line size (hopefully)

  int32 line_size=m_width*m_format.bytes();

  char8 scanbuffer[4096];

  int32 pitch = m_width * m_format.bytes();
  // BMPs are always stored BOTTOMUP
  for (int y=m_height-1; y>=0; y--)
  {
    if (fread(scanbuffer, bmp_line_size,1, m_imagefile)==0)
    {
      throw Error("Titan error - Unexpected end of file");
    }
    memcpy((char*)image+pitch*y, scanbuffer, line_size);
  }

  // close file
  fclose(m_imagefile);

  // success
  return 1;
}

int BMPHandler::valid()
{
  return (m_header[0] == 'B' && m_header[1] == 'M' &&    // BMP file
	      m_header[30] == 0);							 // & Uncompressed
}

int BMPHandler::save(char *filename, int32 width, int32 height, 
					 Format *format, Palette *palette, void *pixels, 
					 void *params)
{
  throw Error("Titan error - BMP save not available");
  return 0;
}


void BMPHandler::Defaults()
{
    // defaults
    m_width=0;
    m_height=0;
    m_paletteflag=0;
    m_imagefile=NULL;
    memset(m_header,0,54);
}

#endif
