/*
   Factory methods implementation for the HERMES library
   Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
   This source code is licensed under the GNU LGPL
  
   Please refer to the file COPYING.LIB contained in the distribution for
   licensing conditions
*/

#include <stdio.h>
#include <stdlib.h>
#include "Factory.h"
#include "HermConf.h"
#include "Clear.h"
#include "Convert.h"
#include "HeadC.h"
#include "HeadX86.h"
#include "HeadMMX.h"


const int PROC_GENERIC = 1;
const int PROC_MMX_PENTIUM = 2;
const int PROC_X86_PENTIUM = 4;

int processor;

void Hermes_Factory_Init()
{ int res=0;

  processor=PROC_GENERIC;

#ifdef X86_ASSEMBLER
  processor|=PROC_X86_PENTIUM;      /* There are no others for the moment */

  res=Hermes_X86_CPU();
/*  printf("CPUID: %x\n",res); */
  if (res&0x800000) processor|=PROC_MMX_PENTIUM;
#endif
}


HermesClearer *Hermes_Factory_getClearer(int32 bits)
{ HermesClearer *tmp;

  /* Try different processors in order of priority..
     Note that for this to work, an MMX processor has to have both MMX and
     X86 flags */

  tmp=(HermesClearer *)malloc(sizeof(HermesClearer));
  if (!tmp) return 0;

  tmp->bits=bits;


#ifdef X86_ASSEMBLER

  if (processor & PROC_MMX_PENTIUM) {
    switch(bits) {
      case 32: tmp->func=ClearMMX_32; return tmp;
      case 24: break;
      case 16: tmp->func=ClearMMX_16; return tmp;
       case 8: tmp->func=ClearMMX_8; return tmp;
    }

  }

  if (processor & PROC_X86_PENTIUM) {
    switch(bits) {
      case 32: tmp->func=ClearX86_32; return tmp;
      case 24: break;
      case 16: tmp->func=ClearX86_16; return tmp;
       case 8: tmp->func=ClearX86_8; return tmp;
    }
  }

#endif /* X86_ASSEMBLER */


  /* C routines here */  

  switch (bits) {
    case 32: tmp->func=ClearC_32; return tmp;
    case 24: tmp->func=ClearC_24; return tmp;
    case 16: tmp->func=ClearC_16; return tmp;
     case 8: tmp->func=ClearC_8; return tmp;
    default: { free(tmp); return 0; }
  }
}


HermesConverter *Hermes_Factory_getConverter(HermesFormat *source,
  HermesFormat *dest)
{ HermesConverter *tmp;
  char found=0;
 
  tmp=(HermesConverter *)malloc(sizeof(HermesConverter));
  if (!tmp) return 0;

  /* Set all conversion routines to null */

  tmp->loopnormal=0;
  tmp->loopstretch=0;
  tmp->normal=0;
  tmp->stretch=0;


  /* Close your eyes right now. :) This will only be executed once so I
     suppose it's not critical. Nevertheless it will be changed in the 
     next release */


#ifdef X86_ASSEMBLER

  if (!found)
  if (processor & PROC_MMX_PENTIUM)
  { tmp->loopnormal=ConvertX86;
    tmp->loopstretch=0;

    switch (source->bits) {
      case 32: if (source->r==0xff0000 && source->g==0xff00 && source->b==0xff)
               { switch (dest->bits) {
	           case 16: if (dest->r==0x7c00 && dest->g==0x3e0 &&
				dest->b==0x1f) {
		              tmp->normal=ConvertMMXp32_16RGB555;
			      found=1; break;
		            }
		            break;
	         }
	       }
    }
  }


  if (!found)
  if (processor & PROC_X86_PENTIUM) {
    tmp->loopnormal=ConvertX86;
    tmp->loopstretch=0;
 
    /* Generate lookup tables */

    Hermes_Factory_x86_Gentables();

    switch (source->bits) {
      case 32: if (source->r==0xff0000 && source->g==0xff00 && source->b==0xff)
	       switch(dest->bits) {
                 case 32: if (dest->r==0xff && dest->g==0xff00 && 
			      dest->b==0xff0000) 
		          { tmp->normal=ConvertX86p32_32BGR888;
			    found=1; break;
			  }
		          else
			  if (dest->r==0xff000000 && dest->g==0xff0000 &&
			      dest->b==0xff00)
			  { tmp->normal=ConvertX86p32_32RGBA888;
			    found=1; break;
			  }
		          else
			  if (dest->r==0xff00 && dest->g==0xff0000 &&
			      dest->b==0xff000000)
			  { tmp->normal=ConvertX86p32_32BGRA888;
			    found=1; break;
			  }
		          break;

                 case 24: if (dest->r==0xff0000 && dest->g==0xff00 && 
			      dest->b==0xff) 
		          { tmp->normal=ConvertX86p32_24RGB888;
			    found=1; break;
			  }
		          else
			  if (dest->r==0xff && dest->g==0xff00 && 
			      dest->b==0xff0000) 
		          { tmp->normal=ConvertX86p32_24BGR888;
			    found=1; break;
			  }
		          break;

                 case 16: if (dest->r==0xf800 && dest->g==0x7e0 && 
			      dest->b==0x1f)
  		          { tmp->normal=ConvertX86p32_16RGB565;
		  	    found=1; break;
		  	  }
		          else
			  if (dest->r==0x1f && dest->g==0x7e0 &&
			      dest->b==0xf800)
			  { tmp->normal=ConvertX86p32_16BGR565;
			    found=1; break;
			  }
		          else
			  if (dest->r==0x7c00 && dest->g==0x3e0 && 
			      dest->b==0x1f)
			  { tmp->normal=ConvertX86p32_16RGB555;
			    found=1; break;
			  }
	                  else
			  if (dest->r==0x1f && dest->g==0x3e0 && 
		  	      dest->b==0x7c00)
			  { tmp->normal=ConvertX86p32_16BGR555;
			    found=1; break;
			  }   
		          break;

                  case 8: if (dest->r==0xe0 && dest->g==0x1c && 
			      dest->b==0x3)
		          { tmp->normal=ConvertX86p32_8RGB332;
			    found=1; break;
			  }
		          break;
               }

      case 16: if (source->r==0xf800 && source->g==0x7e0 && source->b==0x1f)
	       switch(dest->bits) {
                 case 32: if (dest->r==0xff0000 && dest->g==0xff00 && 
			      dest->b==0xff) 
		          { tmp->normal=ConvertX86p16_32RGB888;
			    found=1; break;
			  }
		          else
		          if (dest->r==0xff && dest->g==0xff00 && 
			      dest->b==0xff0000) 
		          { tmp->normal=ConvertX86p16_32BGR888;
			    found=1; break;
			  }
		          else
			  if (dest->r==0xff000000 && dest->g==0xff0000 &&
			      dest->b==0xff00)
			  { tmp->normal=ConvertX86p16_32RGBA888;
			    found=1; break;
			  }
		          else
			  if (dest->r==0xff00 && dest->g==0xff0000 &&
			      dest->b==0xff000000)
			  { tmp->normal=ConvertX86p16_32BGRA888;
			    found=1; break;
			  }
		          break;

                 case 24: if (dest->r==0xff0000 && dest->g==0xff00 && 
			      dest->b==0xff) 
		          { tmp->normal=ConvertX86p16_24RGB888;
			    found=1; break;
			  }
		          else
			  if (dest->r==0xff && dest->g==0xff00 && 
			      dest->b==0xff0000) 
		          { tmp->normal=ConvertX86p16_24BGR888;
			    found=1; break;
			  }
		          break;

                 case 16: if (dest->r==0x1f && dest->g==0x7e0 &&
			      dest->b==0xf800)
			  { tmp->normal=ConvertX86p16_16BGR565;
			    found=1; break;
			  }
		          else
			  if (dest->r==0x7c00 && dest->g==0x3e0 && 
			      dest->b==0x1f)
			  { tmp->normal=ConvertX86p16_16RGB555;
			    found=1; break;
			  }
	                  else
			  if (dest->r==0x1f && dest->g==0x3e0 && 
		  	      dest->b==0x7c00)
			  { tmp->normal=ConvertX86p16_16BGR555;
			    found=1; break;
			  }   
		          break;

                  case 8: if (dest->r==0xe0 && dest->g==0x1c && 
			      dest->b==0x3)
		          { tmp->normal=ConvertX86p16_8RGB332;
			    found=1; break;
			  }
		          break;
               }

       case 8: if (source->indexed)
	       switch (dest->bits) {
	         case 32: tmp->normal=ConvertX86pI8_32;
		          found=1; break;

		 case 24: tmp->normal=ConvertX86pI8_24;
		          found=1; break;

		 case 16: tmp->normal=ConvertX86pI8_16;
		          found=1; break;
	       }

               break;
    }
  }

#endif /* X86_ASSEMBLER */

  if (!found) {
    tmp->loopnormal=ConvertC;
    tmp->loopstretch=0;
  }

  if (!found)
  switch (source->bits) {
    case 32: /* From 32 bit RGB 888 */
             if (source->r==0xff0000 && source->g==0xff00 && source->b==0xff)
             switch (dest->bits) {
  	       case 32: if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_32rgb888_32bgr888;
			  found=1; break;
			}
	                break;

  	       case 24: if (dest->r==0xff0000 && dest->g==0xff00 && 
			    dest->b==0xff)
		        { tmp->normal=ConvertC_32rgb888_24rgb888;
			  found=1; break;
			}
	                else
			if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_32rgb888_24bgr888;
			  found=1; break;
			}  
		        break;
			
 	       case 16: if (dest->r==0xf800 && dest->g==0x7e0 && dest->b==0x1f)
		        { tmp->normal=ConvertC_32rgb888_16rgb565;
			  found=1; break;
			}
	                else
			if (dest->r==0x1f && dest->g==0x7e0 && dest->b==0xf800)
		        { tmp->normal=ConvertC_32rgb888_16bgr565;
			  found=1; break;
			}
	                else
			if (dest->r==0x7c00 && dest->g==0x3e0 && dest->b==0x1f)
			{ tmp->normal=ConvertC_32rgb888_16rgb555;
			  found=1; break;
			}
	                else
			if (dest->r==0x1f && dest->g==0x3e0 && dest->b==0x7c00)
			{ tmp->normal=ConvertC_32rgb888_16bgr555;
			  found=1; break;
			}
	                break;
			
  	        case 8: if (dest->r==0xe0 && dest->g==0x1c && dest->b==0x3 &&
			    !dest->indexed) 
		        { tmp->normal=ConvertC_32rgb888_8rgb332;
			  found=1; break;
			}
		        break;
	       
	     }
	     /* From 32 bit muhmu */
             else
	     if (source->r==0xff<<20 && source->g==0xff<<10 && source->b==0xff)
	     switch (dest->bits) {
	       case 32: if (dest->r==0xff0000 && dest->g==0xff00 && 
			    dest->b==0xff)
		        { tmp->normal=ConvertC_muhmu32_32rgb888;
			  found=1; break;
			}
	                else
		        if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_muhmu32_32bgr888;
			  found=1; break;
			}
	                break;

  	       case 24: if (dest->r==0xff0000 && dest->g==0xff00 && 
			    dest->b==0xff)
		        { tmp->normal=ConvertC_muhmu32_24rgb888;
			  found=1; break;
			}
	                else
			if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_muhmu32_24bgr888;
			  found=1; break;
			}  
		        break;

 	       case 16: if (dest->r==0xf800 && dest->g==0x7e0 && dest->b==0x1f)
		        { tmp->normal=ConvertC_muhmu32_16rgb565;
			  found=1; break;
			}
	                else
			if (dest->r==0x1f && dest->g==0x7e0 && dest->b==0xf800)
		        { tmp->normal=ConvertC_muhmu32_16bgr565;
			  found=1; break;
			}
	                else
			if (dest->r==0x7c00 && dest->g==0x3e0 && dest->b==0x1f)
			{ tmp->normal=ConvertC_muhmu32_16rgb555;
			  found=1; break;
			}
	                else
			if (dest->r==0x1f && dest->g==0x3e0 && dest->b==0x7c00)
			{ tmp->normal=ConvertC_muhmu32_16bgr555;
			  found=1; break;
			}
	                break;

  	        case 8: if (dest->r==0xe0 && dest->g==0x1c && dest->b==0x3 &&
			    !dest->indexed) 
		        { tmp->normal=ConvertC_muhmu32_8rgb332;
			  found=1; break;
			}
		        break;
	     }

             break;
      
    case 24: /* From 24 bit RGB 888 */
             if (source->r==0xff0000 && source->g==0xff00 && source->b==0xff)
             switch (dest->bits) {
  	       case 32: if (dest->r==0xff0000 && dest->g==0xff00 && 
			    dest->b==0xff)
		        { tmp->normal=ConvertC_24rgb888_32rgb888;
			  found=1; break;
			}
	                break;

  	       case 24: if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_24rgb888_24bgr888;
			  found=1; break;
			}  
		        break;

 	       case 16: if (dest->r==0xf800 && dest->g==0x7e0 && dest->b==0x1f)
		        { tmp->normal=ConvertC_24rgb888_16rgb565;
			  found=1; break;
			}
	                else
			if (dest->r==0x7c00 && dest->g==0x3e0 && dest->b==0x1f)
			{ tmp->normal=ConvertC_24rgb888_16rgb555;
			  found=1; break;
			}
	                break;

  	        case 8: if (dest->r==0xe0 && dest->g==0x1c && dest->b==0x3 &&
			    !dest->indexed) 
		        { tmp->normal=ConvertC_24rgb888_8rgb332;
			  found=1; break;
			}
		        break;
	       
	     }

             break;

    case 16:/* From 16 bit RGB 565 */
             if (source->r==0xf800 && source->g==0x7e0 && source->b==0x1f)
             switch (dest->bits) {
  	       case 32: if (dest->r==0xff0000 && dest->g==0xff00 && 
			    dest->b==0xff)
		        { tmp->normal=ConvertC_16rgb565_32rgb888;
			  found=1; break;
			}
	                else
		        if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_16rgb565_32bgr888;
			  found=1; break;
			}
	                break;

  	       case 24: if (dest->r==0xff0000 && dest->g==0xff00 && 
			    dest->b==0xff)
		        { tmp->normal=ConvertC_16rgb565_24rgb888;
			  found=1; break;
			}
	                else
			if (dest->r==0xff && dest->g==0xff00 && 
			    dest->b==0xff0000)
		        { tmp->normal=ConvertC_16rgb565_24bgr888;
			  found=1; break;
			}  
		        break;

 	       case 16: if (dest->r==0x1f && dest->g==0x7e0 && dest->b==0xf800)
		        { tmp->normal=ConvertC_16rgb565_16bgr565;
			  found=1; break;
			}
	                else
			if (dest->r==0x7c00 && dest->g==0x3e0 && dest->b==0x1f)
			{ tmp->normal=ConvertC_16rgb565_16rgb555;
			  found=1; break;
			}
	                else
			if (dest->r==0x1f && dest->g==0x3e0 && dest->b==0x7c00)
			{ tmp->normal=ConvertC_16rgb565_16bgr555;
			  found=1; break;
			}
	                break;

  	        case 8: if (dest->r==0xe0 && dest->g==0x1c && dest->b==0x3 &&
			    !dest->indexed) 
		        { tmp->normal=ConvertC_16rgb565_8rgb332;
			  found=1; break;
			}
		        break; 
	     }
             break;

	     case 8: if (source->indexed) {
               switch (dest->bits) {
	         case 32: tmp->loopnormal=ConvertC_index8_32;
		          tmp->normal=(void (*)(char8*,char8*,int32))0xff;
		          found=1; break;

		 case 24: tmp->loopnormal=ConvertC_index8_24;
		          tmp->normal=(void (*)(char8*,char8*,int32))0xff;
		          found=1; break;

		 case 16: tmp->loopnormal=ConvertC_index8_16;
		          tmp->normal=(void (*)(char8*,char8*,int32))0xff;
		          found=1; break;

	       }
	     }	 
             break;
  }

  if (found) {
    Hermes_FormatCopy(source,&tmp->source);
    Hermes_FormatCopy(dest,&tmp->dest);
    
    return tmp;
  }

  free(tmp);

  return 0;
}




HermesConverter *Hermes_Factory_getEqualConverter(int bits)
{ HermesConverter *tmp;
  char found=0;
 
  tmp=(HermesConverter *)malloc(sizeof(HermesConverter));
  if (!tmp) return 0;

  /* Set all conversion routines to null */

  tmp->loopnormal=0;
  tmp->loopstretch=0;
  tmp->normal=0;
  tmp->stretch=0;


#ifdef X86_ASSEMBLER

  /* Try MMX routines */
  if (!found)
  if (processor & PROC_MMX_PENTIUM)
  switch (bits) {


  }


  /* Try X86 routines */
  if (!found)
  if (processor & PROC_X86_PENTIUM) {
    tmp->loopnormal=ConvertX86;
    tmp->loopstretch=0;
    tmp->stretch=0;
    
    switch (bits) {
      case 32: { tmp->normal=CopyX86p_4byte; found=1; } break;
      case 24: break;
      case 16: { tmp->normal=CopyX86p_2byte; found=1; } break;
       case 8: { tmp->normal=CopyX86p_1byte; found=1; } break;
    }
  }

#endif /* X86_ASSEMBLER */


  if (!found)
  { tmp->loopnormal=ConvertC;
    tmp->loopstretch=0;
    tmp->stretch=0;

    switch (bits) {
      case 32: { tmp->normal=CopyC_4byte; found=1; break; }
      case 24: { tmp->normal=CopyC_3byte; found=1; break; }
      case 16: { tmp->normal=CopyC_2byte; found=1; break; }
       case 8: { tmp->normal=CopyC_1byte; found=1; break; }
    }
  }

  if (found) return tmp;

  free(tmp);

  return 0;
}




void Hermes_Factory_x86_Gentables()
{ 
#ifdef X86_ASSEMBLER
  int i;
  float r,g,b,a;
 
  // Routine taken from PTC 0.72 and modified

  for(i=0;i<256;i++)
  {
    // lower byte

    r = (float)0.0;
    g = (float)((i&0xE0)>>5) * (float)(255.0 / 63.0);
    b = (float)(i&0x1F) * (float)(255.0 / 31.0);
    a = (float)0.0;

    ConvertX86p16_32RGB888_LUT_X86[i*2]=(((int32)r)<<16)|(((int32)g)<<8)|
                                         ((int32)b);
    ConvertX86p16_32BGR888_LUT_X86[i*2]=((int32)r)|(((int32)g)<<8)|
                                        (((int32)b)<<16);
    ConvertX86p16_32RGBA888_LUT_X86[i*2]=(((int32)r)<<24)|(((int32)g)<<16)|
                                         (((int32)b)<<8);
    ConvertX86p16_32BGRA888_LUT_X86[i*2]=(((int32)r)<<8)|(((int32)g)<<16)|
                                         (((int32)b)<<24);

    // upper byte
    r = (float)((i&0xF8)>>3) * (float)(255.0 / 31.0);
    g = (float)((i&0x07)<<3) * (float)(255.0 / 63.0);
    b = (float)0.0;
    a = (float)0.0;

    ConvertX86p16_32RGB888_LUT_X86[i*2+1]=(((int32)r)<<16)|(((int32)g)<<8)|
                                    ((int32)b);
    ConvertX86p16_32BGR888_LUT_X86[i*2+1]=((int32)r)|(((int32)g)<<8)|
                                    (((int32)b)<<16);
    ConvertX86p16_32RGBA888_LUT_X86[i*2+1]=(((int32)r)<<24)|(((int32)g)<<16)|
                                     (((int32)b)<<8);
    ConvertX86p16_32BGRA888_LUT_X86[i*2+1]=(((int32)r)<<8)|(((int32)g)<<16)|
                                     (((int32)b)<<24);
  }
#endif
}

