//
// Console class for PTC 2.0 C++ API, Copyright (c) 1998 Jonathan Matthew
// and Mikko Tiihonen
// The PTC 2.0 C++ API is Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//


// include files
#include <stdio.h>
#include <ctype.h>
#include <conio.h>
#include <stdlib.h>

#include "Area.h"
#include "Clip.h"
#include "Error.h"
#include "Console.h"
#include "VBE.h"
#include "VGA.h"



Console::Console()
{
    // defaults
    m_width  = 0;
    m_height = 0;
    m_pitch  = 0;
    m_locked = false;
    m_vbe = 0;
    m_vga = 0;
    m_offscreen = 0;

    // configuration defaults:
    m_allow_linear = true;
    m_allow_banked = true;
    m_force_offscreen = false;
    m_flexible = true;
    m_wait_retrace = true;
    m_wide_dac = true;
    m_pmode_interface = false;
    m_vga_modes = true;
    m_fakemode_type = FAKEMODE2A;
    m_default_width = 320;
    m_default_height = 200;

    // load user defaults (if available)
    read_config_file("ptc.cfg");
}


Console::~Console()
{
    // close
    close();
}




bool Console::option(const char string[])
{
    // these all use strnicmp so that trailing spaces are ignored.
    // makes it much simpler to add strings to a config file..
    // "echo no vga modes >> ptc.cfg"
    if (!strnicmp(string, "no vbe modes", 12)) {
        m_allow_linear = false;
        m_allow_banked = false;
    } else if (!strnicmp(string, "no linear modes", 15)) {
        m_allow_linear = false;
    } else if (!strnicmp(string, "no banked modes", 15)) {
        m_allow_banked = false;
    } else if (!strnicmp(string, "force offscreen buffering", 25)) {
        m_force_offscreen = true;
    } else if (!strnicmp(string, "force exact match", 17)) {
        m_flexible = false;
    } else if (!strnicmp(string, "no retrace checking", 19)) {
        m_wait_retrace = false;
    } else if (!strnicmp(string, "disable wide dac", 16)) {
        m_wide_dac = false;
    } else if (!strnicmp(string, "no vga modes", 12)) {
        m_vga_modes = false;
    } else if (!strnicmp(string, "enable pmode interface", 22)) {
        m_pmode_interface = true;
    } else if (!strnicmp(string, "fakemode ", 9)) {
        const char *fmtype = &string[9];
        if (!strnicmp(fmtype, "1a", 2)) m_fakemode_type = FAKEMODE1A;
        else if (!strnicmp(fmtype, "1b", 2)) m_fakemode_type = FAKEMODE1B;
        else if (!strnicmp(fmtype, "1c", 2)) m_fakemode_type = FAKEMODE1C;
        else if (!strnicmp(fmtype, "2a", 2)) m_fakemode_type = FAKEMODE2A;
        else if (!strnicmp(fmtype, "2b", 2)) m_fakemode_type = FAKEMODE2B;
        else if (!strnicmp(fmtype, "2c", 2)) m_fakemode_type = FAKEMODE2C;
        else if (!strnicmp(fmtype, "3a", 2)) m_fakemode_type = FAKEMODE3A;
        else if (!strnicmp(fmtype, "3b", 2)) m_fakemode_type = FAKEMODE3B;
        else if (!strnicmp(fmtype, "3c", 2)) m_fakemode_type = FAKEMODE3C;
        else return false;
    } else if (!strnicmp(string, "default width ", 14)) {
        long dw = strtol(&string[14], NULL, 10);
        if (dw != 0) m_default_width = dw;
    } else if (!strnicmp(string, "default height ", 15)) {
        long dw = strtol(&string[15], NULL, 10);
        if (dw != 0) m_default_height = dw;
    } else {
        // option not recognised
        return false;
    }

    return true;
}

void Console::open(const char title[])
{
   Format format(32,0x00FF0000,0x0000FF0000,0x000000FF);
   open(title, format);
}

void Console::open(const char title[],const Format &format)
{
    open(title,m_default_width,m_default_height,format);
}


void Console::open(const char title[],int width,int height,const Format &format)
{

    // close
    close();

    // first try VBE
    try {

        // set up flags to VBE constructor
        int vbeflags = (m_flexible)
                     | (m_wide_dac<<1)
                     | (m_pmode_interface<<2)
                     | (m_allow_banked<<3)
                     | (m_allow_linear<<4);

        m_vbe = new VBE(width, height, format.bytes()-1, format.direct(), vbeflags);

        // setup data
        m_width  = m_vbe->width();
        m_height = m_vbe->height();
        m_pitch  = m_vbe->pitch();
        m_bufmode = m_vbe->pages();
        m_banked = m_vbe->banked();
        m_format = m_vbe->format();

        // set bufmode to 1 when system memory buffering is forced
        if (m_force_offscreen) m_bufmode = 1;

        if (m_bufmode > 1) {
            // set up double/triple buffering
            m_displaypage = 0;
            m_drawpage = 1;
            m_sparepage = 2;
        } else m_drawpage = 0;

        if ((m_bufmode == 1) || m_banked) {
            m_pitch = m_width * m_format.bytes();
            m_offscreen = new char[m_pitch*m_height];
            if (!m_offscreen) throw Error("out of memory");
            memset(m_offscreen, 0, m_pitch*m_height);
        }
    }
    catch (Error e)
    {
        // if VGA allowed, try it, otherwise rethrow
        if (!m_vga_modes) e.rethrow("unable to find a video mode");
        if (m_vbe) {
            delete m_vbe;
            m_vbe = 0;
        }

        int vgamodetype = 0;    // assume fakemode
        if (format.bits() == 8) {
            vgamodetype = 2;
            // rgb332?
            if (format.direct()) vgamodetype = 1;
        }

        m_vga = new VGA(width, height, vgamodetype, m_fakemode_type);

        // setup data
        m_width = m_vga->width();
        m_height = m_vga->height();
        m_format = m_vga->format();
        m_pitch = m_vga->pitch();
        m_bufmode = m_vga->pages();

        // setup offscreen buffer
        m_offscreen = new char[m_height*m_pitch];
        if (!m_offscreen) throw Error("out of memory");
        memset(m_offscreen, 0, m_height*m_pitch);
    }
}


void Console::close()
{
    // unlock the console if required
    if (m_locked) unlock();

    if (m_vbe) {
        delete m_vbe;
        m_vbe  = 0;
    }

    if (m_vga) {
        delete m_vga;
        m_vga = 0;
    }

    if (m_offscreen) {
        delete[] m_offscreen;
        m_offscreen = 0;
    }

    // defaults
    m_width  = 0;
    m_height = 0;
    m_pitch  = 0;
    m_format = Format();
    m_locked = false;
}


inline void Console::update_page_variables()
{
    if (m_bufmode > 1) {
        int tmp = m_displaypage;
        m_displaypage = m_drawpage;
        if (m_bufmode > 2) {
            m_drawpage = m_sparepage;
            m_sparepage = tmp;
        } else {
            m_drawpage = tmp;
        }
    }
}

inline void Console::area_copy(char *dest, char *src, const Area &area,
    int fmtbytes, int width, int pitch)
{
    int btc = (area.right() - area.left()) * fmtbytes;
    int lbs = area.left() * fmtbytes;
    int wbs = width * fmtbytes;

    dest += area.top() * pitch;
    src += area.top() * wbs;
    for (int y=area.top(); y < area.bottom(); y++) {
        memcpy(dest+lbs, src+lbs, btc);
        dest += pitch;
        src += wbs;
    }
}

void Console::update()
{
    if (m_vbe) {
        if (!m_banked) {
            if (m_bufmode > 1) {
                // flip to last drawn page
                m_vbe->page_flip(m_drawpage, m_wait_retrace);
                update_page_variables();
            } else {
                // wait for retrace if requested
                if (m_wait_retrace) m_vbe->page_flip(0, 1);

                char *d = (char *)m_vbe->lock(0);
                char *s = m_offscreen;
                int dpitch = m_vbe->pitch();
                for (int i=0; i<m_height; i++) {
                    memcpy(d, s, m_pitch);
                    d += m_vbe->pitch();
                    s += m_pitch;
                }
                m_vbe->unlock();
            }
        } else {
            m_vbe->banked_load(m_offscreen, m_drawpage, m_wait_retrace);
            update_page_variables();
        }
    } else if (m_vga) {
        if (m_vga->fakemode()) {
            m_vga->fakemode_load(m_offscreen, m_wait_retrace);
        } else {
            if (m_wait_retrace) m_vga->wait_retrace();
            memcpy(m_vga->lock(), m_offscreen, m_pitch*m_height);
            m_vga->unlock();
        }
    }
}

void Console::update(const Area &area)
{
    if (m_vbe) {
        if (m_banked) {
            m_vbe->banked_area_load(m_offscreen, area.top(), area.left(),
                area.bottom(), area.right(), m_drawpage, m_wait_retrace);
            update_page_variables();
        } else if (m_bufmode > 1) {
            // area updates are impossible in double/triple buffered modes.
            update();
        } else {
            if (m_wait_retrace) m_vbe->page_flip(0, 1);

            // copy area to screen
            area_copy((char *)m_vbe->lock(0), m_offscreen,
                area, m_format.bytes(), m_width, m_vbe->pitch());
            m_vbe->unlock();
        }
    } else if (m_vga) {
        if (m_vga->fakemode()) {
            update();
        } else {
            // wait for retrace if necessary
            if (m_wait_retrace) m_vga->wait_retrace();

            // copy area to screen
            area_copy((char *)m_vga->lock(), m_offscreen,
                area, m_format.bytes(), m_width, m_vga->pitch());

            m_vga->unlock();
        }
    }
}




bool Console::key()
{
    // may implement key() using direct access to the BIOS data area,
    // since it would be much faster, and DOS/4GW has weird bugs
    // with realmode interrupts.
    return kbhit();
}


int Console::read()
{
    // may implement read() using direct BIOS data area access.
    return getch();
}




void Console::copy(BaseSurface &surface)
{
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console to surface");
    }
}


void Console::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    // lock console
    void *pixels = lock();

    try
    {
        // load surface pixels to other surface
        surface.load(pixels,width(),height(),source,destination,format(),palette());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console area to surface area");
    }
}




void* Console::lock()
{
    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    // we are locked...
    m_locked = true;

    // get pixels pointer
    if (m_vbe) {
        if ((m_bufmode > 1) && (!m_banked)) {
            return m_vbe->lock(m_drawpage);
        } else return (void *)m_offscreen;
    } else if (m_vga) {
        return (void *)m_offscreen;
    }
    throw Error("there doesn't appear to be a video interface.");
}


void Console::unlock()
{
    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    if (m_vbe) m_vbe->unlock();

    // we are unlocked
    m_locked = false;
}




void Console::load(const void *pixels,int width,int height,const Format &format,const Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,0,0,width,height,width*format.bytes(),console_pixels,0,0,this->width(),this->height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to load pixels to console");
    }
}


void Console::load(const void *pixels,int width,int height,const Area &source,const Area &destination,const Format &format,const Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
	    // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,width,height),source);
        Area clipped_destination = Clip::clip(Area(0,0,this->width(),this->height()),destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),width*format.bytes(),
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to load pixels to console area");
    }
}




void Console::save(void *pixels,int width,int height,const Format &format,const Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(m_format,format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,width*format.bytes());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to save console pixels");
    }
}


void Console::save(void *pixels,int width,int height,const Area &source,const Area &destination,const Format &format,const Palette &palette)
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,this->width(),this->height()),source);
        Area clipped_destination = Clip::clip(Area(0,0,width,height),destination);

            // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),width*format.bytes());
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to save console area pixels");
    }
}




void Console::clear(const Color &color)
{
    // lock console
    void *pixels = lock();

    try
    {
        // request clear
        m_clear.request(format());

        // clear console
        m_clear.clear(pixels,0,0,width(),height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console");
    }
}


void Console::clear(const Color &color,const Area &area)
{
    // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clip::clip(Area(0,0,width(),height()),area);

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console area");
    }
}




void Console::palette(const Palette &palette)
{
    // fail silently if not indexed color
    if (!format().indexed()) return;

    m_palette = palette;
    if (m_vbe) {
        m_vbe->palette(palette.data());
    } else if (m_vga) {
        m_vga->palette(palette.data());
    }
}


const Palette& Console::palette() const
{
    return m_palette;
}




int Console::width() const
{
    return m_width;
}


int Console::height() const
{
    return m_height;
}


int Console::pitch() const
{
    return m_pitch;
}


const Format& Console::format() const
{
    return m_format;
}


const char* Console::name() const
{
    return "DOS";
}

// API extensions:

bool Console::vrstate()
{
    return m_wait_retrace;
}

void Console::vrstate(bool newstate)
{
    m_wait_retrace = newstate;
}

void Console::read_config_file(char *filename)
{
    // Uses stdio for the benefit of DJGPP, which doesn't seem to like
    // iostreams.
    FILE *f = fopen(filename,"rt");
    if (f) {

        // read into memory
        fseek(f, 0, SEEK_END);
        int cfgsize = ftell(f);
        fseek(f, 0, SEEK_SET);
        char *cfg = new char[cfgsize+5];
        memset(cfg, 0, cfgsize+4);
        fread(cfg, cfgsize, 1, f);
        fclose(f);

        // get option strings, one to a line.
        char *b = cfg;
        int cnt = 0;
        while (cnt < cfgsize) {
            // find start of next string
            while (isspace(*b)) b++,cnt++;

            // find end of line character, null terminate the string
            char *w = b;
            while ((*w != '\n') && (*w != 0)) w++, cnt++;
            *w = 0;

            // apply the option string (don't care about failures)
            if ((w - b) > 5) if (option(b) == false) printf(" not");
            b = w;
            b++;
            cnt++;
        }

        // free memory
        delete[] cfg;
    }
}
