//
// Console class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

#ifndef __PTC_CONSOLE_H
#define __PTC_CONSOLE_H

// include files
#include "Base.h"
#include "Copy.h"
#include "Mode.h"
#include "Types.h"
#include "Clear.h"
#include "Format.h"
#include "Palette.h"




class Console : public BaseConsole
{
    public:

        // setup
        Console();
        ~Console();

        // console configuration
        virtual void configure(const char file[]);

        // console option string
        virtual bool option(const char option[]);

        // console modes
        virtual const Mode* modes();

        // console management
        virtual void open(const char title[],int pages = 0);
        virtual void open(const char title[],const Format &format,int pages = 0);
        virtual void open(const char title[],int width,int height,const Format &format,int pages = 0);
        virtual void open(const char title[],const Mode &mode,int pages = 0);
        virtual void close();

        // synchronization
        virtual void flush();
        virtual void finish();
        virtual void update();
        virtual void update(const Area &area);

        // keyboard input
        virtual bool key();
        virtual Key read();

        // copy to surface
        virtual void copy(BaseSurface &surface);
        virtual void copy(BaseSurface &surface,const Area &source,const Area &destination);

        // memory access
        virtual void* lock();
        virtual void unlock();

        // load pixels to console
        virtual void load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        virtual void load(const void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // save console pixels
        virtual void save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette);
        virtual void save(void *pixels,int width,int height,int pitch,const Format &format,const Palette &palette,const Area &source,const Area &destination);

        // clear console
        virtual void clear();
        virtual void clear(const Color &color);
        virtual void clear(const Color &color,const Area &area);

        // console palette
        virtual void palette(const Palette &palette);
        virtual const Palette& palette() const;

        // console clip area
        virtual void clip(const Area &area);

        // data access
        virtual int width() const;
        virtual int height() const;
        virtual int pages() const;
        virtual int pitch() const;
        virtual const Area& area() const;
        virtual const Area& clip() const;
        virtual const Format& format() const;
        virtual const char* name() const;
        virtual const char* title() const;
        virtual const char* information() const;

        //
        // Extension functions
        // -------------------
        //
        // OpenPTC has been designed as a simple portable graphics
        // standard. As such some things that PTC users want fall
        // outside of its scope.
        //
        // For example under the Win32 platform users require a
        // window handle (HWND) in order to do just about anything
        // with the console window. eg: DirectInput requires a HWND
        // to operate correctly, many Win32 API functions require a
        // HWND and so on.
        //
        // To this problem there is a simple solution: Just add the
        // function below and document it in the PTC distribution.
        //
        // In the case of the Win32 distribution, the code below 
        // could be added:
        //
        // HWND window()
        // {
        //     return m_hWnd;
        // }
        //
        // Then the user could just get the console HWND with the
        // following code:
        //
        // HWND hWnd = console.window();
        //
        // The only situation that an extension should not be used
        // is when it duplicates functionality that already exists.
        // Make sure you look at the "Console::option" function
        // before you start adding extensions. Do not use an 
        // extension where an option string will do the job!
        //

        // [insert platform dependent extension functions here]

    private:

        // disallow copy construction
        Console(const Console &console);

        // disallow console assignment
        Console& operator =(const Console &console);

        // console debug checks
        void check_open() const;
        void check_unlocked() const;

        // data
        int m_width;
        int m_height;
        int m_pages;
        int m_pitch;
        Area m_area;
        Area m_clip;
        Format m_format;

        //
        // Console array data
        // ------------------
        //
        // Feel free to increase or decrease the size of the arrays
        // below to match the requirements of your distribution.
        //

        // array data
        Mode m_modes[256];
        char m_title[1024];
        char m_information[1024];

        // flags
        bool m_open;
        bool m_locked;

        // objects
        Copy m_copy;
        Clear m_clear;
        Palette m_palette;
};




#endif
