//
// Interface to x86 assembler routines for the HERMES library
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//
#include <stdio.h>
#include <stdlib.h>
#include "Hermes.h"
#include "Base.h"
#include "Conv_x86.h"
#include "Head_x86.h"


#ifdef NO_ASSEMBLER

void Hermes_x86_init() {}
void Hermes_x86_close() {}
char Hermes_x86_request(HermesFormat *source,HermesFormat *dest,char stretch)
{ return 0; }
void Hermes_x86_palette(int32 *palette) {}
void Hermes_x86_copy(void *s_pixels,int s_x,int s_y,int s_width,int s_height,
		    int s_pitch,void *d_pixels,int d_x,int d_y,int d_width,
		    int d_height,int d_pitch) {}
void Hermes_x86_gentables() {}

#else

static HermesInterface Hermes_x86_Interface;
static Hermes_Generic_Info  Hermes_x86_GI;


void Hermes_x86_init()
{ Hermes_x86.m_source=(HermesFormat *)malloc(sizeof(HermesFormat));
  Hermes_x86.m_dest=(HermesFormat *)malloc(sizeof(HermesFormat));
  Hermes_x86.m_stretch=0;

  Hermes_x86_gentables();
}


void Hermes_x86_close()
{ if(Hermes_x86.m_source)
  { free(Hermes_x86.m_source);
    Hermes_x86.m_source=0;
  }
 
  if(Hermes_x86.m_dest)
  { free(Hermes_x86.m_dest);
    Hermes_x86.m_dest=0;
  }
}


char Hermes_x86_request(HermesFormat *source,HermesFormat *dest,char stretch)
{ char found=0;
  
  if(HermesFormatEquals(source,Hermes_x86.m_source) &&
     HermesFormatEquals(dest,Hermes_x86.m_dest) &&
     stretch==Hermes_x86.m_stretch)
  return 1;

  if(stretch) return 0;

  if(!source->indexed && !dest->indexed)
  switch(source->bpp)
  { case 32: if(source->mask_r==0xff0000 && source->mask_g==0xff00 &&
		source->mask_b==0xff)
             switch(dest->bpp)
	     { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_x86_Interface.conv_func=Convert32_32RGB888;
			  found=1;
			}
			break;

	       case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff)
	                { Hermes_x86_Interface.conv_func=Convert32_24RGB888;
			  found=1;
			}
	                else
			if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			   dest->mask_b==0xff0000)
	                { Hermes_x86_Interface.conv_func=Convert32_24BGR888;
			  found=1;
			}
			break;

	       case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 &&
			   dest->mask_b==0x1f)
	                { Hermes_x86_Interface.conv_func=Convert32_16RGB565;
			  found=1;
			  break;
			}
			else
		        if(dest->mask_r==0x1f && dest->mask_g==0x7e0 &&
			   dest->mask_b==0xf800)
	                { Hermes_x86_Interface.conv_func=Convert32_16BGR565;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x1f)
			{ Hermes_x86_Interface.conv_func=Convert32_16RGB555;
			  found=1;
			  break;
			}
			else
			if(dest->mask_r==0x1f && dest->mask_g==0x3e0 &&
			   dest->mask_b==0x7c00)
			{ Hermes_x86_Interface.conv_func=Convert32_16BGR555;
			  found=1;
			}
			break;

	        case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			   dest->mask_b==0x3)   
		        { Hermes_x86_Interface.conv_func=Convert32_8RGB332;
			  found=1;
			}
		        break;
	     }
	     break;
    
     
    case 16: if(source->mask_r==0xf800 && source->mask_g==0x7e0 &&
		source->mask_b==0x1f)
             { switch(dest->bpp)
	       { case 32: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			     dest->mask_b==0xff)
		          { Hermes_x86_Interface.conv_func=Convert16_32RGB888;
			    found=1;
			    break;
			  }
			  else
		          if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			     dest->mask_b==0xff0000)
		          { Hermes_x86_Interface.conv_func=Convert16_32BGR888;
			    found=1;
			    break;
			  }
			  else
			  if(dest->mask_r==0xff000000 && dest->mask_g==0xff0000 &&
			     dest->mask_b==0xff00)
		          { Hermes_x86_Interface.conv_func=Convert16_32RGBA888;
			    found=1;
			    break;
			  }
			  else
			  if(dest->mask_r==0xff00 && dest->mask_g==0xff0000 &&
			     dest->mask_b==0xff000000)
		          { Hermes_x86_Interface.conv_func=Convert16_32BGRA888;
			    found=1;
			  }

			  break;

 	         case 24: if(dest->mask_r==0xff0000 && dest->mask_g==0xff00 &&
			     dest->mask_b==0xff)
		          { Hermes_x86_Interface.conv_func=Convert16_24RGB888;
			    found=1;
			    break;
			  }
			  else
		          if(dest->mask_r==0xff && dest->mask_g==0xff00 &&
			     dest->mask_b==0xff0000)
		          { Hermes_x86_Interface.conv_func=Convert16_24BGR888;
			    found=1;
			  }
		          break;
		 
	         case 16: if(dest->mask_r==0xf800 && dest->mask_g==0x7e0 &&
			     dest->mask_b==0x1f)
		          { Hermes_x86_Interface.conv_func=Convert16_16RGB565;
			    found=1;
			    break;
			  }
		          else
			  if(dest->mask_r==0x1f && dest->mask_g==0x7e0 &&
			     dest->mask_b==0xf800)
		          { Hermes_x86_Interface.conv_func=Convert16_16BGR565;
			    found=1;
			    break;
			  }
                          else
		          if(dest->mask_r==0x7c00 && dest->mask_g==0x3e0 &&
			     dest->mask_b==0x1f)
		          { Hermes_x86_Interface.conv_func=Convert16_16RGB555;
			    found=1;
			    break;
			  }
		          else
		          if(dest->mask_r==0x1f && dest->mask_g==0x3e0 &&
			     dest->mask_b==0x7c00)
		          { Hermes_x86_Interface.conv_func=Convert16_16BGR555;
			    found=1;
			  }
		          break;
		 
	          case 8: if(dest->mask_r==0xe0 && dest->mask_g==0x1c &&
			     dest->mask_b==0x3)   
		          { Hermes_x86_Interface.conv_func=Convert16_8RGB332;
			    found=1;
			  }
		          break;
			  
	     }
	   }
	   break;
  }
  else
  if(source->indexed && !dest->indexed && source->bpp==8)
  { switch(dest->bpp)
    { case 32: Hermes_x86_Interface.conv_func=ConvertI8_32;
               found=1;
	       break;

      case 24: Hermes_x86_Interface.conv_func=ConvertI8_24;
	       found=1;
	       break;

      case 16: Hermes_x86_Interface.conv_func=ConvertI8_16;
               found=1;
	       break;
    }

    if(found)
    { Hermes_Calculate_Generic_Info(24,16,8,dest,&Hermes_x86_GI);
      ConvertI8_SetLookup(Hermes_Lookup);
    }
  }
  else
  if(source->indexed && source->bpp==8 && dest->indexed && dest->bpp==8)
  { Hermes_x86_Interface.conv_func=ConvertI8_INDEX8;
    found=1;
  }

  if(found)
  { HermesFormatCopy(source,Hermes_x86.m_source);
    HermesFormatCopy(dest,Hermes_x86.m_dest);
    Hermes_x86.m_stretch=stretch;
#ifdef DEBUG
    printf("Hermes X86 Assembler Converter\n");
#endif
    return 1;
  }

  // Couldn't find a converter
  return 0;
}


void Hermes_x86_palette(int32 *palette)
{ int32 i,r,g,b;

  if(!Hermes_x86.m_source->indexed) return;

  // If no left shifts are required, which is the most common case, we 
  // use a faster loop
  if(!Hermes_x86_GI.r_left && !Hermes_x86_GI.g_left && !Hermes_x86_GI.b_left)
  for(i=0;i<256;i++)
  { r=(palette[i]>>Hermes_x86_GI.r_right)&Hermes_x86.m_dest->mask_r;
    g=(palette[i]>>Hermes_x86_GI.g_right)&Hermes_x86.m_dest->mask_g;
    b=(palette[i]>>Hermes_x86_GI.b_right)&Hermes_x86.m_dest->mask_b;
   
    Hermes_Lookup[i]=r|g|b;
  }   
  else
  for(i=0;i<256;i++)
  { r=((palette[i]>>Hermes_x86_GI.r_right)<<Hermes_x86_GI.r_left)&
      Hermes_x86.m_dest->mask_r;
    g=((palette[i]>>Hermes_x86_GI.g_right)<<Hermes_x86_GI.g_left)&
      Hermes_x86.m_dest->mask_g;
    b=((palette[i]>>Hermes_x86_GI.b_right)<<Hermes_x86_GI.b_left)&
      Hermes_x86.m_dest->mask_b;
   
    Hermes_Lookup[i]=r|g|b;
  }  

}


void Hermes_x86_copy(void *s_pixels,int s_x,int s_y,int s_width,int s_height,
		   int s_pitch,void *d_pixels,int d_x,int d_y,int d_width,
		   int d_height,int d_pitch)
{
  if(s_width<=0 || d_width<=0) return;
  if(s_height<=0 || d_height<=0) return;

  // In case the destination gets clipped..
  if(d_width<s_width) s_width=d_width;
  if(d_height<s_height) s_height=d_height;

  Hermes_x86_Interface.s_pixels=(char *)s_pixels;
  Hermes_x86_Interface.s_width=s_width;
  Hermes_x86_Interface.s_height=s_height;
  Hermes_x86_Interface.s_add=s_pitch-s_width*Hermes_x86.m_source->bpp/8;

  Hermes_x86_Interface.d_pixels=(char *)d_pixels;
  Hermes_x86_Interface.d_width=d_width;
  Hermes_x86_Interface.d_height=d_height;
  Hermes_x86_Interface.d_add=d_pitch-d_width*Hermes_x86.m_dest->bpp/8;

  Hermes_x86_Interface.s_pixels+=s_y*s_pitch+s_x*Hermes_x86.m_source->bpp/8;
  Hermes_x86_Interface.d_pixels+=d_y*d_pitch+d_x*Hermes_x86.m_dest->bpp/8;

  ConvertX86(&Hermes_x86_Interface);
}


void Hermes_x86_gentables()
{ int i;
  float r,g,b,a;
 
  // Routine taken from PTC 0.72 and modified

  for(i=0;i<256;i++)
  {
    // lower byte

    r = (float)0.0;
    g = (float)((i&0xE0)>>5) * (float)(255.0 / 63.0);
    b = (float)(i&0x1F) * (float)(255.0 / 31.0);
    a = (float)0.0;

    Convert16_32RGB888_LUT_X86[i*2]=(((int32)r)<<16)|(((int32)g)<<8)|
                                    ((int32)b);
    Convert16_32BGR888_LUT_X86[i*2]=((int32)r)|(((int32)g)<<8)|
                                    (((int32)b)<<16);
    Convert16_32RGBA888_LUT_X86[i*2]=(((int32)r)<<24)|(((int32)g)<<16)|
                                     (((int32)b)<<8);
    Convert16_32BGRA888_LUT_X86[i*2]=(((int32)r)<<8)|(((int32)g)<<16)|
                                     (((int32)b)<<24);

    // upper byte
    r = (float)((i&0xF8)>>3) * (float)(255.0 / 31.0);
    g = (float)((i&0x07)<<3) * (float)(255.0 / 63.0);
    b = (float)0.0;
    a = (float)0.0;

    Convert16_32RGB888_LUT_X86[i*2+1]=(((int32)r)<<16)|(((int32)g)<<8)|
                                    ((int32)b);
    Convert16_32BGR888_LUT_X86[i*2+1]=((int32)r)|(((int32)g)<<8)|
                                    (((int32)b)<<16);
    Convert16_32RGBA888_LUT_X86[i*2+1]=(((int32)r)<<24)|(((int32)g)<<16)|
                                     (((int32)b)<<8);
    Convert16_32BGRA888_LUT_X86[i*2+1]=(((int32)r)<<8)|(((int32)g)<<16)|
                                     (((int32)b)<<24);
  }
}

#endif
