//
// X11 Display abstraction classes for PTC 2.0 C++ API
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// The PTC 2.0 C++ API is (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#ifndef __PTC_X11DISPLAY_H
#define __PTC_X11DISPLAY_H

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <pthread.h>

#include "ptconfig.h"
#include "Area.h"
#include "Format.h"
#include "Copy/Copy.h"
#include "X11Image.h"

enum x11_id{PTC_WINDOW,PTC_DGA};

class X11Display {

public:
  X11Display();
  virtual ~X11Display();

  static bool checkDGA(int width,int height,Display *disp,int screen);
  
  virtual void open(const char title[],int width,int height,Format& format,
		    Display *disp,int screen) = 0;

  // This will always return a windowed console (obviously). The first version
  // fills the whole window, the second one has a custom size
  virtual void open(Display *disp,int screen,Window w,Format& format) = 0;
  virtual void open(Display *disp,int screen,Window window,Format& format,
		    int x,int y,int w,int h) = 0;

  virtual void close() = 0;

  // Set the X11 specific flags
  void flags(long mask) { m_flags=mask; }
  void addflags(long mask) { m_flags|=mask; }

  virtual void update() = 0;

  virtual bool key();
  virtual int read();

  virtual void* lock() = 0;
  virtual void unlock() = 0;

  
  // load pixels to console
  virtual void load(const void *pixels,int width,int height,
		    const Format &format,const int32 palette[]);
  virtual void load(const void *pixels,int width,int height,
		    const Area &source,const Area &destination,
		    const Format &format,const int32 palette[]);

  // save console pixels
  virtual void save(void *pixels,int width,int height,
		    const Format &format,const int32 palette[]) const;
  virtual void save(void *pixels,int width,int height,
		    const Area &source,const Area &destination,
		    const Format &format,const int32 palette[]) const;

  // Set console palette
  virtual void palette(int32 palette[]) = 0;

  int width();
  int height();
  const Format& format();

  virtual int pitch() = 0;

  // X11 helper functions for your enjoyment
  Display* getDisplay() const { return m_disp; }
  int getScreen() const { return m_screen; }

  int id() { return m_id; }

protected:
  // Generate internal format from requested format
  Format getFormat(Format &format);          

  long m_flags;
  unsigned int m_width;
  unsigned int m_height;
  Format m_format;

  Copy m_copy;

  Display *m_disp;                                // X11 Display
  int m_screen;                                   // X11 Screen

  Colormap m_cmap;                                // Color map
  XColor *m_colours;                              // Color cells

  x11_id m_id;
};


class X11WindowDisplay : public X11Display {
public:
  X11WindowDisplay();
  ~X11WindowDisplay();

  virtual void open(const char title[],int width,int height,Format& format,
		    Display *disp,int screen);
  virtual void open(Display *disp,int screen,Window w,Format &format);
  virtual void open(Display *disp,int screen,Window window,Format& format,
		    int x,int y,int w,int h);

  virtual void close();

  virtual void update();

  virtual void* lock();
  virtual void unlock();

  virtual void palette(int32 palette[]);

  virtual int pitch();


  // X11 convenience functions 
  Window getWindow() { return m_window; }
  GC getGC() { return m_gc; }

private:
  X11Image *createImage(Display *disp,int screen,int width,int height,
			Format &format);          // Factory method

  void Handler(void *);                           // Event handler

  Window m_window;

  X11Image *m_primary;
  GC m_gc;
  int m_destx,m_desty;

  Atom m_atom_close;
  pthread_t *m_pthread;
};



#ifdef HAVE_DGA

#include <X11/extensions/xf86vmode.h>

class X11DGADisplay : public X11Display {
public:
  X11DGADisplay();

  ~X11DGADisplay();

  virtual void open(const char title[],int width,int height,Format& format,
		    Display *disp,int screen);
  // Not in DGA mode
  virtual void open(Display *disp,int screen,Window w,Format& format) {}
  virtual void open(Display *disp,int screen,Window window,Format& format,
		    int x,int y,int w,int h) {}

  virtual void close();

  virtual void update();

  virtual void* lock();
  virtual void unlock();

  virtual void palette(int32 palette[]);

  virtual int pitch();

  // check if DGA is available
  static bool check(int width,int height,Display *disp,int screen,long flags);

private:
  XF86VidModeModeInfo **modeinfo;
  int num_modeinfo;
  int previousmode;
  
  char *dga_addr;
  int dga_linewidth;
  int dga_banksize;
  int dga_memsize;
};



#else


class X11DGADisplay : public X11Display {
public:
  static bool check(int width,int height,Display *disp,int screen,long flags)
  { return false;
  }
};

#endif

#endif



