//
// Surface class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Error.h"
#include "Surface.h"
#include "Clip.h"
#include <memory.h>




Surface::Surface(int width,int height,const Format& format)
{
    // setup data
    m_width = width;
    m_height = height;
    m_format = format;
    m_locked = false;
    m_pixels = 0;

    // calculate size of pixels
    int size = width * height * format.bytes();
 
    // check size
    if (!size) throw Error("zero surface size");

    // allocate pixels
    m_pixels = new char8[size];

    // clear pixels    
    memset(m_pixels,0,size);

    // clear palette
    memset(m_palette,0,1024);
}


Surface::~Surface()
{
    // free surface
    delete [] m_pixels;
}


void Surface::copy(BaseSurface &surface)
{ surface.load(m_pixels,m_width,m_height,m_format,m_palette);
}


void Surface::copy(BaseSurface &surface,const Area &source,
  const Area &destination)
{ surface.load(m_pixels,m_width,m_height,source,destination,m_format,
	       m_palette);
}


void* Surface::lock()
{
    // fail if surface is already locked
    if (m_locked) throw Error("surface is already locked");

    // we are locked...
    m_locked = true;

    // lock
    return m_pixels;
}


void Surface::unlock()
{
    // fail if surface is not locked
    if (!m_locked) throw Error("surface is not locked");

    // unlock surface
    m_locked = false;
}


void Surface::load(const void *pixels,int width,int height,
  const Format &format,const int32 palette[])
{
  bool stretch=(m_width!=width)||(m_height!=height);

  m_copy.request(format,m_format,stretch);

  m_copy.palette(palette);

  // Lock and copy (lock() is hidden somewhere in that line, watch out! :)
  m_copy.copy(pixels,0,0,width,height,width*format.bits()/8,
              lock(),0,0,m_width,m_height,pitch());

  unlock();
}


void Surface::load(const void *pixels,int width,int height,
  const Area &source,const Area &destination,const Format &format,
  const int32 palette[])
{
  Area clipped_source = Clip::clip(Area(0,0,width,height),source);
  Area clipped_destination = Clip::clip(Area(0,0,m_width,m_height),destination);

  bool stretch=(source.width()!=destination.width())||
               (source.height()!=destination.height());

  // set copy formats
  m_copy.request(format,m_format,stretch);
    
  // set copy palette
  m_copy.palette(palette);

  // copy pixels to primary
  m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),
	      clipped_source.width(),clipped_source.height(),
              width*format.bits()/8,
              lock(),clipped_destination.left(),clipped_destination.top(),
              clipped_destination.width(),clipped_destination.height(),pitch());

  // unlock
  unlock();
}




void Surface::save(void *pixels,int width,int height,
  const Format &format,const int32 palette[]) 
{
  bool stretch=(m_width!=width)||(m_height!=height);

  // request format conversion
  m_copy.request(m_format,format,stretch);
    
  // set copy palette
  m_copy.palette(palette);

  // copy surface pixels to 'pixels' buffer
  m_copy.copy(m_pixels,0,0,m_width,m_height,pitch(),
	      pixels,0,0,width,height,width*format.bits()/8);
}


void Surface::save(void *pixels,int width,int height,const Area &source,
  const Area &destination,const Format &format,const int32 palette[])
{
  bool stretch=(source.width()!=destination.width())||
               (source.height()!=destination.height());

  // clip source and destination areas
  Area clipped_source = Clip::clip(Area(0,0,m_width,m_height),source);
  Area clipped_destination = Clip::clip(Area(0,0,width,height),destination);

  // request format conversion
  m_copy.request(m_format,format,stretch);
    
  // set copy palette
  m_copy.palette(palette);

  // copy surface pixels to 'pixels' buffer
  m_copy.copy(m_pixels,clipped_source.left(),clipped_source.top(),
	      clipped_source.width(),clipped_source.height(),pitch(),
              pixels,clipped_destination.left(),clipped_destination.top(),
	      clipped_destination.width(),clipped_destination.height(),
	      width*format.bits()/8);
}



void Surface::clear(const Color &color)
{
  // request clear
  m_clear.request(m_format);

  // clear surface
  m_clear.clear(m_pixels,0,0,m_width,m_height,pitch(),color);
}


void Surface::clear(const Color &color,const Area &area)
{   // clip area
    Area clipped_area = Clip::clip(Area(0,0,m_width,m_height),area);

    // request clear
    m_clear.request(m_format);

    // clear surface area
    m_clear.clear(m_pixels,clipped_area.left(),clipped_area.top(),
		  clipped_area.width(),clipped_area.height(),pitch(),color);
}


void Surface::palette(const int32 palette[])
{
    // set palette
    memcpy(m_palette,palette,1024);
}


const int32* Surface::palette()
{ // Return pointer to internal palette
  return m_palette;
}


int Surface::width() const
{
    // get width
    return m_width;
}


int Surface::height() const
{
    // get height
    return m_height;
}


int Surface::pitch() const
{
    // get pitch
    return m_width * m_format.bytes();
}


const Format& Surface::format() const
{
    // get format
    return m_format;
}

