//
// 16 bit to * converters for the HERMES library
// Copyright (c) 1998 Christian Nentwich (brn@eleet.mcb.at)
// This source code is licensed under the GNU LGPL
//
// Please refer to the file COPYING.LIB contained in the distribution for
// licensing conditions
//

#include "Conv_GC.h"
#include "HermConf.h"

// TO 32 BIT RGB 

void Hermes_GC_16rgb565_32rgb888(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 d_pixel;

  for(i=0;i<count;i++,source+=2,dest+=4)
  { d_pixel=(int32)(READ16(source));

    d_pixel=((d_pixel&0xf800)<<8)|((d_pixel&0x7e0)<<5)|((d_pixel&0x1f)<<3);
    d_pixel|=0x030103;

    WRITE32(dest,d_pixel);
  }
}


// TO 32 BIT BGR 

void Hermes_GC_16rgb565_32bgr888(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 d_pixel;

  for(i=0;i<count;i++,source+=2,dest+=4)
  { d_pixel=(int32)(READ16(source));

    d_pixel=((d_pixel&0xf800)>>8)|((d_pixel&0x7e0)<<5)|((d_pixel&0x1f)<<19);
    d_pixel|=0x030103;

    WRITE32(dest,d_pixel);
  }
}


// TO 24 BIT RGB 

void Hermes_GC_16rgb565_24rgb888(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 d_pixel;
  char8 *d_ptr=(char8 *)&d_pixel;

  for(i=0;i<count;i++,source+=2,dest+=3)
  { d_pixel=(int32)(READ16(source));

    d_pixel=((d_pixel&0xf800)<<8)|((d_pixel&0x7e0)<<5)|((d_pixel&0x1f)<<3);
    d_pixel|=0x030103;

    *(dest+R_24)=*(d_ptr+R_32);
    *(dest+G_24)=*(d_ptr+G_32);
    *(dest+B_24)=*(d_ptr+B_32);
  }
}


// TO 24 BIT BGR 

void Hermes_GC_16rgb565_24bgr888(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 d_pixel;
  char8 *d_ptr=(char8 *)&d_pixel;

  for(i=0;i<count;i++,source+=2,dest+=3)
  { d_pixel=(int32)(READ16(source));

    d_pixel=((d_pixel&0xf800)<<8)|((d_pixel&0x7e0)<<5)|((d_pixel&0x1f)<<3);
    d_pixel|=0x030103;

    // Red and blue are swapped here
    *(dest+B_24)=*(d_ptr+R_32);
    *(dest+G_24)=*(d_ptr+G_32);
    *(dest+R_24)=*(d_ptr+B_32);
  }
}



// TO 16 BIT RGB 565 (aka direct copy :)

void Hermes_GC_16rgb565_16rgb565(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;

  // If we are not aligned to a dword, try and convert a single pixel
  if(((int)dest&0x3)!=0)
  { WRITE16(dest,READ16(source));
    count--;
    dest+=2; source+=2;
  }

  // Now copy blocks of dwords
  for(i=0;i<count>>1;i++,source+=4,dest+=4)
  WRITE32(dest,READ32(source));
  
  if(count&1)
  { WRITE16(dest,READ16(source));
  }
}



// TO 16 BIT BGR 565

void Hermes_GC_16rgb565_16bgr565(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 s_pixel;

  // If we are not aligned to a dword, try and convert a single pixel
  if(((int)dest&0x3)!=0)
  { s_pixel=(int32)READ16(source);

    // Swap around R and B, leave G unchanged
    s_pixel=(s_pixel>>11)|(s_pixel&0x7e0)|((s_pixel<<11)&0xf800);

    WRITE16(dest,(short16)s_pixel);

    count--;
    dest+=2; source+=2;
  }

  // Now copy blocks of dwords
  for(i=0;i<count>>1;i++,source+=4,dest+=4)
  { s_pixel=READ32(source);

    // Leave G unchanged, shift R to the right and B to the left 
    s_pixel=(s_pixel&0x07e007e0)|((s_pixel&0xf800f800)>>11)|
            ((s_pixel&0x001f001f)<<11);
    
    WRITE32(dest,s_pixel);
  }

  if(count&1)
  { s_pixel=(int32)READ16(source);

    // Swap around R and B, leave G unchanged
    s_pixel=(s_pixel>>11)|(s_pixel&0x7e0)|((s_pixel<<11)&0xf800);

    WRITE16(dest,(short16)s_pixel);
  }
}



// TO 16 BIT RGB 555

void Hermes_GC_16rgb565_16rgb555(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 s_pixel;

  // If we are not aligned to a dword, try and convert a single pixel
  if(((int)dest&0x3)!=0)
  { s_pixel=(int32)READ16(source);

    // Leave blue untouched, mask red and shift by one, mask green and shift
    // by one
    s_pixel=(s_pixel&0x1f)|((s_pixel&0xf800)>>1)|((s_pixel&0x7c0)>>1);

    WRITE16(dest,(short16)s_pixel);

    count--;
    dest+=2; source+=2;
  }

  // Now copy blocks of dwords
  for(i=0;i<count>>1;i++,source+=4,dest+=4)
  { s_pixel=READ32(source);

    // Leave blue untouched, mask red and shift by one, mask green and shift
    // by one
    s_pixel=(s_pixel&0x001f001f)|((s_pixel&0xf800f800)>>1)|
            ((s_pixel&0x07c007c0)>>1);
   
    WRITE32(dest,s_pixel);
  }

  if(count&1)
  { s_pixel=(int32)READ16(source);

    // Leave blue untouched, mask red and shift by one, mask green and shift
    // by one
    s_pixel=(s_pixel&0x1f)|((s_pixel&0xf800)>>1)|((s_pixel&0x7c0)>>1);

    WRITE16(dest,(short16)s_pixel);
  }
}



// TO 16 BIT BGR 555

void Hermes_GC_16rgb565_16bgr555(char8 *source,char8 *dest,unsigned int count)
{ unsigned int i;
  int32 s_pixel;

  // If we are not aligned to a dword, try and convert a single pixel
  if(((int)dest&0x3)!=0)
  { s_pixel=(int32)READ16(source);

    // Shift red right by 11, mask green and shift right one, shift blue
    // left ten
    s_pixel=((s_pixel&0xf800)>>11)|((s_pixel&0x7c0)>>1)|((s_pixel&0x1f)<<10);

    WRITE16(dest,(short16)s_pixel);

    count--;
    dest+=2; source+=2;
  }

  // Now copy blocks of dwords
  for(i=0;i<count>>1;i++,source+=4,dest+=4)
  { s_pixel=READ32(source);

    // Leave blue untouched, mask red and shift by one, mask green and shift
    // by one
    s_pixel=((s_pixel&0xf800f800)>>11)|((s_pixel&0x07c007c0)>>1)|
            ((s_pixel&0x001f001f)<<10);
   
    WRITE32(dest,s_pixel);
  }

  if(count&1)
  { s_pixel=(int32)READ16(source);

    // Leave blue untouched, mask red and shift by one, mask green and shift
    // by one
    s_pixel=((s_pixel&0xf800)>>11)|((s_pixel&0x7c0)>>1)|((s_pixel&0x1f)<<10);

    WRITE16(dest,(short16)s_pixel);
  }
}


// TO 8 BIT RGB 332

void Hermes_GC_16rgb565_8rgb332(char8 *source,char8 *dest,unsigned int count)
{ int32 s_block,d_block;
  unsigned int i;

  // Align to dword first
  while(count && ((int)dest&0x3)!=0)
  { s_block=(int32)READ16(source);
  
    s_block=((s_block&0xe000)>>8)|((s_block&0x0700)>>6)|
            ((s_block&0x18)>>3);

    *(dest)=(char8)s_block;

    count--;
    source+=2;
    dest++;
  }


  // Write blocks of four pixels
  for(i=0;i<count>>2;i++,source+=8,dest+=4)
  { // Read and process first two pixels
    s_block=READ32(source);
  
    d_block=((s_block&0xe000e000)>>8)|((s_block&0x07000700)>>6)|
            ((s_block&0x00180018)>>3);
    d_block=(d_block&0xff)|((d_block&0xff0000)>>8);

    // And the second two
    s_block=READ32(source+1);

    s_block=((s_block&0xe000e000)>>8)|((s_block&0x07000700)>>6)|
            ((s_block&0x00180018)>>3);
    s_block=(s_block&0xff)|((s_block&0xff0000)>>8);
   
    // Put it all in one dword and write it
    #if defined __LITTLE_ENDIAN__
    d_block|=(s_block<<16);
    #elif defined __BIG_ENDIAN__
    d_block<<=16;
    d_block|=s_block;
    #endif

    WRITE32(dest,d_block);
  }

  // Clean up remaining odd pixel
  if(count&1)
  { s_block=(int32)READ16(source);
  
    s_block=((s_block&0xe000)>>8)|((s_block&0x0700)>>6)|
            ((s_block&0x18)>>3);

    *(dest)=(char8)s_block;
  }
}

// GENERIC ROUTINES


void Hermes_GC_16rgb565_Generic32(char8 *source,char8 *dest,unsigned int count)
{ int32 s_pixel,r,g,b;

  while(count--)
  { s_pixel=(int32)READ16(source);

    r=s_pixel&0xf800;
    g=s_pixel&0x7e0;
    b=s_pixel&0x1f;

    r=((r>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((g>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((b>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    WRITE32(dest,(r|g|b));

    source+=2;
    dest+=4;
  }
}


void Hermes_GC_16rgb565_Generic16(char8 *source,char8 *dest,unsigned int count)
{ short16 s_pixel;
  int32 r,g,b;

  while(count--)
  { s_pixel=READ16(source);

    r=((int32)s_pixel)&0xf800;
    g=((int32)s_pixel)&0x7e0;
    b=((int32)s_pixel)&0x1f;

    r=((r>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((g>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((b>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;

    WRITE16(dest,(short16)(r|g|b));

    source+=2;
    dest+=2;
  }
}



// Even slower, most generic routines from any 16 bit format

void Hermes_GC_Generic16_Generic32(char8 *source,char8 *dest,unsigned int count)
{ int32 s_pixel,r,g,b;

  while(count--)
  { s_pixel=(int32)READ16(source);

    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    WRITE32(dest,(r|g|b));
    
    source+=2;
    dest+=4;
  }
}


void Hermes_GC_Generic16_Generic24(char8 *source,char8 *dest,unsigned int count)
{ int32 s_pixel,r,g,b;
  char8 *d_ptr=(char8 *)((int)&s_pixel);

  while(count--)
  { s_pixel=(int32)READ16(source);

    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    s_pixel=(r|g|b);

    *(dest+R_24)=*(d_ptr+R_32);
    *(dest+G_24)=*(d_ptr+G_32);
    *(dest+B_24)=*(d_ptr+B_32);

    source+=2;
    dest+=3;
  }
}


void Hermes_GC_Generic16_Generic16(char8 *source,char8 *dest,unsigned int count)
{ int32 s_pixel,r,g,b;

  while(count--)
  { s_pixel=(int32)READ16(source);

    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    WRITE16(dest,(short16)(r|g|b));
    
    source+=2;
    dest+=2;
  }
}


void Hermes_GC_Generic16_Generic8(char8 *source,char8 *dest,unsigned int count)
{ int32 s_pixel,r,g,b;

  while(count--)
  { s_pixel=(int32)READ16(source);

    r=((s_pixel>>Hermes_GC_GI.r_right)<<Hermes_GC_GI.r_left)&
      Hermes_Generic_C.m_dest->mask_r;
    g=((s_pixel>>Hermes_GC_GI.g_right)<<Hermes_GC_GI.g_left)&
      Hermes_Generic_C.m_dest->mask_g;
    b=((s_pixel>>Hermes_GC_GI.b_right)<<Hermes_GC_GI.b_left)&
      Hermes_Generic_C.m_dest->mask_b;
    
    *dest=(char8)(r|g|b);
    
    source+=2;
    dest++;
  }
}
