//
// Flower demo for OpenPTC 1.0 Java Implementation
// Copyright (c) Scott Buchanan (aka Goblin)
// This source code is licensed under the GNU GPL
// See http://www.gnu.org/copyleft/gpl.html for details
//

// import classes
import ptc.Error;
import ptc.Timer;
import ptc.Format;
import ptc.Palette;
import ptc.Surface;


class Flower extends ptc.java.Applet
{
    public static void main(String argv[])
    {
        try
        {
            // create console
            ptc.Console console = new ptc.Console();
        
            // create format
            Format format = new Format(8);
 
            // open the console
            console.open("Flower demo",format);

            // create flower
            Flower flower = new Flower();

            // run flower demo
            flower.run(console);
        }
        catch (Error error)
        {
            // report error
            error.report();
        }

        // exit program
        System.exit(0);
    }

    public ptc.base.Console create() throws Error
    {
        // create format
        Format format = new Format(8);

        // create console
        ptc.java.Console console = new ptc.java.Console();

        // open console on applet
        console.open(this,format);

        // return console
        return console;
    }

    public void run(ptc.base.Console console) throws Error
    {
        // create format
        Format format = new Format(8);

        // create surface
        Surface surface = new Surface(320,200,format);

        // create flower surface
        Surface flower = new Surface(640,400,format);

        // generate flower
        generate(flower);

        // create palette
        Palette palette = new Palette();

        // generate palette
        generate(palette);

        // set console palette
        console.palette(palette);

        // set surface palette
        surface.palette(palette);

        // create timer
        Timer timer = new Timer();

        // start timer
        timer.start();

        // loop until a key is pressed
        while (!console.key())
        {
            // get current time from timer
            final double time = timer.time();

            // lock surface pixels
            byte scr[] = (byte[]) surface.lock();
            byte map[] = (byte[]) flower.lock();

            // get surface dimensions
            final int width  = surface.width();
            final int height = surface.height();
            final int mapWidth = flower.width();

            // calculate flower 1 location
            final float xo = (float)(width/2) + 120 * (float)Math.sin( time * 1.1f + 1.5f);
            final float yo = (float)(height/2) + 90 * (float)Math.cos( time * 0.8f + 1.1f);
            final int offset1 = (int)xo + ((int)yo) * mapWidth;

            // calculate flower 2 location
            final float xo2 = (float)(width/2) + 120 * (float)Math.sin( time * 0.9f + 4.2f );
            final float yo2 = (float)(height/2) + 90 * (float)Math.cos( time * 0.7f + 6.9f );
            final int offset2 = (int)xo2 + ((int)yo2) * mapWidth;

            // calculate flower 3 location
            final float xo3 = (float)(width/2) + 120 * (float)Math.sin( time * 0.9f + 3.1f );
            final float yo3 = (float)(height/2) + 90 * (float)Math.cos( time * 1.1f + 1.2f );
            final int offset3 = (int)xo3 + ((int)yo3) * mapWidth;

            // vertical loop
            for (int y=0; y<height; y++)
            {
                // calculate line offset
                final int line = y * mapWidth;

                // horizontal loop
                for (int x=0; x<width; x++)
                {
                    // calculate pixel offset
                    final int pixel = x + line;

                    // sum three flowers into one pixel
                    scr[x + y*width] = (byte) ( map[pixel + offset1] +
                                                map[pixel + offset2] +
                                                map[pixel + offset3] );
                }
            }

            // unlock surfaces
            surface.unlock();
            flower.unlock();

            // copy surface to console
            surface.copy(console);

            // update console
            console.update();
        }
    }

    static void generate(Surface flower) throws Error
    {
        // lock surface pixels
        byte data[] = (byte[]) flower.lock();

        // surface width and height constants for cleaner code
        int fx = flower.width();
        int fy = flower.height();
        int fx2 = fx / 2;
        int fy2 = fy / 2;

        // useful 2*pi constant
        final float TWO_PI = 6.283185307f;

        // generate flower image
        for (int y=0; y < fy; y++)
          for (int x=0; x < fx; x++)
            data[x + y*fx] = (byte) ( 1.0 * Math.cos(18*Math.atan2((y-fy2),(x-fx2)))*255/TWO_PI +
                                      0.3 * Math.sin(15*Math.atan2((y-fy2),(x-fx2)))*255/TWO_PI +
                                      Math.sqrt((y-fy2)*(y-fy2) + (x-fx2)*(x-fx2)) );

        // You might want to move the 1.0 and 0.3 and the 18 and the 15
        // to parameters passed to the generate function...
        // the 1.0 and the 0.3 define the 'height' of the flower, while the
        // 18 and 15 control the number of 'petals'

        // unlock surface
        flower.unlock();
    }

    static void generate(Palette palette) throws Error
    {
        // lock palette data
        int data[] = palette.lock();
    
        // black to yellow
        int i=0;
        int c=0;
        while (i<64)
        {
            data[i] = pack(c,c,0);
            c+=4;
            i++;
        }

        // yellow to red
        c=0;
        while (i<128)
        {
            data[i] = pack(255,255-c,0);
            c+=4;
            i++;
        }

        // red to white
        c=0;
        while (i<192)
        {
            data[i] = pack(255,c,c);
            c+=4;
            i++;
        }

        // white to black
        c=0;
        while (i<256) 
        {
            data[i] = pack(255-c,255-c,255-c);
            c+=4;
            i++;
        }

        // unlock palette
        palette.unlock();
    }

    static int pack(int r,int g,int b)
    {
        // pack color integer
        return (r<<16) | (g<<8) | b; 
    }
}
