//
// Console class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// Modified for port to DOS by Jonathan Matthew (jmatthew@uq.net.au)

// include files
#include <stdio.h>
#include <conio.h>

#include "Area.h"
#include "Clip.h"
#include "Error.h"
#include "Console.h"
#include "VBE.h"



Console::Console()
{
    // defaults
    m_width  = 0;
    m_height = 0;
    m_pitch  = 0;
    m_locked = false;
    m_vbe = 0;

    // configuration defaults:
    m_disallow_banked = 0;
    m_force_triple = 0;
    m_flexible = 1;
    m_wait_retrace = 0;
}


Console::~Console()
{
    // close
    close();
}




bool Console::option(const char string[])
{
    // if i implemented the protected mode interface, which i don't
    // think i will, there would be an option to enable it.
    // it's too buggy to have enabled by default.

    if (!strcmp(string, "no banked modes")) {
        m_disallow_banked = 1;
    } else if (!strcmp(string, "force triple buffering")) {
        m_force_triple = 1;
    } else if (!strcmp(string, "force exact match")) {
        m_flexible = 0;
    } else if (!strcmp(string, "wait for retrace")) {
        m_wait_retrace = 1;
    }

    // option not recognized
    return false;
}




void Console::open(const char title[],const Format &format)
{
    // default width and height
    int width  = 640;
    int height = 400;

    // open console
    open(title,width,height,format);
}


void Console::open(const char title[],int width,int height,const Format &format)
{
    int flags;

    // close
    close();

    // setup flags
    flags = m_flexible | (m_disallow_banked << 1);

    m_vbe = new VBE(width, height, format.bytes()-1, flags);

    // setup data
    m_width  = m_vbe->width();
    m_height = m_vbe->height();
    m_format = m_vbe->format();
    m_pitch  = m_vbe->pitch();
    m_bufmode = m_vbe->pages();

    // check for double buffering when triple buffering is forced
    if (m_force_triple && (m_bufmode != 3)) {
        close();
        throw Error("double buffering disallowed");
    }

    if (m_bufmode > 1) {
        // set up double/triple buffering
        m_displaypage = 0;
        m_drawpage = 1;
        m_sparepage = 2;
    } else {
        // set up single buffered mode (sucks)
        m_pitch = m_width * m_format.bytes();
        m_offscreen = new char[m_width*m_height*m_format.bytes()];
        memset(m_offscreen, 0, m_width*m_height*m_format.bytes());
    }
}


void Console::close()
{
    // unlock the console if required
    if (m_locked) unlock();

    if (m_vbe) {
        delete m_vbe;
        m_vbe  = 0;
    }

    // defaults
    m_width  = 0;
    m_height = 0;
    m_pitch  = 0;
    m_format = Format();
    m_locked = false;
}




void Console::update()
{
    if (m_bufmode > 1) {
        // flip to last drawn page
        m_vbe->page_flip(m_drawpage, m_wait_retrace);

        // update page variables for double or triple buffering
        int tmp = m_displaypage;
        m_displaypage = m_drawpage;

        if (m_bufmode == 3) {
            m_drawpage = m_sparepage;
            m_sparepage = tmp;
        } else {
            m_drawpage = tmp;
        }
    } else {
        // wait for retrace if requested
        if (m_wait_retrace) m_vbe->page_flip(0, 1);

        // copy offscreen buffer to video memory
        char *dest = (char *)m_vbe->page_ptr(0);
        char *src = m_offscreen;
        int wbytes = m_width * m_format.bytes();
        int pitch = m_vbe->pitch();
        for (int y=0; y<m_height; y++) {
            memcpy(dest, src, wbytes);
            dest += pitch;
            src += wbytes;
        }
    }
}


void Console::update(const Area &area)
{
    if (m_bufmode > 1) {
        // area updates are impossible in double/triple buffered modes
        update();
    } else {
        if (m_wait_retrace) m_vbe->page_flip(0, 1);

        char *dest = (char *)m_vbe->page_ptr(0);
        char *src = m_offscreen;
        int btc = (area.right() - area.left()) * m_format.bytes();
        int lbs = area.left() * m_format.bytes();
        int wbs = m_width * m_format.bytes();
        int wbd = m_vbe->pitch();
        dest += area.top() * wbd;
        src += area.top() * wbs;
        for (int y=area.top(); y < area.bottom(); y++) {
            memcpy(dest+lbs, src+lbs, btc);
            dest += wbd;
            src += wbs;
        }
    }
}




bool Console::key()
{
    return kbhit();
}


int Console::read()
{
    return getch();
}




void Console::copy(BaseSurface &surface)
{
    // lock console
    void *pixels = lock();

    try
    {
        // palette data
        const int32 *data = 0;

        // get palette for indexed color
        if (format().indexed()) data = palette();

        // load surface pixels to other surface
        surface.load(pixels,width(),height(),format(),data);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console to surface");
    }
}


void Console::copy(BaseSurface &surface,const Area &source,const Area &destination)
{
    // lock console
    void *pixels = lock();

    try
    {
        // palette data
        const int32 *data = 0;

        // get palette for indexed color
        if (format().indexed()) data = palette();

        // load surface pixels to other surface
        surface.load(pixels,width(),height(),source,destination,format(),data);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to copy console area to surface area");
    }
}




void* Console::lock()
{
    // fail if the console is already locked
    if (m_locked) throw Error("console is already locked");

    // we are locked...
    m_locked = true;

    // get pixels pointer
    if (m_bufmode > 1) {
        return m_vbe->page_ptr(m_drawpage);
    } else return (void *)m_offscreen;
}


void Console::unlock()
{
    // fail if the console is not locked
    if (!m_locked) throw Error("console is not locked");

    // we are unlocked
    m_locked = false;
}




void Console::load(const void *pixels,int width,int height,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,0,0,width,height,width*format.bytes(),console_pixels,0,0,this->width(),this->height(),this->pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to load pixels to console");
    }
}


void Console::load(const void *pixels,int width,int height,const Area &source,const Area &destination,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
	    // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,width,height),source);
        Area clipped_destination = Clip::clip(Area(0,0,this->width(),this->height()),destination);

        // request format conversion
        m_copy.request(format,this->format());
    
        // set copy palettes
        m_copy.palette(palette,this->palette());

        // copy pixels to surface
        m_copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),width*format.bytes(),
                    console_pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to load pixels to console area");
    }
}




void Console::save(void *pixels,int width,int height,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // request format conversion
        m_copy.request(m_format,format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,0,0,this->width(),this->height(),this->pitch(),pixels,0,0,width,height,width*format.bytes());

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();
        
        // error message
        error.rethrow("failed to save console pixels");
    }
}


void Console::save(void *pixels,int width,int height,const Area &source,const Area &destination,const Format &format,const int32 palette[])
{
    // lock console
    void *console_pixels = lock();

    try
    {
        // clip source and destination areas
        Area clipped_source = Clip::clip(Area(0,0,this->width(),this->height()),source);
        Area clipped_destination = Clip::clip(Area(0,0,width,height),destination);

        // request format conversion
        m_copy.request(this->format(),format);
    
        // set copy palettes
        m_copy.palette(this->palette(),palette);

        // copy console pixels to 'pixels' buffer
        m_copy.copy(console_pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch(),
                    pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),width*format.bytes());
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to save console area pixels");
    }
}




void Console::clear(const Color &color)
{
    // lock console
    void *pixels = lock();

    try
    {
        // request clear
        m_clear.request(format());

        // clear console
        m_clear.clear(pixels,0,0,width(),height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console");
    }
}


void Console::clear(const Color &color,const Area &area)
{
    // lock console
    void *pixels = lock();

    try
    {
        // clip area
        Area clipped_area = Clip::clip(Area(0,0,width(),height()),area);

        // request clear
        m_clear.request(format());

        // clear console area
        m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

        // unlock
        unlock();
    }
    catch (Error &error)
    {
        // unlock
        unlock();

        // error message
        error.rethrow("failed to clear console area");
    }
}




void Console::palette(const int32 palette[])
{
    // fail silently if not indexed color
    if (!format().indexed()) return;

    memcpy(m_palette, palette, 1024);
    m_vbe->palette((char *)palette);
}


const int32* Console::palette()
{
    return m_palette;
}




int Console::width() const
{
    return m_width;
}


int Console::height() const
{
    return m_height;
}


int Console::pitch() const
{
    return m_pitch;
}


const Format& Console::format() const
{
    return m_format;
}


const char* Console::name() const
{
    // get name
    return "VBE";
}

bool Console::vrstate()
{
    return m_wait_retrace;
}

void Console::vrstate(bool newstate)
{
    m_wait_retrace = newstate;
}
