//
// DirectX Primary class for OpenPTC 1.0 C++ Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// include files
#include "Core/Error.h"
#include "Core/Color.h"
#include "Core/Clipper.h"
#include "DirectX/Check.h"
#include "DirectX/Primary.h"




DirectXPrimary::DirectXPrimary()
{
    // defaults
    m_locked = 0;
    m_window = 0;
    m_width = 0;
    m_height = 0;
    m_back = 0;
    m_front = 0;
    m_pages = 0;
    m_lpDD2 = 0;
    m_lpDDC = 0;
    m_lpDDS_primary = 0;
    m_lpDDS_secondary = 0;
    m_active = true;
    m_synchronize = true;
    m_fullscreen = false;
    m_primary_width = 0;
    m_primary_height = 0;
    m_secondary_width = 0;
    m_secondary_height = 0;
}


DirectXPrimary::~DirectXPrimary()
{
    // close
    close();
}



void DirectXPrimary::initialize(Win32Window &window,LPDIRECTDRAW2 lpDD2)
{
    // close
    close();

    // setup window
    m_window = &window;

    // setup ddraw
    m_lpDD2 = lpDD2;
}


void DirectXPrimary::primary(int pages,bool video,bool fullscreen)
{
    // check that the number of pages requested is valid
    if (pages<=0) throw Error("invalid number of pages");

    // setup fullscreen flag
    m_fullscreen = fullscreen;

    // create primary surface
    DDSURFACEDESC descriptor;
    descriptor.dwSize  = sizeof(descriptor);
    descriptor.dwFlags = DDSD_CAPS;
    if (pages>1) descriptor.dwFlags |= DDSD_BACKBUFFERCOUNT;
    descriptor.dwBackBufferCount = pages - 1;
    descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
    if (video) descriptor.ddsCaps.dwCaps |= DDSCAPS_VIDEOMEMORY;
    if (pages>1) descriptor.ddsCaps.dwCaps |= DDSCAPS_COMPLEX | DDSCAPS_FLIP;
    DirectXCheck(m_lpDD2->CreateSurface(&descriptor,&m_lpDDS_primary,0),"m_lpDD2->CreateSurface failed in DirectXPrimary::primary");

    // get surface descriptor
    descriptor.dwSize = sizeof(descriptor);
    DirectXCheck(m_lpDDS_primary->GetSurfaceDesc(&descriptor),"m_lpDDS_primary->GetSurfaceDest failed in DirectXPrimary::primary");
    
    // get pixel format
    DDPIXELFORMAT ddpf;
    ddpf.dwSize = sizeof(ddpf);
    DirectXCheck(m_lpDDS_primary->GetPixelFormat(&ddpf),"m_lpDDS_primary->GetPixelFormat failed in DirectXPrimary::primary");

    // setup data
    m_front  = m_lpDDS_primary;
    m_pages  = pages;
    m_width  = descriptor.dwWidth;
    m_height = descriptor.dwHeight;
    m_format = translate(ddpf);

    // setup primary data
    m_primary_width = m_width;
    m_primary_height = m_height;

    // check if fullscreen
    if (!fullscreen)
    {
        // get window client area
        RECT rectangle;
        GetClientRect(m_window->handle(),&rectangle);

        // set data to match window
        m_width  = rectangle.right;
        m_height = rectangle.bottom;
    }
    
    // setup primary surface area
    m_area = Area(0,0,m_width,m_height);

    // setup clip area
    m_clip = m_area;

    // check pages
    if (pages>1)
    {
        // get back surface
        DDSCAPS capabilities;
        capabilities.dwCaps = DDSCAPS_BACKBUFFER;
        DirectXCheck(m_front->GetAttachedSurface(&capabilities,&m_back),"m_lpDDS_primary->GetAttachdSurface failed in DirectXPrimary::primary");
    }
    else
    {
        // only one surface
        m_back = m_front;
    }
    
    // clear all pages
    while (pages--)
    {
        // clear
        clear();
        update();
    }
}


void DirectXPrimary::secondary(int width,int height)
{
    // create the secondary surface
    DDSURFACEDESC descriptor;
    descriptor.dwSize  = sizeof(descriptor);
    descriptor.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
    descriptor.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
    descriptor.dwHeight = height;
    descriptor.dwWidth = width;
    DirectXCheck(m_lpDD2->CreateSurface(&descriptor,&m_lpDDS_secondary,0),"m_lpDD2->CreateSurface failed in DirectXPrimary::secondary");

    // check fullscreen
    if (!m_fullscreen)
    {
        // create a clipper object
        DirectXCheck(m_lpDD2->CreateClipper(0,&m_lpDDC,0),"m_lpDD2->CreateClipper failed in DirectXPrimary::secondary");

        // set clipper to hWnd
        DirectXCheck(m_lpDDC->SetHWnd(0,m_window->handle()),"m_lpDDC->SetHWnd failed in DirectXPrimary::secondary");

        // attach clipper object to primary surface
        DirectXCheck(m_lpDDS_primary->SetClipper(m_lpDDC),"m_lpDDS_primary->SetClipper failed in DirectXPrimary::secondary");
    }

    // setup data
    m_width = width;
    m_height = height;

    // setup primary surface area
    m_area = Area(0,0,m_width,m_height);

    // setup clip area
    m_clip = m_area;

    // setup secondary data
    m_secondary_width = m_width;
    m_secondary_height = m_height;

    // set back surface to secondary
    m_back = m_lpDDS_secondary;

    // clear
    clear();
}


void DirectXPrimary::synchronize(bool update)
{
    // set synchronize flag
    m_synchronize = update;
}


void DirectXPrimary::close()
{
    // check m_lpDDC
    if (m_lpDDC)
    {
        // release interface
        m_lpDDC->Release();

        // null interface
        m_lpDDC = 0;
    }

    // check m_lpDDS_secondary
    if (m_lpDDS_secondary)
    {
        // release interface
        m_lpDDS_secondary->Release();

        // null interface
        m_lpDDS_secondary = 0;
    }

    // check m_lpDDS_primary
    if (m_lpDDS_primary)
    {
        // release interface
        m_lpDDS_primary->Release();

        // null interface
        m_lpDDS_primary = 0;
    }
}




void DirectXPrimary::update()
{
    // block
    block();

    // paint
    paint();

    // check pages
    if (m_pages>1)
    {
        // flip surface
        DirectXCheck(m_front->Flip(0,DDFLIP_WAIT),"m_front->Flip failed in DirectXPrimary::update");
    }
}




void* DirectXPrimary::lock()
{
    // block
    block();

    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof descriptor;

    // check fullscreen and secondary
    if (m_fullscreen || m_back==m_lpDDS_secondary)
    {
        // lock surface
        DirectXCheck(m_back->Lock(0,&descriptor,DDLOCK_WAIT,0),"m_back->Lock failed in DirectXPrimary::lock");

        // update locked pointer
        m_locked = descriptor.lpSurface;
    }
    else
    {
        // get origin of client area
        POINT point = { 0,0 };
        ClientToScreen(m_window->handle(),&point);

        // setup client area rectangle
        RECT rect = { point.x, point.y, point.x+m_width, point.y+m_height };

        // lock surface
        DirectXCheck(m_back->Lock(&rect,&descriptor,DDLOCK_WAIT,0),"m_back->Lock(rect) failed in DirectXPrimary::lock");

        // update locked pointer
        m_locked = descriptor.lpSurface;
    }

    // locked pointer
    return m_locked;
}


void DirectXPrimary::unlock()
{
    // block
    block();

    // unlock surface
    DirectXCheck(m_back->Unlock(m_locked),"m_back->Unlock failed in DirectXPrimary::unlock");
}




void DirectXPrimary::clear()
{
    // block
    block();

    // check fullscreen and secondary
    if (m_fullscreen || m_back==m_lpDDS_secondary)
    {
        // clear surface
        DDBLTFX fx;
        fx.dwSize = sizeof(fx);
        fx.dwFillColor = 0;
        DirectXCheck(m_back->Blt(0,0,0,DDBLT_COLORFILL,&fx),"m_back->Blt failed in DirectXPrimary::clear");
    }
    else
    {
        // todo: replace with hardware clear!

        // check console format
        if (format().direct())
        {
            // direct color
            clear(Color(0,0,0,0),m_area);
        }
        else
        {
            // indexed color
            clear(Color(0),m_area);
        }
    }
}


void DirectXPrimary::clear(const Color &color,const Area &area)
{
    // block
    block();

    // check fullscreen and secondary
    if (m_fullscreen || m_back==m_lpDDS_secondary)
    {
        // clip clear area
        Area clipped = Clipper::clip(area,m_clip);

        // clear color
        DWORD clear_color = 0;
    
        // check color
        if (color.direct())
        {
            // pack clear color integer from direct color class
            clear_color = ( int32( color.a() * 255.0f ) << 24 ) |
                          ( int32( color.r() * 255.0f ) << 16 ) |
                          ( int32( color.g() * 255.0f ) <<  8 ) |
                          ( int32( color.b() * 255.0f ) <<  0 );
        }
        else
        {
            // clear color is an index
            clear_color = color.index();
        }

        // setup blt rect
        RECT rect;
        rect.left = clipped.left();
        rect.top = clipped.top();
        rect.right = clipped.right();
        rect.bottom = clipped.bottom();

        // clear surface
        DDBLTFX fx;
        fx.dwSize = sizeof(fx);
        fx.dwFillColor = clear_color;
        DirectXCheck(m_back->Blt(&rect,0,0,DDBLT_COLORFILL,&fx),"m_back->Blt(rect) failed in DirectXPrimary::clear");
    }
    else
    {
        // todo: replace with accelerated clearing code!

        // lock console
        void *pixels = lock();

        try
        {
            // clip area
            Area clipped_area = Clipper::clip(area,this->clip());

            // request clear
            m_clear.request(format());

            // clear console area
            m_clear.clear(pixels,clipped_area.left(),clipped_area.top(),clipped_area.width(),clipped_area.height(),pitch(),color);

            // unlock
            unlock();
        }
        catch (Error &error)
        {
            // unlock
            unlock();

            // error message
            throw Error("failed to clear console area",error);
        }
    }
}




void DirectXPrimary::palette(const Palette &palette)
{
    // block
    block();

    // check format and palette
    if (!m_format.indexed()) return;

    // get palette data
    const int32 *data = palette.data();

    // convert palette data
    PALETTEENTRY temp[256];
    for (int i=0; i<256; i++)
    {
        temp[i].peRed   = (BYTE) ( (data[i] & 0x00FF0000) >> 16 );
        temp[i].peGreen = (BYTE) ( (data[i] & 0x0000FF00) >> 8  );
        temp[i].peBlue  = (BYTE) ( (data[i] & 0x000000FF) >> 0  );
        temp[i].peFlags = 0;
    }

    // directdraw palette interface
    LPDIRECTDRAWPALETTE lpDDP = 0;

    // get attached surface palette
    if (m_lpDDS_primary->GetPalette(&lpDDP)!=DD_OK)
    {
        // create palette object
        DirectXCheck(m_lpDD2->CreatePalette(DDPCAPS_8BIT|DDPCAPS_ALLOW256|DDPCAPS_INITIALIZE,temp,&lpDDP,0),"m_lpDD2->CreatePalette failed in DirectXPrimary::palette");

        // attach palette to surface
        DirectXCheck(m_lpDDS_primary->SetPalette(lpDDP),"m_lpDDS_primary->SetPalette failed in DirectXPrimary::palette");
    }
    else
    {
        // set palette entries
        DirectXCheck(lpDDP->SetEntries(0,0,256,temp),"lpDDP->SetEntries failed in DirectXPrimary::palette");
    }

    // update palette
    m_palette = palette;
}


const Palette& DirectXPrimary::palette() const
{
    // get palette
    return m_palette;
}




void DirectXPrimary::clip(const Area &area)
{
    // set new clip area
    m_area = Clipper::clip(area,m_area);
}




int DirectXPrimary::width() const
{
    // get width
    return m_width;
}


int DirectXPrimary::height() const
{
    // get height
    return m_height;
}


int DirectXPrimary::pages() const
{
    // get pages
    return m_pages;
}


int DirectXPrimary::pitch() const
{
    // block
    block();

    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof(descriptor);

    // get surface descriptor
    DirectXCheck(m_back->GetSurfaceDesc(&descriptor),"m_back->GetSurfaceDesc failed in DirectXPrimary::pitch");

    // return surface pitch
    return descriptor.lPitch;
}


const Area& DirectXPrimary::area() const
{
    // get area
    return m_area;
}


const Area& DirectXPrimary::clip() const
{
    // get clip
    return m_clip;
}


const Format& DirectXPrimary::format() const
{
    // get format
    return m_format;
}




LPDIRECTDRAWSURFACE DirectXPrimary::lpDDS() const
{
    // return secondary if exists, otherwise primary
    if (m_lpDDS_secondary) return m_lpDDS_secondary;
    else return m_lpDDS_primary;
}                      


LPDIRECTDRAWSURFACE DirectXPrimary::lpDDS_primary() const
{
    // get primary surface interface
    return m_lpDDS_primary;
}                      


LPDIRECTDRAWSURFACE DirectXPrimary::lpDDS_secondary() const
{
    // get secondary surface interface
    return m_lpDDS_secondary;
}                      




void DirectXPrimary::activate()
{
    // activate console
    m_active = true;
}


void DirectXPrimary::deactivate()
{
    // deactivate console
    m_active = false;
}


bool DirectXPrimary::active() const
{
    // is active?
    return m_active;
}


void DirectXPrimary::block() const
{
    // check if active
    if (!active())
    {
        // block until active
        while (!active()) m_window->update();

        // update window
        m_window->update();

        // restore
        restore();
    }

    // check primary surface is not lost
    if (m_lpDDS_primary->IsLost()!=DD_OK)
    {
        // this should not occur!
        throw Error("primary surface lost unexpectedly! report this bug to ptc@gaffer.org immediately!");
    }

    // check secondary surface is not lost
    if (m_lpDDS_secondary && m_lpDDS_secondary->IsLost()!=DD_OK)
    {
        // this should not occur!
        throw Error("secondary surface lost unexpectedly! report this bug to ptc@gaffer.org immediately!");
    }
}


void DirectXPrimary::save() const
{
    // todo: save contents of primary and secondary
}


void DirectXPrimary::restore() const
{
    // restore primary surface
    DirectXCheck(m_lpDDS_primary->Restore(),"m_lpDDS_primary->Restore failed in DirectXPrimary::restore");

    // restore secondary surface
    if (m_lpDDS_secondary) DirectXCheck(m_lpDDS_secondary->Restore(),"m_lpDDS_secondary->Restore failed in DirectXPrimary::restore");

    // todo: restore contents of primary and secondary
}




void DirectXPrimary::paint()
{
    // no painting if not active
    if (!active()) return;

    // check secondary
    if (m_lpDDS_secondary)
    {
        // setup source rectangle
        RECT source = { 0,0,m_secondary_width,m_secondary_height };

        // setup destination rectangle
        RECT destination = { 0,0,m_primary_width,m_primary_height };
        
        // check fullscreen
        if (!m_fullscreen)
        {
            // get origin of client area
            POINT point = { 0,0 };
            ClientToScreen(m_window->handle(),&point);

            // get window client area
            GetClientRect(m_window->handle(),&destination);

            // offset destination rectangle
            destination.left += point.x;
            destination.top += point.y;
            destination.right += point.x;
            destination.bottom += point.y;
        }

        // synchronized paint
        if (m_synchronize)
        {
            // setup fx
            DDBLTFX fx;
            fx.dwSize = sizeof fx;
            fx.dwDDFX = DDBLTFX_NOTEARING;

            try
            {
                // blt from secondary to primary
                DirectXCheck(m_lpDDS_primary->Blt(&destination,m_lpDDS_secondary,&source,DDBLT_WAIT|DDBLT_DDFX,&fx),"m_lpDDS_primary->Blt (synchronized) failed in DirectXPrimary::update");
            }
            catch (Error&)
            {
                // synchronize failure
                m_synchronize = false;
            }
        }
        
        // unsynchonized paint
        if (!m_synchronize)
        {
            // blt from secondary to primary
            DirectXCheck(m_lpDDS_primary->Blt(&destination,m_lpDDS_secondary,&source,DDBLT_WAIT,0),"m_lpDDS_primary->Blt (unsynchronized) failed in DirectXPrimary::update");
        }
    }
}




Format DirectXPrimary::translate(DDPIXELFORMAT const &ddpf)
{
    // pixel format
    Format format;

    // check rgb data
    if (ddpf.dwFlags & DDPF_PALETTEINDEXED8)
    {
        // indexed color
        format = Format(8);
    }
    else if (ddpf.dwFlags & DDPF_RGB)
    {
        // direct color
        format = Format(ddpf.dwRGBBitCount,ddpf.dwRBitMask,ddpf.dwGBitMask,ddpf.dwBBitMask);
    }
    else
    {
        // error message
        throw Error("invalid pixel format");
    }

    // note: i should probably add alpha component support...

    // return format
    return format;
}
