////////////////////////////////////////////////////////////////////////////
//	write your comments to : loadall@hotmail.com
//  chat with the author (loadall) on IRCNet, channel #coders
////////////////////////////////////////////////////////////////////////////
#include "system.hpp"
#include "pak.hpp"

#define CHUNK_VERTEXCOORDS       "v"
#define CHUNK_TEXCOORDS         "vt"
#define CHUNK_NORMALCOORDS      "vn"
#define CHUNK_SUBMESH		     "g"
#define CHUNK_FACE			     "f"
#define CHUNK_USEMATERIAL	"usemtl"
#define CHUNK_ID_UNKNOWN      0  
#define CHUNK_ID_VERTEXCOORDS 1 
#define CHUNK_ID_TEXCOORDS    2
#define CHUNK_ID_NORMALCOORDS 3
#define CHUNK_ID_SUBMESH	  4
#define CHUNK_ID_FACE		  5
#define CHUNK_ID_USEMATERIAL  6

void Tokenizer_init(char* pSrcMin, char* pSrcMax);
char* GetToken();
void Tokenizer_reset();
void Tokenizer_skipLine();
void Tokenizer_skipLineOn();
void Tokenizer_skipLineOff();
int GetChunkId(char* pChunkName);
float parseFloat(char* szSrc);
void CountElements(int* pNumTexCoords, int* pNumVertices, \
				   int* pNumSubMeshes, int* pNumTriangles,\
				   int* pNumTriGroups);
char* ParseTexCoords(char* pDest, int numTexCoords);
char* ParseVertices(char* pDest, int numVertices);
char* ParseSubMeshes(char* pDest, int numSubMeshes);
void ParseVertexIndex(int* pVIndex, int* pTIndex, char* pSrc);
char* ParseTriangles(char* pDest, int numTriGroups, int numTriangles);

typedef struct {
	char*	pSrcMin;
	char*	pSrcMax;
	char*	pSrcCurr;
	char*	pLastToken;
	int		skipLine;
} Tokenizer_t;
Tokenizer_t tokenizer;

char* GetToken()
{
	while (1)
	{
		if (tokenizer.pSrcCurr >= tokenizer.pSrcMax)
			return 0;
		else if (((*tokenizer.pSrcCurr) == '\n') && (!tokenizer.skipLine))
			return 0;	
		else if ((*tokenizer.pSrcCurr) == '#')
				Tokenizer_skipLine();
		else if ((*tokenizer.pSrcCurr) > ' ')
			break;
		tokenizer.pSrcCurr++;
	}
	tokenizer.pLastToken = tokenizer.pSrcCurr;
	while (((*tokenizer.pSrcCurr) > ' ') \
			 && (tokenizer.pSrcCurr < tokenizer.pSrcMax))
		tokenizer.pSrcCurr++;
	(*tokenizer.pSrcCurr) = 0;
	return tokenizer.pLastToken;
}
void Tokenizer_init(char* pSrcMin, char* pSrcMax)
{
	tokenizer.pSrcCurr = tokenizer.pSrcMin = tokenizer.pLastToken = pSrcMin;
	tokenizer.pSrcMax = pSrcMax;
	tokenizer.skipLine = true;
}
void Tokenizer_reset()
{
	tokenizer.pSrcCurr = tokenizer.pSrcMin;
}
void Tokenizer_skipLineOn()
{
	tokenizer.skipLine = true;
}
void Tokenizer_skipLineOff()
{
	tokenizer.skipLine = false;
}
void Tokenizer_skipLine()
{
	while (((*tokenizer.pSrcCurr) != '\n') \
			&& (tokenizer.pSrcCurr < tokenizer.pSrcMax))
		tokenizer.pSrcCurr++;
}
float parseFloat(char* szSrc)
{
	float retValue, sign, comma, decimalPart, divisor;
	_asm {
	pushad
	and	[sign], 0
	mov	esi, [szSrc]
	cmp	byte ptr [esi], '-'
	jne	WW1000
	inc	esi
	or	[sign], 080000000h
	WW1000:
	mov	ebx, esi
	WW1100:
	mov	al, [esi]
	inc	esi
	cmp	al, '0'
	jb	WW1900
	cmp	al, '9'
	ja	WW1900
	jmp	WW1100
	WW1900:
	mov	byte ptr [esi - 1], 0
	mov	[comma], eax
	push	ebx
	call	Sys_atoi
//	add		esp, 4
	mov	[retValue], eax
	fild	[retValue]
	fstp	[retValue]
	cmp	byte ptr [comma], ','
	jne	WW3000
	mov	ebx, esi
	xor	ecx, ecx
	WW2000:
	mov	al, [esi]
	inc	esi
	cmp	al, '0'
	jb	WW2900
	cmp	al, '9'
	ja	WW2900
	inc	ecx
	jmp	WW2000
	WW2900:
	mov	byte ptr [esi - 1], 0
	push	ecx
	push	ebx
	call	Sys_atoi
//	pop		ecx
	pop		ecx
	mov	[decimalPart], eax
	push	1
	pop	edx
	WW2950:
	imul	edx, 0ah
	dec	ecx
	jnz	WW2950
	mov	[divisor], edx
	fild	[decimalPart]
	fidiv	[divisor]
	fadd	[retValue]
	fstp	[retValue]
	mov		eax, [sign]
	or		[retValue], eax
	WW3000:
	popad
	}
	return retValue;
}
int GetChunkId(char* pChunkName)
{
	if (Sys_strcmp(pChunkName, CHUNK_VERTEXCOORDS))
		return CHUNK_ID_VERTEXCOORDS;
	else if (Sys_strcmp(pChunkName, CHUNK_TEXCOORDS))
		return CHUNK_ID_TEXCOORDS;
	else if (Sys_strcmp(pChunkName, CHUNK_NORMALCOORDS))
		return CHUNK_ID_NORMALCOORDS;
	else if (Sys_strcmp(pChunkName, CHUNK_SUBMESH))
		return CHUNK_ID_SUBMESH;
	else if (Sys_strcmp(pChunkName, CHUNK_FACE))
		return CHUNK_ID_FACE;
	else if (Sys_strcmp(pChunkName, CHUNK_USEMATERIAL))
		return CHUNK_ID_USEMATERIAL;
	else
		return CHUNK_ID_UNKNOWN;
}
int AddMeshFile(int hOutputFile, char* szSrcFile)
{
	int numTexCoords, numVertices, numSubMeshes, numTriangles, numTriGroups;
	int srcSize;
	char* pDest;
	char* pSrcBase = (char*)Sys_FileLoad(szSrcFile, &srcSize);
	Tokenizer_init(pSrcBase, pSrcBase + srcSize);
	CountElements(&numTexCoords, &numVertices, \
				  &numSubMeshes, &numTriangles, &numTriGroups);
	Sys_logPrintf("      %i texCoords \r\n" \
				  "      %i vertices \r\n" \
				  "      %i subMeshes\r\n" \
				  "      %i triangles\r\n" \
				  "      %i triGroups\r\n",\
				  numTexCoords, numVertices,\
				  numSubMeshes, numTriangles, numTriGroups);
	int outputSize = ((numTexCoords * sizeof(TexCoords_t)) \
					+ (numVertices * sizeof(VertexCoords_t)) \
					+ (numSubMeshes * sizeof(SubMeshDesc_t)) \
					+ (numTriangles * sizeof(Triangle_t)) \
					+ (numTriGroups * sizeof(TriGroupDesc_t)) \
					+ sizeof(int) * 6);
	char* pDestBase = pDest = (char*)Sys_HeapAllocTemp(outputSize);
	*(int*)pDest = MESH_SIGNATURE;
	pDest += sizeof(int);
	pDest = ParseTexCoords(pDest, numTexCoords);
	pDest = ParseVertices(pDest, numVertices);
	pDest = ParseSubMeshes(pDest, numSubMeshes);
	pDest = ParseTriangles(pDest, numTriGroups, numTriangles);
	Sys_FileWrite("", hOutputFile, pDestBase, outputSize);
	Sys_HeapFreeTemp(pDestBase);
	Sys_MemFree(pSrcBase);
	return outputSize;
}
void CountElements(int* pNumTexCoords, int* pNumVertices, \
				   int* pNumSubMeshes, int* pNumTriangles,\
				   int* pNumTriGroups)
{
	char* pSzLastMaterialName = 0;
	char* pSzCurrMaterialName = 0;
	Tokenizer_reset();
	(*pNumTexCoords) = (*pNumVertices) = (*pNumSubMeshes)\
				     = (*pNumTriangles) = (*pNumTriGroups) = 0;
	while (char* pSrc = GetToken())
	{
		Tokenizer_skipLineOff();
		switch (GetChunkId(pSrc))
		{
			case CHUNK_ID_VERTEXCOORDS:
				(*pNumVertices)++;
				break;
			case CHUNK_ID_TEXCOORDS:
				(*pNumTexCoords)++;
				break;
			case CHUNK_ID_NORMALCOORDS:
				break;
			case CHUNK_ID_SUBMESH:
				if (GetToken())
					(*pNumSubMeshes)++;
				break;
			case CHUNK_ID_FACE:
				while (GetToken())
					(*pNumTriangles)++;
				(*pNumTriangles) -= 2;
				break;
			case CHUNK_ID_USEMATERIAL:
				pSzCurrMaterialName = GetToken();
				if (!pSzLastMaterialName)
					(*pNumTriGroups)++;
				else if (!Sys_strcmp(pSzLastMaterialName,pSzCurrMaterialName))
					(*pNumTriGroups)++;
				pSzLastMaterialName = pSzCurrMaterialName;
				break;
			default:
				Error("Identifieur inconnu : %s", pSrc);
		}
		Tokenizer_skipLineOn();
		Tokenizer_skipLine();
	}
}
char* ParseTexCoords(char* pDest, int numTexCoords)
{
	int i;
	TexCoords_t* pTexCoord;
	Tokenizer_reset();
	(*(int*)pDest) = numTexCoords;
	pDest += 4;
	while (char* pSrc = GetToken())
	{
		Tokenizer_skipLineOff();
		switch (GetChunkId(pSrc))
		{
			case CHUNK_ID_VERTEXCOORDS:
				break;
			case CHUNK_ID_TEXCOORDS:
				pTexCoord = (TexCoords_t*)pDest;
				for (i =0; i < 2; i++)
					pTexCoord->coords[i] = parseFloat(GetToken());
				pDest += sizeof(TexCoords_t);
				break;
			case CHUNK_ID_NORMALCOORDS:
				break;
			case CHUNK_ID_SUBMESH:
				break;
			case CHUNK_ID_FACE:
				break;
			case CHUNK_ID_USEMATERIAL:
				break;
			default:
				Error("Identifieur inconnu : %s", pSrc);
		}
		Tokenizer_skipLineOn();
		Tokenizer_skipLine();
	}
	return pDest;
}
char* ParseVertices(char* pDest, int numVertices)
{
	int i;
	VertexCoords_t* pVertexCoord;
	Tokenizer_reset();
	(*(int*)pDest) = numVertices;
	pDest += 4;
	while (char* pSrc = GetToken())
	{
		Tokenizer_skipLineOff();
		switch (GetChunkId(pSrc))
		{
			case CHUNK_ID_VERTEXCOORDS:
				pVertexCoord = (VertexCoords_t*)pDest;
				for (i =0; i < 3; i++)
					pVertexCoord->coords[i] = parseFloat(GetToken());
				pDest += sizeof(VertexCoords_t);
				break;
			case CHUNK_ID_TEXCOORDS:
				break;
			case CHUNK_ID_NORMALCOORDS:
				break;
			case CHUNK_ID_SUBMESH:
				break;
			case CHUNK_ID_FACE:
				break;
			case CHUNK_ID_USEMATERIAL:
				break;
			default:
				Error("Identifieur inconnu : %s", pSrc);
		}
		Tokenizer_skipLineOn();
		Tokenizer_skipLine();
	}
	return pDest;
}
char* ParseSubMeshes(char* pDest, int numSubMeshes)
{
	SubMeshDesc_t* pSubMesh;
	int triangleIndex = 0;
	Tokenizer_reset();
	(*(int*)pDest) = numSubMeshes;
	pDest += 4;
	while (char* pSrc = GetToken())
	{
		Tokenizer_skipLineOff();
		switch (GetChunkId(pSrc))
		{
			case CHUNK_ID_VERTEXCOORDS:
				break;
			case CHUNK_ID_TEXCOORDS:
				break;
			case CHUNK_ID_NORMALCOORDS:
				break;
			case CHUNK_ID_SUBMESH:
				if (!(pSrc = GetToken()))
					break;
				pSubMesh = (SubMeshDesc_t*)pDest;
				pDest += sizeof(SubMeshDesc_t);
				if (Sys_strlen(pSrc) >= sizeof(pSubMesh->name))
					Error("%s est un nom de subMesh trop long",pSrc);
				Sys_strcpy(pSubMesh->name, pSrc);
				pSubMesh->triIndexMin = triangleIndex;
				if (triangleIndex)
					(--pSubMesh)->triIndexMax = triangleIndex - 1;
				break;
			case CHUNK_ID_FACE:
				while (GetToken())
					triangleIndex++;
				triangleIndex -= 2;
				break;
			case CHUNK_ID_USEMATERIAL:
				break;
			default:
				Error("Identifieur inconnu : %s", pSrc);
		}
		Tokenizer_skipLineOn();
		Tokenizer_skipLine();
	}
	pSubMesh = (SubMeshDesc_t*)pDest;
	pSubMesh--;
	pSubMesh->triIndexMax = triangleIndex;
	return pDest;
}
void ParseVertexIndex(int* pVIndex, int* pTIndex, char* pSrc)
{
	char* pSrcTIndex = Sys_strchr(pSrc, (int)'/');
	if (pSrcTIndex)
	{
		(*(pSrcTIndex++)) = 0;
		(*pTIndex) = Sys_atoi(pSrcTIndex) - 1;
	}
	else
		(*pTIndex) = 0;
	(*pVIndex) = Sys_atoi(pSrc) - 1;
}
char* ParseTriangles(char* pDest, int numTriGroups, int numTriangles)
{
	char* pSzLastMaterialName = 0;
//	char* pOldSrc;
	int vIndex1, vIndex2, vIndex3, tIndex1, tIndex2, tIndex3;
	TriGroupDesc_t* pTriGroup = 0;
	Triangle_t* pTriangle;
	Tokenizer_reset();
	(*(int*)pDest) = numTriangles;
	pDest += 4;
	(*(int*)pDest) = numTriGroups;
	pTriangle = (Triangle_t*)(pDest + 4);
	while (char* pSrc = GetToken())
	{
		Tokenizer_skipLineOff();
		switch (GetChunkId(pSrc))
		{
			case CHUNK_ID_VERTEXCOORDS:
				break;
			case CHUNK_ID_TEXCOORDS:
				break;
			case CHUNK_ID_NORMALCOORDS:
				break;
			case CHUNK_ID_SUBMESH:
				break;
			case CHUNK_ID_FACE:
				ParseVertexIndex(&vIndex1, &tIndex1, GetToken());
				ParseVertexIndex(&vIndex2, &tIndex2, GetToken());
				ParseVertexIndex(&vIndex3, &tIndex3, GetToken());
				pTriangle->vertexIndices[0][0] = vIndex1;
				pTriangle->vertexIndices[0][1] = tIndex1;
				pTriangle->vertexIndices[1][0] = vIndex2;
				pTriangle->vertexIndices[1][1] = tIndex2;
				pTriangle->vertexIndices[2][0] = vIndex3;
				pTriangle->vertexIndices[2][1] = tIndex3;
				pTriangle++;
				pTriGroup->numTriangles++;
				while (pSrc = GetToken())
				{
					vIndex2 = vIndex3;
					tIndex2 = tIndex3;
					ParseVertexIndex(&vIndex3, &tIndex3, pSrc);
					pTriangle->vertexIndices[0][0] = vIndex1;
					pTriangle->vertexIndices[0][1] = tIndex1;
					pTriangle->vertexIndices[1][0] = vIndex2;
					pTriangle->vertexIndices[1][1] = tIndex2;
					pTriangle->vertexIndices[2][0] = vIndex3;
					pTriangle->vertexIndices[2][1] = tIndex3;
					pTriangle++;
					pTriGroup->numTriangles++;
				}
				break;
			case CHUNK_ID_USEMATERIAL:
				pSrc = GetToken();
				if ((pSzLastMaterialName) && Sys_strcmp(pSzLastMaterialName,pSrc))
					break;
				if (pTriGroup)
					Sys_logPrintf("    %s : %i triangles",pTriGroup->textureName,pTriGroup->numTriangles);
				pSzLastMaterialName = pSrc;
				pTriGroup = (TriGroupDesc_t*)pTriangle;
				Sys_strcpy(pTriGroup->textureName, pSrc);
				pTriGroup->numTriangles = 0;
				pTriangle = (Triangle_t*)(pTriGroup + 1);
				break;
			default:
				Error("Identifieur inconnu : %s", pSrc);
		}
		Tokenizer_skipLineOn();
		Tokenizer_skipLine();
	}
	if (pTriGroup)
		Sys_logPrintf("    %s : %i triangles",pTriGroup->textureName,pTriGroup->numTriangles);
	return (char*)pTriangle;
}
