////////////////////////////////////////////////////////////////////////////
//	write your comments to : loadall@hotmail.com
//  chat with the author (loadall) on IRCNet, channel #coders
////////////////////////////////////////////////////////////////////////////
#include "system.hpp"

Matrix4x3 MatrixIdentity = {{1.0f,0.0f,0.0f},\
							{0.0f,1.0f,0.0f},\
							{0.0f,0.0f,1.0f},\
							{0.0f,0.0f,0.0f}};
#define NUM_TEMP_VECTOR_STRINGS 8
#define TEMP_VECTOR_STRING_LENGTH 0x80
char* pTempVectorStrings = 0;
int tempVectorStringIndex = 0;
int MatrixMultiplyCount = 0;
int MatrixOrthonormalizeMask = 0;

char* VectorToString(Vector3 v)
{
	if (!pTempVectorStrings)
		pTempVectorStrings = (char*)Sys_HeapAlloc((NUM_TEMP_VECTOR_STRINGS * TEMP_VECTOR_STRING_LENGTH));
	char* dest = pTempVectorStrings + (tempVectorStringIndex * TEMP_VECTOR_STRING_LENGTH);
	tempVectorStringIndex++;
	if (tempVectorStringIndex >= NUM_TEMP_VECTOR_STRINGS)
		tempVectorStringIndex = 0;
	Sys_sprintf(dest, "(%i,%i,%i)",(int)v[0],(int)v[1],(int)v[2]);
	return dest;
}
void VectorTransform(Matrix4x3 mat, Vector3 dest, Vector3 src)
{
	float x = src[0];
	float y = src[1];
	float z = src[2];
	dest[0] = x*mat[0][0] + y*mat[1][0] + z*mat[2][0] + mat[3][0];
	dest[1] = x*mat[0][1] + y*mat[1][1] + z*mat[2][1] + mat[3][1];
	dest[2] = x*mat[0][2] + y*mat[1][2] + z*mat[2][2] + mat[3][2];
}
void VectorInverseTransform(Matrix4x3 mat, Vector3 dest, Vector3 src)
{
	float x = src[0] - mat[3][0];
	float y = src[1] - mat[3][1];
	float z = src[2] - mat[3][2];
	dest[0] = x*mat[0][0] + y*mat[0][1] + z*mat[0][2];
	dest[1] = x*mat[1][0] + y*mat[1][1] + z*mat[1][2];
	dest[2] = x*mat[2][0] + y*mat[2][1] + z*mat[2][2];
}
void VectorRotate(Matrix4x3 mat, Vector3 dest, Vector3 src)
{
	float x = src[0];
	float y = src[1];
	float z = src[2];
	dest[0] = x*mat[0][0] + y*mat[1][0] + z*mat[2][0];
	dest[1] = x*mat[0][1] + y*mat[1][1] + z*mat[2][1];
	dest[2] = x*mat[0][2] + y*mat[1][2] + z*mat[2][2];
}
void VectorInverseRotate(Matrix4x3 mat, Vector3 dest, Vector3 src)
{
	float x = src[0];
	float y = src[1];
	float z = src[2];
	dest[0] = x*mat[0][0] + y*mat[0][1] + z*mat[0][2];
	dest[1] = x*mat[1][0] + y*mat[1][1] + z*mat[1][2];
	dest[2] = x*mat[2][0] + y*mat[2][1] + z*mat[2][2];
}
float VectorNormalize2(Vector3 v)
{
	return VectorNormalize(v, v);
}
float VectorNormalize(Vector3 d, Vector3 s)
{
	float length = VectorLength(s);
	float rLength;
	VectorCopy(d, s);
	(length)?(rLength = 1.0f / length):(rLength = 0);
	d[0] *= rLength;
	d[1] *= rLength;
	d[2] *= rLength;
	return length;
}
float VectorSqrdLength(Vector3 v)
{
	float length = 0;
	for (int i=0;i<3;i++)
		length += v[i] * v[i];
	return length;
}
float VectorLength(Vector3 v)
{
	float length = 0;
	for (int i=0;i<3;i++)
		length += v[i] * v[i];
	return M_fsqrt(length);
}
float VectorDistance(Vector3 a, Vector3 b)
{
	Vector3 v;
	VectorSub(v, a, b);
	return VectorLength(v);
}
float VectorSqrdDistance(Vector3 a, Vector3 b)
{
	Vector3 v;
	VectorSub(v, a, b);
	return VectorSqrdLength(v);
}
void CrossProduct(Vector3 d,Vector3 a, Vector3 b)
{
	d[0] = a[1] * b[2] - a[2] * b[1];
	d[1] = a[2] * b[0] - a[0] * b[2];
	d[2] = a[0] * b[1] - a[1] * b[0];
}
float DotProduct(Vector3 a, Vector3 b)
{
	return (a[0] * b[0] + a[1] * b[1] + a[2] * b[2]);
}
void VectorScale(Vector3 d, Vector3 s, float scalar)
{
	d[0] = s[0] * scalar;
	d[1] = s[1] * scalar;
	d[2] = s[2] * scalar;
}
void VectorScale2(Vector3 v, float scalar)
{
	v[0] *= scalar;
	v[1] *= scalar;
	v[2] *= scalar;
}
void VectorSub(Vector3 d, Vector3 a, Vector3 b)
{
	d[0]=a[0]-b[0];
	d[1]=a[1]-b[1];
	d[2]=a[2]-b[2];
}
void VectorSub2(Vector3 d, Vector3 s)
{
    d[0]-=s[0];
	d[1]-=s[1];
	d[2]-=s[2];
}
void VectorAdd(Vector3 d, Vector3 a, Vector3 b)
{    
	d[0]=a[0]+b[0];
	d[1]=a[1]+b[1];
	d[2]=a[2]+b[2];
}
void VectorAdd2(Vector3 d, Vector3 s)
{
	d[0]+=s[0];
	d[1]+=s[1];
	d[2]+=s[2];
}
void VectorNegate(Vector3 d, Vector3 s)
{
    d[0]=-s[0];
	d[1]=-s[1];
	d[2]=-s[2];
}
void VectorNegate2(Vector3 v)
{
	v[0]=-v[0];
	v[1]=-v[1];
	v[2]=-v[2];
}
bool VectorIsNull(Vector3 v)
{
	return ((v[0] == 0.0f) && (v[1] == 0.0f) && (v[2] == 0.0f));
}
void VectorRandom(Vector3 v)
{
	v[0] = M_Rand() - 0.5f;
	v[1] = M_Rand() - 0.5f;
	v[2] = M_Rand() - 0.5f;
	VectorNormalize2(v);
}
//////////////////////////////////////////////////////////////////////
float M_fsqrt(float f)
{
	float retValue;
	_asm {
	fld		[f]
	fsqrt
	fstp	[retValue]
	}
	return retValue; 
}
float M_fcos(float angle)
{
	float retValue;
	_asm {
	fld		[angle]
	fcos
	fstp	[retValue]
	}
	return retValue;
}
float M_fsin(float angle)
{
	float retValue;
	_asm {
	fld		[angle]
	fsin
	fstp	[retValue]
	}
	return retValue;
}
float M_fabs(float f)
{
	int retValue = *((int*)&f);
	retValue &= 0x7fffffff;
	return *((float*)&retValue);
}
float M_fmod(float a, float b)
{
	float retValue;
	_asm {
	fld		[b]
	fld		[a]
	fprem
	fstp	[retValue]
	fcomp
	}
	return retValue;
}
float M_fatan2(float n, float d)
{
	float angle;
	if (d == 0)
		return 0;
	_asm {
	fld		[n]
	fabs
	fld		[d]
	fabs
	fpatan
	fstp	[angle]
	}
	if ((n < 0) && (d > 0))
		angle = - angle;
	else if ((n < 0) && (d < 0))
		angle += F_PI;
	else if ((n > 0) && (d < 0))
		angle = F_PI - angle;
	if (angle < 0)
		angle += (2.0f * F_PI);
	return angle;
}
float M_AngleMod(float Angle)
{
	while (Angle > (2.0f*F_PI))
		Angle -= (2.0f*F_PI);
	while (Angle < 0.0f)
		Angle += (2.0f*F_PI);
	return Angle;
}
void M_AnglesVectorMod(Vector3 AnglesVector)
{
	for (int i=0; i<3; i++)
		AnglesVector[i] = M_AngleMod(AnglesVector[i]);
}
//////////////////////////////////////////////////////////////////////
void Matrix_GetAngles(Vector3 angles, Matrix4x3 matrix)
{
	float cosY, sinY, cosX;
	angles[1] = M_fatan2(matrix[2][0],matrix[2][2]);
	cosY = M_fcos(angles[1]);
	sinY = M_fsin(angles[1]);
	cosX = (M_fabs(cosY) > M_fabs(sinY))?(matrix[2][2]/cosY):(matrix[2][0]/sinY);
	angles[0] = M_fatan2(-matrix[2][1], cosX);
	angles[2] = (cosX)?(M_fatan2(matrix[0][1]/cosX, matrix[1][1]/cosX)):0;
}
void Matrix_SetProjectionTransform(float* mat, float fFOV, float fAspect,
                                   float fNearPlane, float fFarPlane)
{
	float halfFOV = fFOV / 2;
	float h = M_fcos(halfFOV) / M_fsin(halfFOV);
	float w = h * fAspect;
    float Q = fFarPlane / (fFarPlane - fNearPlane);
	Sys_memClear(mat, 16*sizeof(float));
	mat[0*4+0] = w;
	mat[1*4+1] = h;
	mat[2*4+2] = Q;
	mat[2*4+3] = 1.0f;
	mat[3*4+2] = -Q * fNearPlane;
}
/*void Matrix_SetViewTransform(Matrix4x3 mat, Vector3 vFrom,
                             Vector3 vAt, Vector3 vWorldUp)
{
    Vector3 a, b;
	mat[0][3] = mat[1][3] = mat[2][3] = 0.0f;
	mat[3][3] = 1.0f;
	VectorCopy(mat[3], vFrom);
	VectorSub(mat[2], vAt, vFrom);
    float fLength = VectorNormalize2(mat[2]);
    if (fLength < 1e-6f)
		Error("Math_setViewMatrix() : fLength < 1e-6f");
	VectorScale(a, mat[2], DotProduct(mat[2], vWorldUp));
	VectorSub(mat[1], vWorldUp, a);
	if (VectorLength(mat[1]) < 1e-6f)
	{
		VectorSet(a, 0, 1.0f, 0);
		VectorScale(b, mat[2], mat[2][1]);
		VectorSub(mat[1], a, b); 
		if (VectorLength(mat[1]) < 1e-6f)
		{
			VectorSet(a, 0, 0, 1.0f);
			VectorScale(b, mat[2], mat[2][2]);
			VectorSub(mat[1], a, b);
		}
	}	
	VectorNormalize2(mat[1]);
	CrossProduct(mat[0], mat[1], mat[2]);
}*/
void Matrix_Set(Matrix4x3 mat, Vector3 translation, Vector3 rotation, Vector3 scaling)
{
	Matrix_SetIdentity(mat);
	if (rotation)
	{
		float cosX = M_fcos(rotation[0]);
		float sinX = M_fsin(rotation[0]);
		float cosY = M_fcos(rotation[1]);
		float sinY = M_fsin(rotation[1]);
		float cosZ = M_fcos(rotation[2]);
		float sinZ = M_fsin(rotation[2]);
/*m1=cbch+sbspsh
m2=cbshsp-sbch
m3=shcp
m4=sbcp
m5=cbcp
m6=-sp
m7=sbchsp-cbsh
m8=sbsh+cbchsp
m9=chcp*/
		mat[0][0] = cosZ*cosY + sinZ*sinX*sinY;
		mat[1][0] = cosZ*sinY*sinX - sinZ*cosY;
		mat[2][0] = sinY*cosX;

		mat[0][1] = sinZ*cosX;
		mat[1][1] = cosZ*cosX;
		mat[2][1] = -sinX;

		mat[0][2] = sinZ*cosY*sinX - cosZ*sinY;
		mat[1][2] = sinZ*sinY + cosZ*cosY*sinX;
		mat[2][2] = cosY*cosX;
	}
	if (translation)
		VectorCopy(mat[3], translation);
	if (scaling)
		Matrix_Scale(mat, mat, scaling);
}
void Matrix_OrthoNormalize(Matrix4x3 mat)
{
	CrossProduct(mat[0], mat[1], mat[2]);
	CrossProduct(mat[1], mat[2], mat[0]);
	VectorNormalize2(mat[0]);
	VectorNormalize2(mat[1]);
	VectorNormalize2(mat[2]);
}
void Matrix_MultiplyRotations(Matrix4x3 dest, Matrix4x3 a, Matrix4x3 b)
{
	#define MATRIX_ORTHONORMALIZE_MASK 0xff
	Matrix4x3 m;
	m[0][0] = a[0][0]*b[0][0] + a[0][1]*b[1][0] + a[0][2]*b[2][0];
	m[0][1] = a[0][0]*b[0][1] + a[0][1]*b[1][1] + a[0][2]*b[2][1];
	m[0][2] = a[0][0]*b[0][2] + a[0][1]*b[1][2] + a[0][2]*b[2][2];
	m[1][0] = a[1][0]*b[0][0] + a[1][1]*b[1][0] + a[1][2]*b[2][0];
	m[1][1] = a[1][0]*b[0][1] + a[1][1]*b[1][1] + a[1][2]*b[2][1];
	m[1][2] = a[1][0]*b[0][2] + a[1][1]*b[1][2] + a[1][2]*b[2][2];
	m[2][0] = a[2][0]*b[0][0] + a[2][1]*b[1][0] + a[2][2]*b[2][0];
	m[2][1] = a[2][0]*b[0][1] + a[2][1]*b[1][1] + a[2][2]*b[2][1];
	m[2][2] = a[2][0]*b[0][2] + a[2][1]*b[1][2] + a[2][2]*b[2][2];
	VectorCopy(dest[0], m[0]);
	VectorCopy(dest[1], m[1]);
	VectorCopy(dest[2], m[2]);
	if ((MatrixMultiplyCount & MATRIX_ORTHONORMALIZE_MASK) == MatrixOrthonormalizeMask)
		Matrix_OrthoNormalize(dest);
	MatrixMultiplyCount++;
	if (MatrixMultiplyCount > MATRIX_ORTHONORMALIZE_MASK)
	{
		MatrixMultiplyCount = 0;
		MatrixOrthonormalizeMask = ((MatrixOrthonormalizeMask+1) & MATRIX_ORTHONORMALIZE_MASK);
	}
}
void Matrix_Rotate(Matrix4x3 Dest, Matrix4x3 Src, Vector3 Angles)
{
	Matrix4x3 MatrixRotation;
	Matrix_Set(MatrixRotation, 0, Angles, 0);
	Matrix_MultiplyRotations(Dest, Src, MatrixRotation);
}
void Matrix_Transpose(Matrix4x3 dest, Matrix4x3 src)
{
	Matrix4x3 tmpMat;
	for (int i=0; i<3; i++)
		for (int j=0; j<3; j++)
				tmpMat[i][j] = src[j][i];
	tmpMat[3][0] = -DotProduct(src[3], src[0]);
	tmpMat[3][1] = -DotProduct(src[3], src[1]);
	tmpMat[3][2] = -DotProduct(src[3], src[2]);
	Matrix_Copy(dest, tmpMat);
}
void Matrix_Scale(Matrix4x3 dest, Matrix4x3 src, Vector3 scaling)
{
	if (dest != src)
		Matrix_Copy(dest, src);
	dest[0][0] *= scaling[0];
	dest[1][0] *= scaling[0];
	dest[2][0] *= scaling[0];
	dest[0][1] *= scaling[1];
	dest[1][1] *= scaling[1];
	dest[2][1] *= scaling[1];
	dest[0][2] *= scaling[2];
	dest[1][2] *= scaling[2];
	dest[2][2] *= scaling[2];
}
void Matrix_SetIdentity(Matrix4x3 mat)
{
	Matrix_Copy(mat, MatrixIdentity);
}
void Matrix_Copy(Matrix4x3 d, Matrix4x3 s)
{
	Sys_memCopy(d, s, sizeof(Matrix4x3));
}
void Matrix_Clear(Matrix4x3 mat)
{
	Sys_memClear(mat,sizeof(Matrix4x3));
}
//////////////////////////////////////////////////////////////////////
int M_GetNextSeed(int seed);
#define RANDOMTABLE_SIZE 250
int* RandomTable = 0;
int RandomGenIndex;

int M_GetNextSeed(int seed)
{
	seed = (seed * 0x015a4e35) + 1;
	return ((seed >> 0x10) & 0x7fff);
}
void M_Randomize(int seed)
{
	unsigned int mask, msb;
	RandomGenIndex = 0;
	if (!RandomTable)
		RandomTable = (int*)Sys_HeapAlloc(RANDOMTABLE_SIZE * sizeof(int));
	for (int i=0; i<RANDOMTABLE_SIZE; i++)
		RandomTable[i] = seed = M_GetNextSeed(seed);
	for (i=0; i<RANDOMTABLE_SIZE; i++)
		if ((seed = M_GetNextSeed(seed)) > 16384)
			RandomTable[i] |= 0x8000;
	mask = 0xffff; msb = 0x8000;
	for (i=0; i<16; i++)
	{
		int j = 11 * i + 3;
		RandomTable[j] &= mask;
		RandomTable[j] |= msb;
		mask >>= 1;
		msb >>= 1;
	}
}
float M_Rand()
{

	int newRand, j;
	if (RandomGenIndex >= 147)
		j = RandomGenIndex - 147;
	else
		j = RandomGenIndex + 103;
	newRand = RandomTable[RandomGenIndex] ^= RandomTable[j];
	if (++RandomGenIndex >= RANDOMTABLE_SIZE)
		RandomGenIndex = 0;
	return ((float)newRand / 65536);
}
//////////////////////////////////////////////////////////////////////
void M_BBoxAddPoint(Vector3 mins, Vector3 maxs, Vector3 point)
{
	if (point[0] < mins[0])
		mins[0] = point[0];
	if (point[1] < mins[1])
		mins[1] = point[1];
	if (point[2] < mins[2])
		mins[2] = point[2];
	if (point[0] > maxs[0])
		maxs[0] = point[0];
	if (point[1] > maxs[1])
		maxs[1] = point[1];
	if (point[2] > maxs[2])
		maxs[2] = point[2];
}
//////////////////////////////////////////////////////////////////////
void M_GetNormalFromTriangle(Vector3 normal, Vector3 a, Vector3 b, Vector3 c)
{
		Vector3 v1, v2;
		VectorSub(v1, b, a);
		VectorSub(v2, c, b);
		CrossProduct(normal, v2, v1);
		VectorNormalize2(normal);
}
//////////////////////////////////////////////////////////////////////
void M_BBoxGetCorners(Vector3* corners, Vector3 mins, Vector3 maxs)
{
	VectorSet(corners[0], mins[0], mins[1], mins[2]);
	VectorSet(corners[1], mins[0], mins[1], maxs[2]);
	VectorSet(corners[2], mins[0], maxs[1], mins[2]);
	VectorSet(corners[3], maxs[0], mins[1], mins[2]);
	VectorSet(corners[4], maxs[0], maxs[1], mins[2]);
	VectorSet(corners[5], mins[0], maxs[1], maxs[2]);
	VectorSet(corners[6], maxs[0], mins[1], maxs[2]);
	VectorSet(corners[7], maxs[0], maxs[1], maxs[2]);
}
void Mesh_GetBBox(Mesh_t* mesh, Vector3 mins, Vector3 maxs, 
				Matrix4x3 matrix, Vector3 translation, Vector3 rotation, Vector3 scaling)
{
	Vector3 corners[8], *corner;
	Matrix4x3 tmpMatrix;
	if (matrix)
		if (scaling)
			Matrix_Scale(tmpMatrix, matrix, scaling);
		else
			Matrix_Copy(tmpMatrix, matrix);
	else
		Matrix_Set(tmpMatrix, translation, rotation, scaling);
	M_BBoxGetCorners(corners, mesh->Mins, mesh->Maxs);
	corner = &corners[0];
	for (int i=0; i<8; i++)
	{
		VectorTransform(tmpMatrix, *corner, *corner);
		if (i == 0)
		{
			VectorCopy(mins, *corner);
			VectorCopy(maxs, *corner);
		}
		else
			M_BBoxAddPoint(mins, maxs, *corner);
		corner++;
	}
}
void Mesh_SetTrianglesDesc(Mesh_t* mesh)
{
	char* pDest = mesh->pTriangles;
	TriangleDesc_t* pTriDesc = mesh->pTriDesc;
	for (int i=0; i<mesh->NumTriGroups; i++)
	{
		TriGroup_t* pTriGroup = (TriGroup_t*)pDest;
		pDest += sizeof(TriGroup_t);
		for (int j=0; j<pTriGroup->NumVertices; j++)
		{
			float* pVertex = (float*)pDest;
			float distance = VectorLength(pVertex);
			if (((i+j) == 0) || (distance > mesh->Radius))
				mesh->Radius = distance;
			if ((i+j) == 0)
			{
				VectorCopy(mesh->Mins,pVertex);
				VectorCopy(mesh->Maxs,pVertex);
			}
			M_BBoxAddPoint(mesh->Mins, mesh->Maxs, pVertex);
			if (!(j % 3))
			{
				float* a = (float*)pDest;
				float* b = (float*)(pDest +   sizeof(LVERTEX));
				float* c = (float*)(pDest + 2*sizeof(LVERTEX));
				M_GetNormalFromTriangle(pTriDesc->Normal, a, b, c);
				VectorCopy(pTriDesc->Mins, a);
				VectorCopy(pTriDesc->Maxs, a);
				M_BBoxAddPoint(pTriDesc->Mins, pTriDesc->Maxs, b);
				M_BBoxAddPoint(pTriDesc->Mins, pTriDesc->Maxs, c);
				pTriDesc->pVertices = (LVERTEX*)pDest;
				Sys_logPrintf("mins = %s, maxs = %s, normal = %s",
							VectorToString(pTriDesc->Mins),
							VectorToString(pTriDesc->Maxs),
							VectorToString(pTriDesc->Normal));
				pTriDesc++;
			}
			pDest += sizeof(LVERTEX);
		}
	}
}
