/***************************************************************************
 *   Copyright (C) 2012                                                    *
 *   Anatole Duprat <anatole.duprat@gmail.com>                             *
 *   Charles Bulckaen  <xtrium@frequency.fr>                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU General Public License           *
 *   as published by the Free Software Foundation; either version 2        *
 *   of the License, or (at your option) any later version.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA          *
 ***************************************************************************/



#include <QtCore>
#include <QStringList>
#include "sceneeditortreeitem.hh"

SceneEditorTreeItem::SceneEditorTreeItem(QString varName, float* mappedValue, float defaultValue, SceneEditorTreeItem* parent) {
    _varName = QString(varName);
    _mappedValue = mappedValue;
    _defaultValue = defaultValue;
    setParent(parent);
}

SceneEditorTreeItem::~SceneEditorTreeItem() {}



SceneEditorTreeItem* SceneEditorTreeItem::parent() const {
    return _parent;
}

QList<SceneEditorTreeItem*> SceneEditorTreeItem::children() const {
    return _childItems;
}

SceneEditorTreeItem* SceneEditorTreeItem::child(QString longVarName) {
    if(longVarName.isEmpty()) return NULL;
    if(longVarName == _varName) return this;

    QStringList components = longVarName.split(QString("."), QString::SkipEmptyParts);
    if(_parent) {
        if(components[0] != _varName) return NULL;
        components.removeAt(0);
    }

    SceneEditorTreeItem* childItem = NULL;

    foreach(SceneEditorTreeItem* seti, _childItems) {
        childItem = seti->child(components.join("."));
        if(childItem) break;
    }
    return childItem;
}

SceneEditorTreeItem* SceneEditorTreeItem::child(int index)
{
    if(index < 0 || index >= _childItems.size()) return NULL;
    return _childItems[index];
}

void SceneEditorTreeItem::appendChild(SceneEditorTreeItem* child) {
    _childItems.append(child);
    child->setParent(this);
}

void SceneEditorTreeItem::setParent(SceneEditorTreeItem* parent) {
    _parent = parent;
    if(_parent)
        if(!(_parent->children().contains(this)))
            _parent->appendChild(this);

    _longVarName.clear();
    if(_parent) _longVarName.append(_parent->longVarName());
    if(!(_longVarName.isEmpty())) _longVarName.append(".");
    _longVarName.append(_varName);
}



int SceneEditorTreeItem::totalChildCount() const {
    int n = 0;
    foreach(SceneEditorTreeItem* seti, _childItems) {
        n += seti->totalChildCount();
        n++;
    }
    return n;
}

int SceneEditorTreeItem::childCount() const {
    return _childItems.size();
}

int SceneEditorTreeItem::row() const {
    if(!_parent) return -1;
    SceneEditorTreeItem* seti = (SceneEditorTreeItem*)this;
    return _parent->children().indexOf(seti);
}



QString SceneEditorTreeItem::varName() const {
    return _varName;
}

QString SceneEditorTreeItem::longVarName() const {
    return _longVarName;
}

float* SceneEditorTreeItem::mappedValue() const {
    return _mappedValue;
}

float SceneEditorTreeItem::defaultValue() const {
    return _defaultValue;
}
