// External functions used
// rand
// memset

#include <strings.h> 
#include <stdlib.h>
#include <spritepack.h> 

extern struct sp_Rect *sp_ClipStruct; 
#asm 
LIB SPCClipStruct 
._sp_ClipStruct         defw SPCClipStruct 
#endasm 
extern uchar *sp_NullSprPtr;
#asm
LIB SPNullSprPtr
._sp_NullSprPtr         defw SPNullSprPtr
#endasm 

#pragma output STACKPTR=49152

#include "sprites.h"
#include "tables.h"

// Music routines
#define VT_PLAY			50193
#define VT_LOAD			50176
#define VT_STOP			50203
#define VT_EFFECT		50205

uchar music_playing=0;
uchar music_enabled=1;


struct ball
{
  struct sp_SS *sprite;	
  int posx,posy;	// Position in x and y; 1 bit sign; 8 bits _real_ position, 7 bits decimal part
  uchar player;		// To which player it belongs
  uchar ball_type;
  int dx,dy;
};

struct arrow
{
  struct sp_SS *sprite;	
  uint angle;    	
  uchar frame;
};

struct player		
{
  uchar frame_sprite;	
  uint time_to_go;	// time to launch the next ball
  uchar destroy_anim_pos; 
  uint score;		// Player score
};

char arrow_offsets[8]={0,96,0,0,0,-96,0,0};
int bomb_offsets[16]={0,0,96,0,0,0,96,0,0,0,96,0,0,0,-288,0};
char ball_offsets[16]={0,0,0,96,0,0,0,0,0,0,0,-96,0,0,0,0};
uchar cursor_moved[2]; // See if the player's cursor changed


#define BALL_OFFSETX_1_P1_2PLAYERS 	8
#define BALL_OFFSETX_2_P1_2PLAYERS 	16
#define BALL_OFFSETX_1_P1_1PLAYER 	72
#define BALL_OFFSETX_2_P1_1PLAYER 	80
#define BALL_OFFSET_Y 			8
#define INITPOS_PLAYER1_1_1PLAYER	120
#define INITPOS_PLAYER1_2_1PLAYER 	160
#define INITPOS_PLAYER1_1_2PLAYERS	56
#define INITPOS_PLAYER1_2_2PLAYERS 	160
#define WAIT_TIME			250 	// If we do not do anything in 5 seconds, the ball will be auto-launched

uchar BALL_OFFSET_X_PLAYER1[2]={8,  16};
uchar BALL_OFFSET_X_PLAYER2[2]={136,144};
uchar INITPOS_PLAYER1[2]={56,160};
uchar INITPOS_PLAYER2[2]={184,160};
uchar NEXTBALL_POSX[2] = {12,29};  // Where to place the next ball icon
uchar PLAYER_SPRPOS[2] = {3,26};
uchar NEXTBALL_POSY = 21;



// Each player's grid is defined as:
// Row 0: B1 B2 B3 B4 B5 B6 B7      <BLANK>
// Row 1: B1 B2 B3 B4 B5 B6 <BLANK> <BLANK>
// We are wasting some bytes, but it will be worth in the end

uchar grid_player1[80];
uchar grid_player2[80];

struct ball *current_ball[2];
uchar next_ball[2]; // Next ball for each player (just the ball type)
struct arrow Arrows[2];
struct player Players[2];

// Used by the SetNullSpr function

uchar counter_till_nullspr;
uchar new_color;

uchar ball_colors[9]={ TRANSPARENT, 
			PAPER_BLACK | INK_MAGENTA ,
			PAPER_BLACK | INK_RED ,
			PAPER_BLACK | INK_GREEN ,
			PAPER_BLACK | INK_CYAN ,
			PAPER_BLACK | INK_YELLOW ,
			PAPER_BLACK | INK_WHITE,
			PAPER_BLACK | INK_BLUE,
			PAPER_BLACK | INK_BLUE
		      };

struct sp_Rect cliprect[2];  // 2 clip rectangles: 1 for each player

// Sprites
struct sp_SS *sprite_ball[2];
uchar ballspr_frame[2];
struct sp_SS *sprite_skull[2];
struct sp_SS *sprite_bomb[2];
uchar bombspr_frame[2];

// Keys defined for each player. 
struct sp_UDK keys[2];
uint   key_music;
// Joystick/key reading functions
void *joyfunc[2]; 

// Player states
#define PL_WAITING 	0
#define PL_BALL_MOVING	1
#define PL_LINES	2
#define PL_FIND_HANGING 3
#define PL_FALLING 	4
#define PL_DESTROY_ANIM 5
#define PL_BALLS_COMING_UP 6
#define PL_WAITING_0	7	// We will go to this state, then decide the next ball, then to PL_WAITING

uchar state_player[2];
uchar num_players;
uchar playing;
uchar winner;         // Winning player!
uchar player1_level;  // Current level for 1 player games
uchar player1_valid_balltypes; // Available ball types for 1 player matches
uchar player1_left_to_trigger; // how many balls we can launch until we get more balls added
uchar PL1_BALLS_TO_TRIGGER;
uchar player1_continues=1;	// we will only allow one continue per game
uchar continuing;		// To allow continues
uchar cheat_mode=0;		// well... you know what it is :)
uchar cheat_allowed=0;

uchar addballs_trigger;	// Bit 0: we'll add balls to player1's grid, Bit 1: we'll do it for player 2
uchar addballs_count[2]; // Should be somewhere between 1 and 6
uchar cannon_pos[11]={12,17,22,27,32,39,44,49,54,59,64};
uchar ball_touched;	// only usefull for the skull ball

// Some variables we prefer to have global instead of local (speed and code size)
char modifier;  

#define PLAYER2SCR 49152
#define MAINMENU 49152+2488

uint PLAYER1SCR[5]={49152+2488+4619+3986,49152+2488+4619,49152,49152+4150,49152+4150+3531};
#define continue 49152+4150+3531+4133

// Auxiliary functions location

#define BallLineLength_ASM	49152
#define MarkLines_ASM		49394
#define FindHangingBalls_ASM	49629
#define myitoa_ASM		49869


/* Create memory allocator for sprite routines */

void *my_malloc(uint bytes)
{
   	return sp_BlockAlloc(0);
}

void *u_malloc = my_malloc;
void *u_free = sp_FreeBlock; 

uchar current_bank=0;
void SetRAMBank(uchar bank)
{
	current_bank=bank;
#asm	
	ld  A, (_current_bank)
	ld  B, A
	ld  A, (5B5Ch)
	and F8h
	or  B
	ld  BC, 7FFDh
	ld  (5B5Ch), A
	out (C), A
#endasm	
}

uint COMPRESSED_SCREEN_START;
#define UNCRUNCH_START 24576

void UncompressToScreen(uint source)
{
	COMPRESSED_SCREEN_START=source;
	
	memset(22528,0,768);
#asm
		ld hl, (_COMPRESSED_SCREEN_START)
		ld de, 16384		
		call UNCRUNCH_START
#endasm		
}

char *BIOS_string;
uint BIOS_xy;
uint BIOS_color;

void BIOS_printf(char *string, uint xy, uint color)
{
	BIOS_string = string;
	BIOS_xy=xy;
	BIOS_color=color;
#asm
	di
#endasm
   	SetRAMBank(1);
#asm
	
	LD BC,(_BIOS_xy)
	LD DE,(_BIOS_color)
	LD HL, (_BIOS_string)
	CALL 52215
#endasm
	SetRAMBank(0);
#asm
	ei
#endasm
}
// Alternate font, alternate place
void BIOS_printf2(char *string, uint xy, uint color)
{
	BIOS_string = string;
	BIOS_xy=xy;
	BIOS_color=color;
#asm
	LD BC,(_BIOS_xy)
	LD DE,(_BIOS_color)
	LD HL, (_BIOS_string)
	CALL 50008
#endasm
}

uchar bank;
	
void PlayEffect(uchar effect)
{
	bank = current_bank;	
	#asm
	di
	#endasm	
	SetRAMBank(0);
	new_color = effect;  // Do not want to use another variable :)
	#asm
		ld a,(_new_color)
		call VT_EFFECT
	#endasm
	SetRAMBank(bank);		
	#asm
	ei
	#endasm	
}

void LoadMusic(uchar song)
{
	if(song==0)		
	{
		new_color=(player1_level/3)&1;
	}
	else new_color=song;
	
	#asm
	di
	#endasm	
	bank = current_bank;
	SetRAMBank(0);
	#asm
		ld A, (_new_color)
		call VT_LOAD		
		ld hl, 50208		; setup byte
		ld A, (_new_color)
		cp 5		
		jr nz, music_with_loop	; for song 5 (game over) we set no loop
		set 0, (hl)		; set bit 0 to 1 (no loop!)	
		jr continue_loadmusic
	.music_with_loop
		res 0, (hl)		; set bit 0 to 0 (loop!)
	.continue_loadmusic
	#endasm		
	
	music_playing=1;
	SetRAMBank(bank);		
	#asm
	ei
	#endasm		
}

void StopMusic(void)
{
	bank = current_bank;		
	#asm
	di
	#endasm	
	SetRAMBank(0);
	#asm
		call VT_STOP
	#endasm		
	SetRAMBank(bank);		
	#asm
	ei
	#endasm	
}


void printGameArea(void)
{
#asm
	di
#endasm	
	
   if(num_players==1)  
   {
   	if(player1_level < 12) SetRAMBank(3);
   	 else SetRAMBank(4);
   	 #asm
   	 ei
   	 #endasm
   	UncompressToScreen(PLAYER1SCR[player1_level / 6]);
   }
   else
   {
   	SetRAMBank(3);
   	#asm
   	ei
   	#endasm
   	UncompressToScreen(PLAYER2SCR);
   }
#asm
   di
#endasm
   SetRAMBank(0);
#asm
	ei
#endasm	
  sp_ClearRect(&(cliprect[0]), INK_WHITE | PAPER_BLACK, ' ', sp_CR_ALL);    
  sp_Invalidate(&(cliprect[0]), sp_ClipStruct);
  if (num_players == 2)
  {
  	sp_ClearRect(&(cliprect[1]), INK_WHITE | PAPER_BLACK, ' ', sp_CR_ALL);  
  	sp_Invalidate(&(cliprect[1]), sp_ClipStruct);
  } 
}

void SetNullSpr(struct sp_CS *cs)
{   
   cs->colour = new_color;
   if (counter_till_nullspr == 0) 
   {
   	cs->graphic = sp_NullSprPtr;   	
   }
   else 
   {
   	cs->colour = new_color;
   	counter_till_nullspr--;   
   }	
   return;
} 

// The ball selection algorithm is:
//
// "Normal" balls have a chance of 42 out of 256 
// "Special" balls (skull & bomb) have a chance of 2 out of 256
//
// For a 2 player game, we just return the new ball 
//
// For a 1 player game, we only return the ball if it has the same color
// as an existing ball in the grid, or it is a special ball.
// This makes the chance of getting a special ball higher when there are less
// colors available, but well...In the worst case it is 4 againg 42, so 1 special
// ball every 10 normal balls. It should be acceptable, but we can still change it


uchar value;
uchar special_chance;
uchar NextBall(void)
{
	if(cheat_mode) return 7; 
	if (num_players == 1 & player1_level > 15) special_chance=40;
	 else special_chance=42;
		
	value = ((rand()&0xff) / special_chance) + 1;
	if(value > 6 ) value = (rand() & 1)+7;

	
	//value=(rand()%8)+1;
	if(num_players==2) return value;
	else
	{
	 while ((((1<<value) & player1_valid_balltypes)==0) && (value < 7)) // just for testing, needs to be adjusted
	 {
		value = ((rand()&0xff) / special_chance)+1;
		if(value > 6) value = (rand() & 1)+7;
	 }	 
	}
	return value;		
}

void LaunchBall(struct ball *b)
{	    
   	counter_till_nullspr = 6;   	   

   	switch (b->ball_type)
   	{
   		case 7: // skull
   			b->sprite = sprite_skull[b->player]; //Rainbow effect will come later
   			break;
   		case 8: // bomb
   			b->sprite = sprite_bomb[b->player];	
   			break;
   		default: // normal ball	
   			b->sprite = sprite_ball[b->player];   			
   			break;   		
   	}	
   	new_color = ball_colors[b->ball_type];   	   	
   	
	sp_IterateSprChar(b->sprite, SetNullSpr);  

   	b->dx=(int)(costable[Arrows[b->player].angle >> 1])*2;		// Did not have *2 before optimizing (WARNING)
   	b->dy=-((int)(costable[(Arrows[b->player].angle >> 1)+18]))*2;	// Did not have *2 before optimizing (WARNING)
   	
   	// Set the arrow color to the new ball's color
	new_color = ball_colors[next_ball[b->player]];
   		
    	counter_till_nullspr = 6;   	   	
   	sp_IterateSprChar(Arrows[b->player].sprite, SetNullSpr);      	   	
   	//sp_Invalidate(&(cliprect[b->player]), sp_ClipStruct);
      	
   	if((num_players==1)&&(player1_level > 4)) 	// Decrement the left_to_trigger counter, trigger the addballs function if required
   	{
   		player1_left_to_trigger--;
		if(player1_left_to_trigger == 0)
		{
	  		addballs_trigger |= 1;	  		
	  		player1_left_to_trigger = PL1_BALLS_TO_TRIGGER;
		}
	}
	PlayEffect(0);
}



// Ball surrounding (x,y) = (x-1, y) , (x+1, y), (x-1+(y&1), y-1), (x+(y&1), y-1), (x-1+(y&1), y+1), (x+(y&1), y+1)
char init_diffx[6] = {-1,1,-1,0,-1,0};
char diffy[6]=  {0,0,-1,-1,1,1};


/* It could even be generic for all functions, since they all have the same parameters */	

char parameter1,parameter2;
uint parameter3;


uchar BallLineLength(char gridx, char gridy, uchar *grid)
{
	parameter1 = gridx;
	parameter2 = gridy;
	parameter3 = (uint)grid;
	#asm
		ld a, (_parameter1)
		ld c,a
		ld a, (_parameter2)
		ld e,a
		ld hl, (_parameter3)
		call BallLineLength_ASM
		ld (_parameter1), a
	#endasm
	return parameter1;
	
}

void MarkLines(char gridx, char gridy, uchar *grid)
{
	parameter1 = gridx;
	parameter2 = gridy;
	parameter3 = (uint)grid;
	#asm
		ld a, (_parameter1)
		ld c,a
		ld a, (_parameter2)
		ld e,a
		ld hl, (_parameter3)
		call MarkLines_ASM
	#endasm	
}

void FindHangingBalls(char gridx, char gridy, uchar *grid)
{
	parameter1 = gridx;
	parameter2 = gridy;
	parameter3 = (uint)grid;
	#asm
		ld a, (_parameter1)
		ld c,a
		ld a, (_parameter2)
		ld e,a
		ld hl, (_parameter3)
		call FindHangingBalls_ASM
	#endasm	
}

uchar bomb_i;
char bomb_x,bomb_y;
uchar bomb_kk;
char bombdiffx[6] = {-1,1,0,0,0,0};
void ExplodeBomb(char gridx, char gridy, uchar *grid)
{  	
  modifier = gridy&1;
  for(bomb_i=2;bomb_i<6;bomb_i++) 
    bombdiffx[bomb_i] = init_diffx[bomb_i]+modifier;

  
  // We have to search all 6 directions  
  
  for(bomb_i=0;bomb_i<6;bomb_i++)
  {
  	// calc the new gridx / gridy
  	bomb_x = gridx+bombdiffx[bomb_i];
  	bomb_y = gridy+diffy[bomb_i];
  	
  	bomb_kk=bomb_x | (bomb_y<<3);
  	
	// If we are at the border of the game area, continue
  	if ( (bomb_x< 0) || (bomb_y<0) || (bomb_x > 7) || (bomb_y> 8))
  	{}
         else if (grid[bomb_kk]) grid[bomb_kk]|=0x80;		
  }  	  	
}



// Returns 0: no lines found
// Returns 1: lines found

uchar FindLines(char gridx, char gridy, uchar *grid)
{
 if (BallLineLength(gridx,gridy,grid) > 2)
 {
  	MarkLines(gridx,gridy,grid);  	
  	return 1;
 }  	 
 return 0;
}

// Posible values for draw_animation:
// 0: draw blank tile (erase)
// 1: draw solid ball
// 2,3,4,5: draw destroy animations


uchar posx,posy;
void DrawTileBall(uchar player, uchar pos, uchar draw_animation, uchar color)
{
		
	if(player == 0) modifier = cliprect[0].col_coord;
	 else modifier=cliprect[1].col_coord;
	 
	posx = (pos & 0x7) << 1;
 	posy = (pos >> 3);
 	modifier += posy & 1;
 	posy <<= 1;
	
	if(draw_animation==0) // erase
	{
		sp_PrintAtInv(posy + 1, posx + modifier, TRANSPARENT, ' ');
		//sp_PrintAtInv(posy + 1, posx + 1 + modifier, TRANSPARENT, ' ');
		sp_PrintAtInv(posy + 2, posx + modifier, TRANSPARENT, ' ');
		//sp_PrintAtInv(posy + 2, posx + 1 + modifier, TRANSPARENT, ' ');			
	}
	else if(draw_animation == 1) //draw
	{
		sp_PrintAtInv(posy + 1, posx + modifier, color, 0);
		sp_PrintAtInv(posy + 2, posx + modifier, color, 2);

	}
	else // animation
	{
		sp_PrintAtInv(posy + 1, posx + modifier, color,     12 + ((draw_animation - 2) << 2));
		sp_PrintAtInv(posy + 1, posx + 1 + modifier, color, 13 + ((draw_animation - 2) << 2));
		sp_PrintAtInv(posy + 2, posx + modifier, color,     14 + ((draw_animation - 2) << 2));
		sp_PrintAtInv(posy + 2, posx + 1 + modifier, color, 15 + ((draw_animation - 2) << 2));				
	}	
	
	if(((pos & 7) < 6) || !((pos >>3) & 1))
	{
		if(draw_animation == 0) // erase
		{
			sp_PrintAtInv(posy + 1, posx + 1 + modifier, TRANSPARENT, ' ');			
			sp_PrintAtInv(posy + 2, posx + 1 + modifier, TRANSPARENT, ' ');			
		}
		else if(draw_animation == 1) //draw
		{			
			sp_PrintAtInv(posy + 1, posx + 1 + modifier, color, 1);
			sp_PrintAtInv(posy + 2, posx + 1 + modifier, color, 3);
		}		
	}
}

uchar kknb;
uchar kkposx;
void DrawNextBall(uchar player)
{
	kknb = next_ball[player];
	kkposx=NEXTBALL_POSX[player];
	
	if(kknb < 7 ) modifier=0;
	 else modifier = (kknb - 6) << 2;
	 
	sp_PrintAtInv(NEXTBALL_POSY, kkposx, ball_colors[kknb], modifier); 
	sp_PrintAtInv(NEXTBALL_POSY, kkposx + 1, ball_colors[kknb], modifier + 1); 
	sp_PrintAtInv(NEXTBALL_POSY + 1, kkposx, ball_colors[kknb], modifier + 2); 
	sp_PrintAtInv(NEXTBALL_POSY + 1, kkposx + 1, ball_colors[kknb], modifier + 3); 
}

uchar x,y;
uchar i_cannon,j_cannon;

void DrawTileCannon(uchar player, uchar frame)
{	
	uchar frameinit= (frame*12)+105;
	
	if(player == 0)
	{
		x= (INITPOS_PLAYER1[0] >> 3)-1;
		y= (INITPOS_PLAYER1[1] >> 3)-1;
	}
	else
	{
		x= (INITPOS_PLAYER2[0] >> 3)-1;
		y= (INITPOS_PLAYER2[1] >> 3)-1;		
	}
	for(j_cannon=0;j_cannon<3;j_cannon++)
	 for(i_cannon=0;i_cannon<4;i_cannon++)
	 {
	   sp_PrintAtInv(y+j_cannon,x+i_cannon,PAPER_BLACK | INK_BLUE , frameinit++);	   
	 }				
}

uchar k;
void CleanupGrid(uchar player, uchar *grid)
{
 for(k=0;k<80;k++)
 {
 	if(grid[k] & 0x80)
 	{
 		grid[k]=0;
 		DrawTileBall(player, k, 0, 0); 	
 	}	
 }	
}
// This function is used in 1p and 2p matches:
// 1p: after a certain period of time, some balls will be added, to increase difficulty
// 2p: when a player makes a line and leaves some hanging balls, those hanging balls will be added to the
//     other player's grid (with a maximum of 6)

uchar xx;
uchar *addb_grid;
void AddBalls(uchar player, uchar count)
{	
	if(player==0) addb_grid=grid_player1;
	 else addb_grid=grid_player2;

		for(;count;count--)
		{
			xx=(rand()%6)+64;
			if(addb_grid[xx]==0)  // we were lucky, this is a free position
			{
				addb_grid[xx]=((rand()%6)+1) | 0x80;  //marked as coming up								 	
				DrawTileBall(player,xx,1,ball_colors[addb_grid[xx]&0x7f]);
			}
		}	
}

// This function will be called whenever a player goes to the PL_WAITING_0 state, just before going to PL_WAITING. 
// If there are any pending balls to be added, we will trigger it
void CheckAddBalls(uchar player)
{
	if(addballs_trigger & (1<<player)) // We must add balls
	{
		AddBalls(player, addballs_count[player]);
		addballs_trigger ^= (1<<player); // reset this field
		state_player[player]=PL_BALLS_COMING_UP; // set the new state
	}	

}

uchar skull_i;
void StoreBall(char gridx, char gridy, uchar ball_type,uchar player)
{	
	struct ball *b;
	uchar *grid;
	
	PlayEffect(1);	// Sound effect
	
	if(gridy > 8)  // Game over for this player
	{
		playing=0;
		winner= player ^ 1; // The other player wins
		Players[winner].score ++;	// The other player has one more point
		if(num_players==2)
		{
		LoadMusic(3);	// level end jingle	
		#asm	
		ld a, 250
.sleep_halt6
		halt
		dec a
		jp nz,sleep_halt6
		#endasm
		StopMusic();
		}
		return;	
	}
		
	if(player == 0) grid=grid_player1;
	  else grid=grid_player2;
	
	switch(ball_type)
	{
		case 7: //skull -> special FindLines
		  	if(ball_touched)
  			{
  				for(skull_i=0;skull_i<80;skull_i++) // Mark all balls with the same color
  				{
  					if(grid[skull_i]==ball_touched) grid[skull_i]  |= 0x80;
  				}	
  				state_player[player] = PL_LINES;
  			}
			else // Nothing found, opportunity lost	
			{
				state_player[player] = PL_WAITING_0;	
				CheckAddBalls(player);
			}			
			break;
		case 8: // bomb -> special state while exploding bombs...
			ExplodeBomb(gridx,gridy,grid);
			state_player[player] = PL_LINES;	
			break;
		default: // normal ball
			grid[gridx | (gridy<<3)]=ball_type;
			if (num_players==1) Players[0].score ++;
			DrawTileBall(player,gridx | (gridy<<3),1,ball_colors[ball_type]);
			if (FindLines(gridx,gridy,grid))
				state_player[player] = PL_LINES;
			else 
			{
				state_player[player] = PL_WAITING_0;	
				CheckAddBalls(player);
			}			
			break;		
	}	
	// Send the ball back to the launch position
	b=current_ball[player];
	if(player == 0)
	{
	     b->posx = INITPOS_PLAYER1[0]; b->posx <<= 7; 
   	     b->posy = INITPOS_PLAYER1[1]; b->posy <<= 7; 
   	}
   	else
   	{
	     b->posx = INITPOS_PLAYER2[0]; b->posx <<= 7; 
   	     b->posy = INITPOS_PLAYER2[1]; b->posy <<= 7;    		   	     
   	}
    	sp_MoveSprAbs(b->sprite,&(cliprect[player]),0, 
    		b->posy >> 10,  
    		b->posx >> 10, 
    		(b->posx >> 7) & 0x7, 
    		(b->posy >> 7) & 0x7);   
    		
    	// Assign the new ball type 
    	b->ball_type = next_ball[player];		   	
   	Players[player].time_to_go = WAIT_TIME;
}


#asm
defw 0                        /* two free bytes before start of a hook */
#endasm

void timerISR(void)
{  
  if(music_playing)
  {
  	bank = current_bank;
  	SetRAMBank(0);  
  	#asm
		call VT_PLAY
  	#endasm
  	SetRAMBank(bank);  	
  }

  if(playing)
  {
  	if(state_player[0] == PL_WAITING)
  	{
  		Players[0].time_to_go--;
  		if(Players[0].time_to_go == 0) 
  		{
  			LaunchBall(current_ball[0]);
			state_player[0] = PL_BALL_MOVING;	
	  	}
	}
  	if(num_players==2 && state_player[1] == PL_WAITING)
  	{
  		Players[1].time_to_go--;
  		if(Players[1].time_to_go == 0) 
  		{
  			LaunchBall(current_ball[1]);
			state_player[1] = PL_BALL_MOVING;	
	  	}  	
  	}
 }
}

void intro(void)
{
   SetRAMBank(1);
#asm
	call 53485
#endasm		
   SetRAMBank(0);	
}

void menu(void)
{
  uchar test;
  uchar defined_keys[9];
#asm
   di
#endasm	
   SetRAMBank(3);
#asm
   ei
#endasm	
   UncompressToScreen(MAINMENU);
#asm
   di
#endasm	
   SetRAMBank(1);
#asm
   ei
#endasm	

#asm
	ld a, (_player1_continues)
   	call 51552
#endasm	

   test = *(uchar*)(65526);
   defined_keys[0]=*(uchar*)(65527);
   defined_keys[1]=*(uchar*)(65528);
   defined_keys[2]=*(uchar*)(65529);
   defined_keys[3]=*(uchar*)(65530);
   defined_keys[4]=*(uchar*)(65531);
   defined_keys[5]=*(uchar*)(65532);
   defined_keys[6]=*(uchar*)(65533);
   defined_keys[7]=*(uchar*)(65534);
   defined_keys[8]=*(uchar*)(65535);
#asm
	di
#endasm	
   SetRAMBank(0);
#asm
	ei
#endasm	

   num_players = (test & 1)+1;

   switch(test & 0x06) //Player 1 controls
   {
   	case 0: // 00x
   		joyfunc[0] = sp_JoyKempston;
   		break;
   	case 2: // 01x
   		joyfunc[0] = sp_JoySinclair1;
   		break;
   	case 4: // 10x
   		joyfunc[0] = sp_JoySinclair2;
   		break;
   	case 6: //11x
   		joyfunc[0] = sp_JoyKeyboard; 
   		keys[0].left  = sp_LookupKey(defined_keys[0]);
   		keys[0].right = sp_LookupKey(defined_keys[1]);
   		keys[0].up  = sp_LookupKey(defined_keys[2]);
   		keys[0].fire  = sp_LookupKey(defined_keys[3]);
   		break;   		
   }
   switch(test & 0x18) //Player 2 controls
   {
   	case 0: // 00xxx
   		joyfunc[1] = sp_JoyKempston;
   		break;
   	case 0x08: // 01xxx
   		joyfunc[1] = sp_JoySinclair1;
   		break;
   	case 0x10: // 10xxx
   		joyfunc[1] = sp_JoySinclair2;
   		break;
   	case 0x18: //11xxx
   		joyfunc[1] = sp_JoyKeyboard; 
   		keys[1].left  = sp_LookupKey(defined_keys[4]);
   		keys[1].right = sp_LookupKey(defined_keys[5]);
   		keys[1].up  = sp_LookupKey(defined_keys[6]);
   		keys[1].fire  = sp_LookupKey(defined_keys[7]); 
   		break;   		
   }
   key_music = sp_LookupKey(defined_keys[8]); 

   if(test & 0x80) cheat_allowed = 1;


   if(num_players==1)
   {

	BALL_OFFSET_X_PLAYER1[0]=BALL_OFFSETX_1_P1_1PLAYER;
	BALL_OFFSET_X_PLAYER1[1]=BALL_OFFSETX_2_P1_1PLAYER;
	INITPOS_PLAYER1[0]=INITPOS_PLAYER1_1_1PLAYER;
	INITPOS_PLAYER1[1]=INITPOS_PLAYER1_2_1PLAYER;
	NEXTBALL_POSX[0]= 7; 
	PLAYER_SPRPOS[0]= 11; 
	NEXTBALL_POSY=21;
	
	cliprect[0].row_coord = 1; 
   	cliprect[0].col_coord = 9; 
   	cliprect[0].height = 18; 
   	cliprect[0].width  = 14;
   }
   else // num_players==2
   {
   	cliprect[0].row_coord = 1; 
   	cliprect[0].col_coord = 1; 
   	cliprect[0].height = 18; 
   	cliprect[0].width  = 14;
      
   	cliprect[1].row_coord = 1; 
   	cliprect[1].col_coord = 17; 
   	cliprect[1].height = 18; 
   	cliprect[1].width  = 14;   	

	BALL_OFFSET_X_PLAYER1[0]=BALL_OFFSETX_1_P1_2PLAYERS;
	BALL_OFFSET_X_PLAYER1[1]=BALL_OFFSETX_2_P1_2PLAYERS;
	INITPOS_PLAYER1[0]=INITPOS_PLAYER1_1_2PLAYERS;
	INITPOS_PLAYER1[1]=INITPOS_PLAYER1_2_2PLAYERS;
	NEXTBALL_POSX[0]=1; // FIXME!! NEED TO CHECK THIS!!!
	PLAYER_SPRPOS[0] =3;   	
	NEXTBALL_POSY=22;
   }
}

void happy_ending(void)
{
	LoadMusic(2);	// Load the ending music
  SetRAMBank(6);
#asm
	call 49152
#endasm	
  SetRAMBank(0);
}

uchar m;
char *level_string="Level   \xff";
char *level_numbers="1 2 3 4 5 6 7 8 9 101112131415161718192021222324252627282930";

void SetOnePlayerGrid(uchar force_redraw)
{
	//if((!(player1_level % 6)) && (player1_level > 0))
	if(!(player1_level % 6) || force_redraw)
	{
		printGameArea();
		Arrows[0].angle=36;	// Place it vertical
		cursor_moved[0]=1;
	}
#asm
    di
#endasm		
   SetRAMBank(1);
   memcpy(grid_player1,(uchar*)(49152+80*player1_level),80);
   SetRAMBank(0);
#asm
    ei
#endasm	
	
	player1_valid_balltypes=0;
			
	for(m=0;m<80;m++)
	{
	 if(grid_player1[m]) 
	 {
	 	DrawTileBall(0,m,1,ball_colors[grid_player1[m]]);		
	 	player1_valid_balltypes |= 1 << grid_player1[m];  // Set the corresponding bit
 	 } 	  	 
 	}
 	if(player1_valid_balltypes==0) player1_valid_balltypes = 126; 	
	sp_UpdateNow(); 	
	
	//Print level number
	level_string[6]=level_numbers[player1_level<<1];
	level_string[7]=level_numbers[(player1_level<<1)+1];
	
	BIOS_printf(level_string,0x0c0a,0x0600);

	playing=0;	
	LoadMusic(4);		// level start jingle
	
	#asm	
	ld a, 90
.sleep_halt1
	halt
	dec a
	jp nz,sleep_halt1	
	#endasm
	
	StopMusic();
	if(music_enabled) LoadMusic(0);	// main game music

	sp_Invalidate(&(cliprect[0]), sp_ClipStruct);
	sp_UpdateNow();	
	playing=1;
}

char *p1_string="Player one\xff";
char *p2_string="Player two\xff";
char *scores="012";
char *textscore1="Wins  \xff";
char *textscore2="  wins\xff";

void DrawTwoPlayerScore(void)
{
	BIOS_printf(p1_string,0x0308,0x0600);
	BIOS_printf(p2_string,0x1308,0x0600);

	textscore1[5]=scores[Players[0].score];
	textscore2[0]=scores[Players[1].score];

	BIOS_printf(textscore1,0x050a,0x0600);
	BIOS_printf(textscore2,0x150a,0x0600);	
	playing=0;
	LoadMusic(4); // level start jingle
	
	#asm	
	ld a, 90
.sleep_halt3
	halt
	dec a
	jp nz,sleep_halt3
	#endasm
	
	StopMusic();
	if(music_enabled) LoadMusic(0);

	sp_Invalidate(&(cliprect[0]), sp_ClipStruct);
	sp_Invalidate(&(cliprect[1]), sp_ClipStruct);
	sp_UpdateNow();	
	playing=1;
}

char *p1score="    0\xff";
uint draw_score;
void DrawOnePlayerScore(void)
{
	if (Players[0].score > 9999) Players[0].score = 9999; // avoid a possible bug if we go beyond 9999 points
	
	draw_score=Players[0].score;
#asm
	LD IX, (_p1score)
	LD HL, (_draw_score)
	CALL myitoa_ASM
#endasm
	//itoa(p1score,Players[0].score);
	BIOS_printf2(p1score,0x0f00,0x0400);
}

uchar n;
void CheckNextLevelPlayer1(void)
{
	uchar found;
	
	found=0;
	for(n=0;n<80;n++)
	 if(grid_player1[n]) found=1;

	if(!found)	// We should do something else, such as play some level-end music or whatever
	{
		sp_UpdateNow();
		LoadMusic(3);	// level end jingle
	
		#asm	
		ld a, 250
.sleep_halt2
		halt
		dec a
		jp nz,sleep_halt2	
		#endasm
		StopMusic();
	    
	    player1_level++;		    
	    if(player1_level == 30)
	    {
	    	playing=0;
	    	player1_continues=0;
	    	happy_ending();
	    } 
	    else
	    {			
	    	if(player1_level == 14)	// Get one continue if we got to level 15!
	    		player1_continues=1;	// Watch out, no ++, no more than 1 continue allowed!
	    		
   	    	SetOnePlayerGrid(0);	
   	    	addballs_trigger &= 2; // We reset the addball trigger
   	       
   	        // This could change, to make the number be different depending on the current level   	    
   	    	if (player1_level < 5) 
   	    	{
	   	    	PL1_BALLS_TO_TRIGGER=15;
   	    		addballs_count[0]=6;
   	    	}
   	    	else if(player1_level < 12)
   	    	{
	   	    	PL1_BALLS_TO_TRIGGER=7;
   	    		addballs_count[0]=2;	
   	    	}
	    	else if(player1_level < 20) 
	    	{
		    	PL1_BALLS_TO_TRIGGER=8;
	    		addballs_count[0]=3;	
	    	}
	    	else if(player1_level < 25)
	    	{
	    	 	PL1_BALLS_TO_TRIGGER=9;
	    	 	addballs_count[0]=4;	
	    	}
	    	else 
	    	{
		    	PL1_BALLS_TO_TRIGGER=10;
	    		addballs_count[0]=6;	
	    	}
	    	player1_left_to_trigger=PL1_BALLS_TO_TRIGGER;
	    }
	}
}

uchar i_player,j_player;

void DrawPlayer(uchar player, uchar frame)
{
	uchar frameinit=frame*9+36*player+33;

	for(j_player=0;j_player<3;j_player++)
	 for(i_player=0;i_player<3;i_player++)
	 {
	   sp_PrintAtInv(20+j_player,PLAYER_SPRPOS[player]+i_player,PAPER_BLACK| INK_YELLOW, frameinit++);	   
	 }			 	
}

char *gameover_string="Game Over\xff";
char *youwin_string="Player X wins\xff";
char *print_string;
/*uchar x_gameover;*/

uint x_gameover;

void GameOver(void)
{
   if(num_players==1)
   {
   	  x_gameover=0x0c0a;   	  
   	  print_string=gameover_string;
   }
   else 
   {
   	x_gameover=0x0A00;
   	if(winner==0) youwin_string[7]='1';
    		else youwin_string[7]='2';      
    	print_string=youwin_string;
   #asm
   	di
   #endasm
   	SetRAMBank(1);
   #asm
   	ei
   #endasm
   	UncompressToScreen(0xD6B0);
   #asm
   	di
   #endasm
   	SetRAMBank(0);
   #asm
   	ei
   #endasm   
   
   }
   if (player1_level < 30) 
   {   	
   	BIOS_printf(print_string,x_gameover,0x0600);   	
   	LoadMusic(5);	// level end jingle 
   }
     	sp_WaitForNoKey();	  
   	sp_WaitForKey();
	//StopMusic();
   	//if(music_enabled) 
   	LoadMusic(1);
}


void CalcNewPosition(struct ball *);

uchar i, j;
uchar hanging_balls;
int offset;
uchar tictac;  // A simple frame counter; I don't even care its initial value
main() 
{ 
   uint posx, posy; 
   uchar *grid;

// This is all generic initialization, so it must stay as it is now
      
   #asm 
   di 
   #endasm 
   sp_InitIM2(0xb1b1); 
   sp_CreateGenericISR(0xb1b1); 
   sp_RegisterHook(255, timerISR);
   #asm 
   ei 
   #endasm 
  
   // Initialization
   
   sp_Initialize(INK_WHITE | PAPER_BLACK, ' '); 
   sp_Border (BLACK );
   sp_AddMemory(0, 100, 14, 60200); // Adjust this as needed
    
   for(i=0;i<2;i++)
   {
    	// Define ball sprites
	sprite_ball[i]= sp_CreateSpr(sp_OR_SPRITE, 3, bola_1, 1, TRANSPARENT);  
    	sp_AddColSpr(sprite_ball[i], bola_2, TRANSPARENT); 
    	sp_AddColSpr(sprite_ball[i], bola_2, TRANSPARENT);
    	ballspr_frame[i]=0;
    	// Define current ball structure
    	current_ball[i]=(struct ball *)u_malloc(sizeof(struct ball));
    	current_ball[i]->sprite = sprite_ball[i];
    	current_ball[i]->player=i;
    	// Define skull sprites
    	sprite_skull[i]= sp_CreateSpr(sp_OR_SPRITE,3,skull_1,1,TRANSPARENT);
    	sp_AddColSpr(sprite_skull[i], skull_2, TRANSPARENT); 
    	sp_AddColSpr(sprite_skull[i], skull_2, TRANSPARENT);
    	// Define bomb sprites
    	sprite_bomb[i] = sp_CreateSpr(sp_OR_SPRITE,3,bomb_1,1,TRANSPARENT);
    	sp_AddColSpr(sprite_bomb[i], bomb_2, TRANSPARENT); 
    	sp_AddColSpr(sprite_bomb[i], bomb_2, TRANSPARENT);
    	bombspr_frame[i]=0;
   }   
   
     
   // Define tiles
   sp_TileArray(0,ball_tile_1);
   sp_TileArray(1,ball_tile_2);
   sp_TileArray(2,ball_tile_3);
   sp_TileArray(3,ball_tile_4);   
   sp_TileArray(4,skull_tile_1);
   sp_TileArray(5,skull_tile_2);
   sp_TileArray(6,skull_tile_3);
   sp_TileArray(7,skull_tile_4);
   sp_TileArray(8,bomb_tile_1);
   sp_TileArray(9,bomb_tile_2);
   sp_TileArray(10,bomb_tile_3);
   sp_TileArray(11,bomb_tile_4);   
      
   for(i=0;i<16;i++) sp_TileArray(12+i,ball_destroy_anim + (i<<3)); // Ball being destroyed, frames
   for(i=0;i<72;i++) sp_TileArray(33+i, pirate_f1 + ((uint)(i)<<3)); // Pirate & soldier, frames
   for(i=0;i<132;i++) sp_TileArray(105+i,cannon_f1 + (i<<3)); // Cannon, frames
  
   LoadMusic(1);        
   intro();

   while(1)
   {   	
   	menu();
   	Players[0].score = Players[1].score = 0;
	StopMusic();	   
      	player1_level=0;	
   	player1_left_to_trigger = 0;
   	continuing=1;
   	while(continuing)
   	{   	
	 do
	 {		 	
	  continuing=0;
   	  // All of this initialization must be done everytime we start a new game   
   	  memset(grid_player1,0,80);
   	  memset(grid_player2,0,80);   
   	  current_ball[0]->posx=INITPOS_PLAYER1[0]; current_ball[0]->posx<<=7;
     	  current_ball[0]->posy=INITPOS_PLAYER1[1]; current_ball[0]->posy<<=7;
   	  current_ball[1]->posx=INITPOS_PLAYER2[0]; current_ball[1]->posx<<=7;
   	  current_ball[1]->posy=INITPOS_PLAYER2[1]; current_ball[1]->posy<<=7;   // We still need to assign the new color FIXME!!!   
   	  state_player[0]=state_player[1]=PL_WAITING;
   	  Players[0].time_to_go = WAIT_TIME;
   	  Players[1].time_to_go = WAIT_TIME;
	  addballs_trigger=0;
	  addballs_count[0]=addballs_count[1]=0;
     	
    	  // Set pirate frame
   	  Players[0].frame_sprite=0;
   	  // Set soldier frame
   	  Players[1].frame_sprite=0;
                             
   	  // Draw main screen
   	  sp_ClearRect(sp_ClipStruct, INK_WHITE | PAPER_BLACK, ' ', sp_CR_ALL);    
  	  sp_Invalidate(sp_ClipStruct, sp_ClipStruct);  	
   	  sp_UpdateNow();    
   	  //printGameArea();    
   	  //sp_UpdateNow();
  
   	  Arrows[0].angle=36;
   	  Arrows[1].angle=36;    
   	  DrawTileCannon(0,5);
   	  DrawPlayer(0,Players[0].frame_sprite & 3);
     	
     	  //if (music_enabled) LoadMusic(1);
     	
     	
   	  if(num_players==2) 
   	  {
   	  	printGameArea();    
   		DrawTileCannon(1,5);   		
   		DrawPlayer(1,Players[1].frame_sprite & 3);
   		Players[1].frame_sprite++;
   		current_ball[1]->ball_type = NextBall();
   		next_ball[1]=NextBall();
   		DrawNextBall(1);
   		DrawTwoPlayerScore();
   	  }
   	  else  // One player game
   	  {
   		SetOnePlayerGrid(1);	// Force redraw!
   	  } 
   	  sp_UpdateNow();
   	  current_ball[0]->ball_type = NextBall();
   	  next_ball[0]=NextBall();
   	  DrawNextBall(0);

	  for(i=0;i<2;i++)
	  {
	  	Arrows[i].sprite=sp_CreateSpr(sp_MASK_SPRITE, 3, puntero_1, 1, TRANSPARENT);
   		sp_AddColSpr(Arrows[i].sprite, puntero_2, TRANSPARENT); 
   		sp_AddColSpr(Arrows[i].sprite, puntero_2, TRANSPARENT);
   		counter_till_nullspr = 6;   	   	   	
   		new_color = ball_colors[current_ball[i]->ball_type];
   		sp_IterateSprChar(Arrows[i].sprite, SetNullSpr); 
   		Arrows[i].frame=0;		   		
	  }

   	  sp_UpdateNow();
   	  winner=0;
   	  playing=1;
	
   	  // This is the game loop itself
   
     	  while(playing) {            
	  for(i=0;i<num_players;i++)
	  {
		if(i==0) grid=grid_player1;
			else grid=grid_player2;
		cursor_moved[i]=0;
		// Read joysticks and update arrow angles
    		posx = (joyfunc[i])(&keys[i]); 
    		if (!(posx & sp_LEFT))
    		{
    			if(Arrows[i].angle > 8) Arrows[i].angle--;  // TEST!!! should be > 4
    			cursor_moved[i]=1;
    		}
    		if (!(posx & sp_RIGHT))
    		{
    			if(Arrows[i].angle < 64) Arrows[i].angle++;    	// TEST!!! should be < 32
    			cursor_moved[i]=1;    			
    		}
    		if (!(posx & sp_UP))
    		{
    			Arrows[i].angle=36;  // Center cannon
    			cursor_moved[i]=1;
    		}
    		
    		// A slight hack
    		if(sp_KeyPressed(sp_LookupKey('y')) && (num_players == 1) && cheat_allowed)
    		{
    				sp_WaitForNoKey();
    				cheat_mode^=1;
    				sp_Border(cheat_mode);
    		}   
    		if(sp_KeyPressed(key_music))
    		{
    			sp_WaitForNoKey();
    			music_enabled ^= 1;
    			if(!music_enabled)
    				StopMusic();
    			else 
    				LoadMusic(0);	
    		} 		
    					
		switch(state_player[i])
		{
			case PL_WAITING_0:// Recalc the next ball type			
					next_ball[i]=NextBall();
					DrawNextBall(i);
					state_player[i]=PL_WAITING;
					if(num_players==1) DrawOnePlayerScore();
					break;
			case PL_WAITING:
					if(!(posx & sp_FIRE))
					{
						LaunchBall(current_ball[i]);
						state_player[i] = PL_BALL_MOVING;						
					}
					break;
			case PL_BALL_MOVING:
					ball_touched=0;	// no ball has been touched by the current ball					
					switch(current_ball[i]->ball_type)
					{
						case 7: // skull
							new_color=ball_colors[rand()%6+1];
							counter_till_nullspr = 6;
							sp_IterateSprChar(current_ball[i]->sprite, SetNullSpr);  
							offset = 0; // no animation
							break;
						case 8: // bomb (should add bomb animation here)
							offset = bomb_offsets[(bombspr_frame[i]) & 15];
							//offset=0;
							bombspr_frame[i]++;
							break;
						default: // animate ball
							offset = ball_offsets[(ballspr_frame[i]) & 15];
							ballspr_frame[i]++;
							break;
					}
					CalcNewPosition(current_ball[i]);
    					sp_MoveSprAbs(current_ball[i]->sprite,&(cliprect[i]), offset , 
    						      current_ball[i]->posy >> 10,  
    						      current_ball[i]->posx >> 10, 
    						      (current_ball[i]->posx >> 7) & 0x7, 
    						      (current_ball[i]->posy >> 7) & 0x7);                   
					break;
			case PL_LINES:  	
					PlayEffect(3);
					Players[i].destroy_anim_pos=0;		
					state_player[i] = PL_DESTROY_ANIM;	
					if(num_players==1)	// Add points
					{
						for(j=0;j<80;j++)
						{
					 		if(grid[j] & 0x80) Players[0].score++;
						}							
					}
					break;		
			case PL_DESTROY_ANIM:
					for(j=0;j<80;j++)
					{
					 if(grid[j] & 0x80)
					 {
					   DrawTileBall(i, j,2+(Players[i].destroy_anim_pos), ball_colors[grid[j] & 0x7f]);
					 }
					}	
					if((Players[i].destroy_anim_pos++) == 4)
					{
						CleanupGrid(i,grid);	
						state_player[i] = PL_FIND_HANGING;
					}
					break;								
			case PL_FIND_HANGING: // First, mark hanging balls
					for(j=0;j<7;j++) 
						if(grid[j]) FindHangingBalls(j,0,grid);
					hanging_balls=0;
					for(j=0;j<80;j++)
					{
						if (grid[j] && (!(grid[j]&0x80)))  // There were some balls left hanging
						{
							state_player[i] = PL_FALLING;					
							grid[j] |= 0x80; // Mark this ball as falling
							hanging_balls++;
						}
						else grid[j] &=0x7f; // Cleanup
					}
					if (num_players == 1) // Calc new player1_valid_balltypes value
					{
						player1_valid_balltypes=0;
						for(j=0;j<80;j++)
						 if ((grid[j]&0x7f) && ((grid[j]&0x80)==0)) player1_valid_balltypes |= 1 << (grid[j]&0x7f);
						if(player1_valid_balltypes==0) player1_valid_balltypes = 126;

						// Add 5 points (50 actually) for each ball hanging
						Players[0].score += 5*hanging_balls;
				        }
					else if(hanging_balls) // add the balls to the other player, we have 2 players
					{
						addballs_trigger |= (1<<(1-i));
						addballs_count[1-i] = (hanging_balls > 6 ? 6 : hanging_balls);
						
					}
					if(state_player[i] != PL_FALLING)
					{  // For one player matches, check if we have finished this level
					  if (num_players == 1)
					  {
						CheckNextLevelPlayer1();
				          }									
					  state_player[i] = PL_WAITING_0;
					  CheckAddBalls(i);
					}
					break;
			case PL_BALLS_COMING_UP:
					for(j=0;j<71;j++)
					{
						if (grid[j] & 0x80)
						{
							if((grid[j-8]) || j < 8) grid[j] &= 0x7f;
							else
							{							
								DrawTileBall(i,j,0,0);
								DrawTileBall(i,j-8,1,ball_colors[grid[j]&0x7f]);		
								grid[j-8]=grid[j];
								grid[j]=0;	
							}	
						}
					}
					// See if we are still coming up
					state_player[i] = PL_WAITING_0;
					for(j=0;j<80;j++)
					{
						if (grid[j]&0x80)  // There were some balls left hanging
							state_player[i] = PL_BALLS_COMING_UP;					
					}								
					break;
			case PL_FALLING:
					for(j=79;j;j--)
					{
						if (grid[j] & 0x80)
						{
							if((j<64) && (grid[j+8] ==0))
							{
								DrawTileBall(i,j,0,0);
								DrawTileBall(i,j+8,1,ball_colors[grid[j]&0x7f]);		
								grid[j+8]=grid[j];
								grid[j]=0;
							}
							else
							{
								grid[j]=0;
								DrawTileBall(i,j,0,0);
							}
								
						}
					}
					// See if we are still falling
					state_player[i] = PL_WAITING_0;
					for(j=0;j<80;j++)
					{
						if (grid[j]&0x80)  // There were some balls left hanging
							state_player[i] = PL_FALLING;					
					}		
					if ((num_players == 1) && (state_player[i] == PL_WAITING_0))
					{
					 CheckNextLevelPlayer1();
				        }				
				        if(state_player[i] == PL_WAITING_0) CheckAddBalls(i);
					break;
		}
	}

	if(playing)	// If we are still playing (i.e. checknextlevelplayer1 did not cause a game end)
	{
    		// Redraw pointer sprites and cannon
		for(i=0;i<num_players;i++)
		{
			if(i==0)
			{
				posx = INITPOS_PLAYER1[0]; 
				posy = INITPOS_PLAYER1[1]; 
			}
			else
			{
				posx = INITPOS_PLAYER2[0]; 
				posy = INITPOS_PLAYER2[1]; 
			}
	
			posx <<=7; 
			posy <<=7;
		
			posx += ((int)(costable[Arrows[i].angle >> 1])) * 32 ;
			posy -= ((int)(costable[(Arrows[i].angle >> 1)+18])) * 32;
	
	   		if(state_player[i]==PL_WAITING)
   			{
	 			Arrows[i].frame ++;	
				offset=arrow_offsets[Arrows[i].frame&7];
				if(current_ball[i]->ball_type > 6)
				{
				   	// Set the arrow color to a random one
					new_color = rand()%6 + 1;
    					counter_till_nullspr = 6;   	   	
   					sp_IterateSprChar(Arrows[i].sprite, SetNullSpr);   	
				}
			}
			else offset=0;
		
		 	sp_MoveSprAbs(Arrows[i].sprite,&(cliprect[i]),offset, 
				posy >> 10,  
				posx >> 10, 
				(posx >> 7) & 0x7, 
				(posy >> 7) & 0x7);   		 			 	
	 	
	 		if(cursor_moved[i])
	 		{    	   	 			
	   			if((tictac&1) == i)
   				{
   					j=0;
   					while(Arrows[i].angle > cannon_pos[j]) j++;
   					DrawTileCannon(i,j) ;
   					Players[i].frame_sprite++;
   					DrawPlayer(i,Players[i].frame_sprite & 3);
   				}
    	 		}	          	  
    		}
    		sp_UpdateNow();           
	}
    	tictac++;           
  	// Wait for the next interrupt
  	if(num_players==1)
  	{
	#asm
	halt
	#endasm 	
	}
      }       
      
      // Delete player and arrow sprites (this is the best way to avoid problems with the animation on the next match...)
      sp_MoveSprAbs(Arrows[0].sprite,sp_ClipStruct,0,40,40,0,0);
      sp_MoveSprAbs(Arrows[1].sprite,sp_ClipStruct,0,40,40,0,0);   	
      sp_DeleteSpr(Arrows[0].sprite);
      sp_DeleteSpr(Arrows[1].sprite);
    }while((num_players==2 && ((Players[0].score < 2) && (Players[1].score < 2))) || playing);
    GameOver();
    if( ((player1_continues > 0) && (num_players ==1)) || num_players == 2)
    {
    	#asm
    	di
    	#endasm
   	SetRAMBank(4);
   	#asm
    	ei
    	#endasm
   	UncompressToScreen(continue);
   	#asm
    	di
    	#endasm
   	SetRAMBank(0);
   	#asm
    	ei
    	#endasm

      	sp_WaitForNoKey();
      	sp_WaitForKey();
      	if(sp_KeyPressed(sp_LookupKey('y')))
      	{
      		continuing=1;
      		if(num_players == 1)
      			player1_continues--;	
      		else
      		  Players[0].score = Players[1].score = 0;
      	}
      }            
    }
    //LoadMusic(1);
   }
}

// Including the  CalcNewPosition function, since it is a huge ASM listing :)

#include "calcnew.h"