#include <time.h>
#include <math.h>

#define MAX_VERTEX 1800

// 3D dot structure
typedef struct {
	long x,y,z;
} Tdot3D;

// 3D object structure
typedef struct {
	word num_vertex;     //number of vertices
	int  pos_x, pos_y;   //position to center in the screen
	int  movz;           //distance in Z
	byte phiX;           //X rotation (0...256)
	byte phiY;           //Y rotation (0...256)
	byte phiZ;           //Z rotation (0...256)
	Tdot3D *points;      //The actual dots position
	word Xp[MAX_VERTEX],Yp[MAX_VERTEX]; //the 2D position in the screen
	word Light[MAX_VERTEX]; //The light for each dot
	word base_light_color; //The base color
	word light_steps;      //The number of colors for the light
} Tobj3D;

float Xc=0;
float Yc=0;
float Zc=150;

float sintab[256];
float costab[256];

#define Sinus(i)   sintab[i]
#define Cosinus(i) costab[i]

//#include "lamp.h"       //A lamp (from 3DStudio)
//#include "vase.h"       //A vase (from 3DStudio)
#include "face3d.h"       //A face (from 3DS)

#define NUM_STARS 1000
#define TUNNEL_RADIUS 30
Tdot3D far starfield_array[NUM_STARS]; //Starfield points
int starfield_direction=4; //Starfield speed and direction

//Object definitions
Tobj3D vase={NUM_VERTEX_FACE,160,100,0,0,0,0,(Tdot3D *)&face_points[0],{0},{0},{0},0,128};
Tobj3D starfield={NUM_STARS,160,100,0,0,0,0,(Tdot3D *)&starfield_array[0],{0},{0},{0},0,256};

//Sine and Cosine tables initialization
void init_3d(void) {
	word i;

	for(i=0;i<256;i++) {
		sintab[i]=sin(i*M_PI/128);
		costab[i]=cos(i*M_PI/128);
	}
}

//Starfield initialization (Tunnel Shape)
void init_starfield(void) {
	word i;
	word x;

	randomize();
	for(i=0;i<NUM_STARS;i++) {
		x=i*NUM_STARS/256;
		starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS;
		starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS;
		starfield_array[i].z=(rand()%250)-125;
	}
}

//Moves the starfield wrapping on Z position
void move_starfield_normal(void) {
	word i;

	for(i=0;i<NUM_STARS;i++) {
		starfield_array[i].z+=starfield_direction;
		if(starfield_array[i].z>125) starfield_array[i].z-=250;
		if(starfield_array[i].z<-125) starfield_array[i].z+=250;
	}
}

//Moves the starfield wrapping on Z position and moving the center when
//it wraps
void move_starfield_normal_movecenter(int cdx,int cdy) {
	word i;
	word x;

	for(i=0;i<NUM_STARS;i++) {
		x=i*NUM_STARS/256;
		starfield_array[i].z+=starfield_direction;
		if(starfield_array[i].z>125) {
			starfield_array[i].z-=250;
			starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS+cdx;
			starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS+cdy;
		}
		if(starfield_array[i].z<-125) {
			starfield_array[i].z+=250;
			starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS+cdx;
			starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS+cdy;
		}
	}
}

//Moves the starfield wrapping on Z position and moving the center when
//it wraps, and changing the tunnel radius (only when wrapping)
void move_starfield_movecenter_wobble(int cdx,int cdy,int wobble) {
	word i;
	word x;

	for(i=0;i<NUM_STARS;i++) {
		x=i*NUM_STARS/256;
		starfield_array[i].z+=starfield_direction;
		if(starfield_array[i].z>125) {
			starfield_array[i].z-=250;
			starfield_array[i].x=Sinus(x)*(TUNNEL_RADIUS+wobble)+cdx;
			starfield_array[i].y=Cosinus(x)*(TUNNEL_RADIUS+wobble)+cdy;
		}
		if(starfield_array[i].z<-125) {
			starfield_array[i].z+=250;
			starfield_array[i].x=Sinus(x)*(TUNNEL_RADIUS+wobble)+cdx;
			starfield_array[i].y=Cosinus(x)*(TUNNEL_RADIUS+wobble)+cdy;
		}
	}
}

//Moves the starfield wrapping on Z position and moving the center when
//it wraps, and changing the stars position each 2 dots
void move_starfield_double_movecenter(int cdx,int cdy) {
	word i;
	word x;

	for(i=0;i<NUM_STARS;i++) {
		x=i*NUM_STARS/256;
		starfield_array[i].z+=starfield_direction;
		if(starfield_array[i].z>125) {
			starfield_array[i].z-=250;
			if(!(i%2)) {
				starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS+cdx;
				starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS+cdy;
			} else {
				starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS-cdx;
				starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS-cdy;
			}
		}
		if(starfield_array[i].z<-125) {
			starfield_array[i].z+=250;
			if(!(i%2)) {
				starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS+cdx;
				starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS+cdy;
			} else {
				starfield_array[i].x=Sinus(x)*TUNNEL_RADIUS-cdx;
				starfield_array[i].y=Cosinus(x)*TUNNEL_RADIUS-cdy;
			}
		}
	}
}

//Moves the starfield wrapping on Z position and moving the center when
//it wraps, it randomizes the stars when they wrap... so, this will become
//a "normal" starfield.
void move_starfield_normal_random(void) {
	word i;

	for(i=0;i<NUM_STARS;i++) {
		starfield_array[i].z+=starfield_direction;
		if(starfield_array[i].z>125) {
			starfield_array[i].z-=250;
			starfield_array[i].x=(rand()%250)-125;
			starfield_array[i].y=(rand()%250)-125;
		}
		if(starfield_array[i].z<-125) {
			starfield_array[i].z+=250;
			starfield_array[i].x=(rand()%250)-125;
			starfield_array[i].y=(rand()%250)-125;
		}
	}
}

//Moves the starfield wrapping on Z position and moving the center when
//it wraps, it randomizes the stars when they wrap... so, this will become
//a "normal" starfield, and this one will make each other dot to go
//in the opposite direction
void move_starfield_normal_randomdir(void) {
	word i;

	for(i=0;i<NUM_STARS;i++) {
		if(!(i%2))
			starfield_array[i].z+=starfield_direction;
		else
			starfield_array[i].z-=starfield_direction;

		if(starfield_array[i].z>125) {
			starfield_array[i].z-=250;
			starfield_array[i].x=(rand()%250)-125;
			starfield_array[i].y=(rand()%250)-125;
		}
		if(starfield_array[i].z<-125) {
			starfield_array[i].z+=250;
			starfield_array[i].x=(rand()%250)-125;
			starfield_array[i].y=(rand()%250)-125;
		}
	}
}

// Balls structure for the bouncing effect at the end
typedef struct {
	int x,y;
	int dx,dy;
	int ddx,ddy;
	byte color;
	int MaxYValue;
} movement;

movement far Balls[NUM_STARS];
int Pull;

// This copies the stars 2D position to the Balls structure
void copy_stars_to_balls(void) {
	word I;

	for(I=0;I<NUM_STARS;I++) {
		Balls[I].x=starfield.Xp[I];
		Balls[I].y=starfield.Yp[I];

		Balls[I].color=starfield.Light[I];
		Balls[I].ddx=0;
		Balls[I].ddy=1;
		Balls[I].dx=(rand()%5)-2;
		if(!Balls[I].dx) Balls[I].dx=1;
		Balls[I].dy=0;
		Balls[I].MaxYValue=Balls[I].y;
	}
}

// This puts a pixel, clipping on 0,0,319,199
void put_pixel_clip(word x, word y, byte color, word where) {
	asm {
		cmp word ptr [x],0
		jb  _end
		cmp word ptr [y],0
		jb  _end
		cmp word ptr [x],319
		jae _end
		cmp word ptr [y],199
		jae _end
		mov ax,[where]
		mov es,ax
		mov di,[y]
		mov bx,di
		shl di,6
		shl bx,8
		add di,bx
		add di,[x]
		mov al,[color]
		mov ah,al
		mov es:[di],al
//		mov es:[di+320],ax
	}
	_end:;
}

byte change_ball_color=0;

// This makes the balls bounce and decrease it color
void draw_adv_balls(word where) {
	word I;

	for(I=0;I<NUM_STARS;I++) {
		change_ball_color=0;
		if(Balls[I].color>0 && !change_ball_color) Balls[I].color--;
		if(Balls[I].y<197 && Balls[I].y>0)
			put_pixel_clip(Balls[I].x,Balls[I].y,0,where);
		Balls[I].dx+=Balls[I].ddx;
		Balls[I].dy+=Balls[I].ddy;
		Balls[I].x+=Balls[I].dx;
		Balls[I].y+=Balls[I].dy;
		if(Balls[I].x<1)   {Balls[I].x=1;  Balls[I].dx*=-1;}
		if(Balls[I].x>319) {Balls[I].x=319;Balls[I].dx*=-1;}
		if(Balls[I].y>198) {
			Balls[I].y=198-(Balls[I].y-198)+1;
			if(Balls[I].MaxYValue>30) {
				Balls[I].dy=(Balls[I].dy*-1) + Pull;
			} else {
				Balls[I].dy=(Balls[I].dy*-1) + 30;
				Balls[I].MaxYValue+=30;
			}
		}
		if(Balls[I].y<197 && Balls[I].y>0) {
			put_pixel_clip(Balls[I].x, Balls[I].y, Balls[I].color, where);
		}
	}
}

// This will draw a 3D dot object in a specified color
void draw_object(Tobj3D obj, byte color, word where) {
	word i;

	for(i=0;i<obj.num_vertex;i++) {
		put_pixel_clip(obj.Xp[i],obj.Yp[i],color, where);
	}
}

// This will draw a 3D dot object using deepness color calculation
void draw_object_light(Tobj3D obj, word where) {
	word i;

	for(i=0;i<obj.num_vertex;i++) {
		put_pixel_clip(obj.Xp[i],obj.Yp[i],obj.Light[i], where);
	}
}

// This will calculate a 3D dot object 2D position, having into account the
// rotation, center, z deepness and will also calculate lighting values
// Intensive use of floating point here... this routine is SLOW!!
void calc_object(Tobj3D *obj) {
	float X,Y,Z,X1,Y1,Z1;
	word i;

	for(i=0;i<obj->num_vertex;i++) {
		X1=Cosinus(obj->phiY)*obj->points[i].x-Sinus(obj->phiY)  *(obj->points[i].z);
		Z1=Sinus(obj->phiY)  *obj->points[i].x+Cosinus(obj->phiY)*(obj->points[i].z);
		X=Cosinus(obj->phiZ) *X1              +Sinus(obj->phiZ)  *obj->points[i].y;
		Y1=Cosinus(obj->phiZ)*obj->points[i].y-Sinus(obj->phiZ)  *X1;
		Z=Cosinus(obj->phiX) *Z1              -Sinus(obj->phiX)  *Y1;
		Y=Sinus(obj->phiX)   *Z1              +Cosinus(obj->phiX)*Y1;

		obj->Xp[i]=(Xc*Z-X*Zc)/(Z+obj->movz-Zc)+obj->pos_x;
		obj->Yp[i]=(Yc*Z-Y*Zc)/(Z+obj->movz-Zc)+obj->pos_y;
		obj->Light[i]=(Z+obj->movz-125)*256/300;
		obj->Light[i]=obj->Light[i]*obj->light_steps/256;
		obj->Light[i]+=obj->base_light_color;
	}
}

