/* $Id: printpar.c 1.18 Sun, 23 May 1999 16:32:31 +0200 hlovdal $ */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <time.h>
#include <string.h>
#include <limits.h>
#include <errno.h>

#include "llseek.h"

#ifndef VERSION
#error "This file should be compiled with a version string defined on the command line"
#endif


#define NOEXTERN
#include "globalvars.h"
#include "error.h"
#undef NOEXTERN
#include "byte.h"
#include "boolean.h"
#include "geometry.h"
#include "partitiontable.h"
#include "usage.h"
#include "mk_partition_name.h"
#include "parse_scale.h"
#include "is_extended_partition.h"

#include "ps_table.h"
#include "ps_chart.h"
#include "html_table.h"
#include "txt_chart.h"
#include "txt_table.h"

/* function pointers */

void (*print_mbr_partition)(table_entry_t par_table[], const char *blockfilename, const char *header_description);

void (*print_drive_table)(table_entry_t par_table[], const char *blockfilename, loff_t sector, unsigned int par_no);

void (*ps_print_prolog)(ps_encoding_t ps_encoding);
void (*ps_print_trailer)(void);


/* prototypes */

boolean_t parse_length_arg(char *optarg, char **length_str);
void print_disk(const char *blockfilename, const char *header_description);
boolean_t entry_is_zero(table_entry_t *te);


int get_ext_par_entry_index (table_entry_t par_tab[]);
boolean_t read_table (int fd, table_entry_t par_table[]);
void read_table_entry (table_entry_t *te, byte raw_te[]);
void warn_if_nonzero(table_entry_t par_table[]);
boolean_t hbit_set(const char *str);


int
main(int argc, char *argv[])
{
	int ch;
	char *outfilename = NULL;
	struct {
		char *text;
		boolean_t nonascii_is_present;
	} header_description = { "", false };
	ps_encoding_t ps_encoding = no_encoding_specified;
	int option_index = 0;
	static struct option long_options[] = {
		{"help",			0, 0, 'h'},
		{"version",			0, 0, 'V'},
		{"machine",			1, 0, 'm'}, /* backward comp. */
		{"description",			1, 0, 'd'},
		{"output",			1, 0, 'o'},
		{"format",			1, 0, 'f'},
		{"one-hundred-mb-length",	1, 0, 'l'}, /* backward comp. */
		{"scale",			1, 0, 's'},
		{"postscript",			0, 0, 'p'},
		{"encoding",			1, 0, 'e'},
		{"width",			1, 0, 'w'},
		{"lines-pr-page",		1, 0, '\0'},
		{"text",			0, 0, '\0'},
		{"html",			0, 0, '\0'},
		{NULL,				0, 0, '\0'}
	};
	argv0 = argv[0];

	if (sizeof(byte) != 1 || sizeof(word) != 2 || sizeof(dword) != 4) {
		error("fatal error: (sizeof(byte) != 1 || "
			"sizeof(word) != 2 || sizeof(dword) != 4)\n");
		exit(-1);
	}
	do {
		ch = getopt_long(argc, argv, "hVm:o:f:l:s:pe:w:",
				 long_options, &option_index);
		switch (ch) {
		case '\0':
#define LONGOPTION_IS(str)	(strcmp(str,long_options[option_index].name)==0)
			if (LONGOPTION_IS("html"))
				output_lang = HTML;
			else if (LONGOPTION_IS("text"))
				output_lang = PLAINTEXT;
			else if (LONGOPTION_IS("lines-pr-page"))
				lines_pr_page = atoi(optarg);
			break;
		case 'V':
			printf("%s: version " VERSION "\n", argv0);
			exit(0);
		case 'm': /* backward comp. */
		case 'd':
			header_description.text = optarg;
			header_description.nonascii_is_present=hbit_set(optarg);
			break;
		case 'o':
			outfilename = optarg;
			if (strncmp("/dev/", outfilename, strlen("/dev/"))==0) {
				error("Whoops! Are you really, really, REALLY "
				      "sure that you want a device file as "
				      "outfile?????\nIf that should be the "
				      "case, do not prefix with \"/dev/\"\n");
				exit(1);
			}
			break;
		case 'f':
			if (strcmp("table", optarg) == 0)
				output_format = TABLE;
			else if (strcmp("chart", optarg) == 0)
				output_format = CHART;
			else if (strcmp("hex", optarg) == 0)
				output_format = HEXPRINT;
			else {
				error("unknown output format: %s\n", optarg);
				exit(1);
			}
			break;
		case 'l':	/* backward comp. */
			if (strcmp("max", optarg) != 0) {
				char *tmp = malloc(strlen(optarg)
					+ strlen("100Mb:") + 1);
				strcpy(tmp, "100Mb:");
				strcat(tmp, optarg);
				optarg = tmp;	/* memory leak */
			}
			/* break through */
		case 's':
			if (strcmp("max", optarg) == 0) {
				scale.use_whole_page = true;
			} else {
				scale.use_whole_page = false;
				parse_scale(optarg, &scale);
			}
			break;
		case 'p':
			output_lang = POSTSCRIPT;
			break;
		case 'w':
			width = atoi(optarg);
			if (! (width == 72 || width == 80)) {
				error("invalid width: %d (must be 72 or 80)\n",
					width);
				exit(1);
			}
			break;
		case 'e':
			if (strcmp(optarg, "StandardEncoding") == 0) {
				ps_encoding = StandardEncoding;
			} else if (strcmp(optarg, "ISOLatin1Encoding") == 0) {
				ps_encoding = ISOLatin1Encoding;
			} else if (strcmp(optarg, "ISOLatin2Encoding") == 0) {
				ps_encoding = ISOLatin2Encoding;
			} else if (strcmp(optarg, "ISOLatin3Encoding") == 0) {
				ps_encoding = ISOLatin3Encoding;
			} else if (strcmp(optarg, "ISOLatin4Encoding") == 0) {
				ps_encoding = ISOLatin4Encoding;
			} else if (strcmp(optarg, "ISOLatin5Encoding") == 0) {
				ps_encoding = ISOLatin5Encoding;
			} else {
				error("unknown postscript encoding '%s'\n",
					optarg);
				exit(1);
			}
		case EOF:
			break;
		default:
			error("Unknown option '-%c'\n", ch);
			/* break through */
		case 'h':
		case ':':
		case '?':
			usage();
			exit(ch == 'h' ? 0 : 1);
		}
	} while (ch != EOF);

	if (output_format == HEXPRINT) {
		error("hex output format not supported (yet)\n");
		exit(1);
	}

	if (output_format == CHART &&
	!(output_lang == POSTSCRIPT || output_lang == PLAINTEXT)) {
		error("chart format (currently) only works with "
		"postscript and text\n");
		exit(1);
	}

	if (scale.use_whole_page == false) {
		if (output_format != CHART) {
			error("Specifying a scale is only useful for"
				" \"chart\" format\n");
			exit(1);
		}
		if (strcmp(scale.length_unit, "lines") == 0) {
			if (output_lang != PLAINTEXT) {
				error("'lines' is only a valid length unit "
					"for text output\n");
				exit(1);
			}
		} else {
			if (output_lang == PLAINTEXT) {
				error("'lines' is the only valid length unit "
					"for text output\n");
				exit(1);
			}
		}
	}

	if (ps_encoding == no_encoding_specified) {
		if (header_description.nonascii_is_present)
			ps_encoding = ISOLatin1Encoding;
		else
			ps_encoding = StandardEncoding;
	}

	switch (output_lang) {
	case PLAINTEXT:
		switch (output_format) {
		case TABLE:
			print_mbr_partition = txt_table__print_mbr_partition;
			print_drive_table   = txt_table__print_drive_table;
			break;
		case CHART:
			print_mbr_partition = txt_chart__print_mbr_partition;
			print_drive_table   = txt_chart__print_drive_table;
			break;
		default: break;
		}
		break;
	case POSTSCRIPT:
		switch (output_format) {
		case TABLE:
			print_mbr_partition = ps_table__print_mbr_partition;
			print_drive_table   = ps_table__print_drive_table;
			ps_print_prolog	    = ps_table__print_prolog;
			ps_print_trailer    = ps_table__print_trailer;
			break;
		case CHART:
			print_mbr_partition = ps_chart__print_mbr_partition;
			print_drive_table   = ps_chart__print_drive_table;
			ps_print_prolog	    = ps_chart__print_prolog;
			ps_print_trailer    = ps_chart__print_trailer;
			break;
		default: break;
		}
		break;
	case HTML:
		print_mbr_partition = html_table__print_mbr_partition;
		print_drive_table   = html_table__print_drive_table;
		break;
	}

	if (outfilename != NULL) {
		outfile = fopen(outfilename, "w+");
		if (outfile == NULL) {
			error("unable to open %s: %s\n", outfilename,
				strerror(errno));
			exit(1);
		}
	}

	switch (output_lang) {
	case POSTSCRIPT:
		ps_print_prolog(ps_encoding);
		break;
	case HTML:
		fprintf(outfile,
			#define __HTML_HEAD
			#include "inc-html_table.h"
			);
		break;
	default:
		break;
	}

	if (optind == argc) {	/* no files specified */
		print_disk("/dev/hda", header_description.text);
		print_disk("/dev/hdb", header_description.text);
	} else {		/* process files specified */
		for (; optind < argc; optind++) {
			print_disk(argv[optind], header_description.text);
		}
	}

	switch (output_lang) {
	case POSTSCRIPT:
		ps_print_trailer();
		break;
	case HTML:
		fprintf(outfile,
			#define __HTML_END
			#include "inc-html_table.h"
			);
		break;
	default:
		break;
	}

	if (outfilename != NULL) {
		fclose(outfile);
	}
	return 0;
}

/*****************************************************************************/

int
get_ext_par_entry_index(table_entry_t par_tab[])
{
	int i;
	for (i = 0; i < 4; i++)
		if (is_extended_partition(par_tab[i]))
			return i;
	return -1;
}


boolean_t
entry_is_zero(table_entry_t *te)
{
	byte *tmp;
	for (tmp = (byte *) te;
	     tmp < ((byte *) te + sizeof(table_entry_t));
	     tmp++) {
		if (*tmp != 0)
			return false;
	}
	return true;
}

boolean_t
read_table(int fd, table_entry_t par_table[])
{
	byte mbr[BYTES_PR_SECTOR];
	int i;
	if (read(fd, &mbr, BYTES_PR_SECTOR) != BYTES_PR_SECTOR) {
		error("read failed: %s\n", strerror(errno));
		return false;
	}
	if (mbr[BYTES_PR_SECTOR-2] !=0x55 || mbr[BYTES_PR_SECTOR-1] !=0xAA)
		error("Warning: invalid partition signature "
			"0x%02x,0x%02x (should be 0x55,0xAA)\n",
			mbr[BYTES_PR_SECTOR - 2], mbr[BYTES_PR_SECTOR - 1]);
	for (i = 0; i < 4; i++) {
		read_table_entry(&par_table[i], &mbr[0x01BE] + (i*16));
	}
	return true;
}

void
read_table_entry(table_entry_t *te, byte raw_te[])
{
	te->bootable		  = raw_te[0];
	te->start_head		  = raw_te[1];
	te->start_cylinder	  = get_cylinder(raw_te[2], raw_te[3]);
	te->start_sector	  = get_sector(raw_te[2]);
	te->system		  = raw_te[4];
	te->end_head		  = raw_te[5];
	te->end_cylinder	  = get_cylinder(raw_te[6], raw_te[7]);
	te->end_sector		  = get_sector(raw_te[6]);
	te->relative_start_sector = get_dword(raw_te[8],
					      raw_te[9],
					      raw_te[10],
					      raw_te[11]);
	te->number_of_sectors	  = get_dword(raw_te[12],
					      raw_te[13],
					      raw_te[14],
					      raw_te[15]);
}


void
print_disk(const char *blockfilename, const char *header_description)
{
	int		fd, i, par_no;
	loff_t		abs_start;
	table_entry_t	par_table[4];
	memset(par_table, 0, sizeof(par_table));
	ext_par_absolute_start_sector		= 0;
	saved_ext_par_relative_start_sect	= 0;

	fd = open(blockfilename, O_RDONLY);
	if (fd < 0) {
		error("failed to open %s: %s\n",blockfilename, strerror(errno));
		return;
	}
	if (get_disk_geometry(fd) == false) {
		error("get_disk_geometry failed: %s\n", strerror(errno));
		close(fd);
		return;
	}
	llseek(fd, 0 /* chs2sectors(0,0,1) * BYTES_PR_SECTOR */, SEEK_SET);
	if (read_table(fd, par_table) == false) {
		error("read_table failed: %s\n", strerror(errno));
		close(fd);
		return;
	}
	for (i=0; i<4; i++) {
		if (par_table[i].system == 0x00)
			par_table[i].entry_type = unused;
		else if (is_extended_partition(par_table[i]))
			par_table[i].entry_type = extended;
		else
			par_table[i].entry_type = primary;
	}
	page_number++;
	print_mbr_partition(par_table, blockfilename, header_description);
	warn_if_nonzero(par_table);

	/* For all the drive tables the field relative_start_sector in
	   the second entry refers to starting sector *within* the
	   extended partition.

	   The first drive table at the start of the extended partition
	   (which defines hda5) is found using the relative_start_sector
	   field from the extended partition entry in the mbr, so for
	   this there is no special case.

	   But for all the successive drive tables (defining hda6 and
	   up) the value of relative_start_sector for the extended
	   partition entry in the MBR must be added with the given
	   relative_start_sector in order to get the absolute starting
	   sector.

	   The value of ext_par_absolute_start_sector is therefore
	   zero first time when processing hda5 and afterwords contains
	   the starting sector of the extended partition.
	 */

	for (par_no=5; ; par_no++) {
		i = get_ext_par_entry_index(par_table);
		if (i == -1)
			break;
		abs_start = (loff_t) par_table[i].relative_start_sector
			    + ext_par_absolute_start_sector;
		if (par_no == 5)
			ext_par_absolute_start_sector = abs_start;
		if (llseek(fd, abs_start * BYTES_PR_SECTOR, SEEK_SET) < 0) {
			error("llseek failed: %s\n", strerror(errno));
			break;
		}
		if (read_table(fd, par_table) == false) {
			error("read_table failed: %s\n", strerror(errno));
			break;
		};
		for (i=0; i<4; i++) {
			if (par_table[i].system == 0x00)
				par_table[i].entry_type = unused;
			else if (is_extended_partition(par_table[i]))
				par_table[i].entry_type = drive_table_link;
			else
				par_table[i].entry_type = logical;
		}
		print_drive_table(par_table, blockfilename, abs_start, par_no);
		warn_if_nonzero(par_table);
	}
	close(fd);
	if (output_lang == POSTSCRIPT)
		fprintf(outfile, "\nshowpage\n");
	if (output_lang == PLAINTEXT && output_format == CHART)
		txt_chart__post_print();
}

void
warn_if_nonzero(table_entry_t par_table[])
{
	int i;
	for (i=0; i<4; i++) {
		if (par_table[i].system == 0x00 &&
		!entry_is_zero(&par_table[i]))
			fprintf(stderr, "Warning: nonzero entry %d "
				"although system == 0x00!\n", i);
	}
}

boolean_t
hbit_set(const char *str)
{
	while(*str++)
		if (*str & 0x80)
			return true;
	return false;
}


