/*
 * $Id: find_drive_table.c 1.1 Mon, 25 Jan 1999 00:53:45 +0100 hlovdal $
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <getopt.h>
#include <string.h>

#include "llseek.h"
#include "byte.h"
#include "boolean.h"
#define NOEXTERN
#include "error.h"
#include "geometry.h"
#undef NOEXTERN

#define BYTES_PR_CYLINDER					\
(								\
	HEADS_PR_CYLINDER*SECTORS_PR_TRACK*BYTES_PR_SECTOR	\
)

#define TABLE_OFFSET	0x1BE

const char drive_table_pattern[] =	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
					"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
					/*"\x55\xAA"*/ "\125\252";

#define week_match(buf)							      \
(									      \
	(buf)[BYTES_PR_SECTOR-2] == 0x55 && (buf)[BYTES_PR_SECTOR-1] == 0xAA  \
)

#define strong_match(buf)						    \
(									    \
	memcmp(drive_table_pattern, (buf)+TABLE_OFFSET+2*16, 2*16+2) == 0   \
)


FILE *outfile = stdout;

void usage(void);
void find_drive_table(const char *blockfilename);

int
main(int argc, char *argv[])
{
	int ch;
	char *outfilename = NULL;
	int option_index = 0;
	static struct option long_options[] = {
		{"help",			0, 0, 'h'},
		{"version",			0, 0, 'V'},
		{"output",			1, 0, 'o'},
		{NULL,				0, 0, '\0'}
	};
	argv0 = argv[0];

	if (sizeof(byte) != 1 || sizeof(word) != 2 || sizeof(dword) != 4) {
		error("fatal error: (sizeof(byte) != 1 || "
			"sizeof(word) != 2 || sizeof(dword) != 4)\n");
		exit(-1);
	}
	do {
		ch = getopt_long(argc, argv,"hVo:",long_options, &option_index);
		switch (ch) {
		case 'V':
			printf("%s: version " VERSION "\n", argv0);
			exit(0);
		case 'o':
			outfilename = optarg;
		case EOF:
			break;
		default:
			error("Unknown option '-%c'\n", ch);
			/* break through */
		case 'h':
		case ':':
		case '?':
			usage();
			exit(ch == 'h' ? 0 : 1);
		}
	} while (ch != EOF);

	if (outfilename != NULL) {
		outfile = fopen(outfilename, "w+");
		if (outfile == NULL) {
			error("unable to open %s: %s\n",outfilename,
				strerror(errno));
			exit(1);
		}
	}

	if (optind == argc) {	/* no files specified */
		find_drive_table("/dev/hda");
		find_drive_table("/dev/hdb");
	} else {		/* process files specified */
		for (; optind < argc; optind++) {
			find_drive_table(argv[optind]);
		}
	}

	if (outfilename != NULL) {
		fclose(outfile);
	}
	return 0;
}

void
usage(void)
{
	printf(
"\n"
"Usage: %s [options] [files]\n"
"files defaults to /dev/hda and /dev/hdb if no arguments are given.\n"
"\noptions:\n"
"  -h, --help\t\t\t"		"Show this message\n"
"  -V, --version\t\t\t"		"Show version and exit\n"
"  -o, --output=OUTPUTFILE\t"	"Write output to OUTPUTFILE\n"
"\n", argv0);
}

void
find_drive_table(const char *blockfilename)
{
	byte buf[BYTES_PR_SECTOR + 1];
	unsigned long current_cyl;
	int fd = open(blockfilename, O_RDONLY);
	if (fd < 0) {
		error("failed to open %s: %s\n",blockfilename, strerror(errno));
		return;
	}
	if (get_disk_geometry(fd) == false) {
		error("get_disk_geometry failed\n");
		close(fd);
		return;
	}
	fprintf(outfile, "\tdisk %s\n", blockfilename);
	for(current_cyl=0; current_cyl<TOTAL_CYLINDERS ;) {
		fprintf(stderr, "\rchecking cylinder %lu", current_cyl);
		if (read(fd, &buf, BYTES_PR_SECTOR) != BYTES_PR_SECTOR) {
			error("read failed: ", strerror(errno));
			return;
		}
		if (strong_match(buf)) {
			fprintf(stderr, "\r                                \r");
			fprintf(outfile, "strong drive table match at "
				"cylinder %lu\n", current_cyl);
		} else if (week_match(buf)) {
			fprintf(stderr, "\r                                \r");
			fprintf(outfile, "week table match at cylinder %lu\n",
				current_cyl);
		}
		current_cyl++;
		if (llseek(fd, current_cyl*BYTES_PR_CYLINDER, SEEK_SET) < 0) {
			error("llseek failed: ", strerror(errno));
                        close(fd);
                        return;
		}
	}
	fprintf(stderr, "\r                                              \r");
	close(fd);
}

