/*
    Mplot++ : a math plotter for Unix(R)/Windows(R)/MacOS X(R) -
              - version 0.78     
    Copyright (C)  2002    Ivano Primi ( ivano.primi@tin.it )    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    You can contact the author of this software by paper mail writing to
    the next address

	Ivano Primi
	via Colle Cannetacce 50/A
	C.A.P. 00038 - Valmontone (ROMA)
	Italy                                                          .

    If you prefer the electronic mail you can write to the address

	ivano.primi@tin.it                                             .
*/

// File containing the callbacks definitions
#include<cstdio>
#include<cstring>
#include<cmath>
#include<cerrno>
#include <FL/fl_show_input.H> // FLTK convenience functions
#include <FL/fl_ask.H>
#include <FL/fl_file_chooser.H>
#include"loadfun.h" // My functions
#include"mplot.h"
#include"mathplot.h"
#include"numbutils.h"
#include"open_files.h"
#include"makeps.h"
#define ABSVCPC 4 // To be added to OFLOW, UFLOW and NAN defined in numbutils.h
#define WDIMV      5
#define WARV       6
#define WIV        7
#define BADNNC2D   8 
#define BADNNC3D   9 
#define BADNNSF   10 
#define BADNN     11 

static MpCanvas* CNV=new MpCanvas;
static MpFormC2* FORMC2=new MpFormC2;
static MpFormC3* FORMC3=new MpFormC3; 
static MpFormSf* FORMSF=new MpFormSf; 
static MpLoad* MLOAD=new MpLoad;
static MpHelp* MHELP=new MpHelp;    
static MpPref* MPREF=new MpPref;
static MpPrint*  MPRINT=new MpPrint;
static int obj_to_plot=C2D;
extern int angle_m;
extern struct __prm prm[];


static void print_error(int errcode, const char* where)
{
  char  s[80];
  char ss[70]; 
   
  strcpy(s,where);
  switch(errcode)
    {
    case UFLOW:
      strcat(s," Underflow occurred");
      break;
    case OFLOW:
      strcat(s," Overflow occurred");
      break;
    case NAN:
      strcat(s," Not a number");
      break;
    case ABSVCPC:
      sprintf(ss," Its absolute value must not be greater than %g",MAX_MOD_CPCOORD);
      strcat(s,ss);
      break;
    case WDIMV:
      sprintf(ss," Must be comprised between %g and %g",MIN_DIM,MAX_DIM);
      strcat(s,ss);
      break;
    case WARV:
      sprintf(ss," Must be comprised between %.1f and %.0f",MIN_AR,MAX_AR);
      strcat(s,ss);
      break;
    case WIV:
      sprintf(ss," Must be greater than 0 and less or equal to  %u",MAX_EXTVIPRM);
      strcat(s,ss);
      break;
    case BADNNC2D:
      sprintf(ss," Must be greater than 1 and less or equal to  %u",MAX_NNOD_C2D);
      strcat(s,ss);
      break;
    case BADNNC3D:
      sprintf(ss," Must be greater than 1 and less or equal to  %u",MAX_NNOD_C3D);
      strcat(s,ss);
      break;
    case BADNNSF:
      sprintf(ss," Must be greater than 3 and less or equal to  %u",MAX_NNOD_SF);
      strcat(s,ss);
      break;
    case BADNN:
      strcat(s," Must be greater than 1");
    }
  fl_alert("%s",s);
}

static int good2darea(double x0, double y0, double w, double h)
{
  double maxcpc= fabs(x0) > fabs(y0) ? fabs(x0) : fabs(y0);
  double mdim  = w <= h ? w : h;

  if( fabs(x0) > MAX_MOD_CPCOORD )
    {
      print_error(ABSVCPC,"x0:");
      return 0;
    }
  else if( fabs(y0) > MAX_MOD_CPCOORD )
    {
      print_error(ABSVCPC,"y0:");
      return 0;
    }
  else if( w < MIN_DIM  || w > MAX_DIM )
    {
      print_error(WDIMV," w:");
      return 0;
    }
  else if( h < MIN_DIM  || h > MAX_DIM )
    {
      print_error(WDIMV," h:");
      return 0;
    }
  else if( (h/w) < MIN_AR || (h/w) > MAX_AR )
    {
      print_error(WARV," h/w:");
      return 0;
    }
  else if( mdim < MIN_DIMOCPC * maxcpc )
    {
      fl_alert("The smallest dimension of the display\narea can not be less than\n %.12f", MIN_DIMOCPC*maxcpc);
      return 0;
    }
  else
    return 1;
}

static int good3darea(double x0, double y0, double z0, double R)
{
  double maxcpc;

  maxcpc= fabs(x0) > fabs(y0) ? fabs(x0) : fabs(y0);
  maxcpc= fabs(z0) > maxcpc ? fabs(z0) : maxcpc;
  if( fabs(x0) > MAX_MOD_CPCOORD )
    {
      print_error(ABSVCPC,"x0:");
      return 0;
    }
  else if( fabs(y0) > MAX_MOD_CPCOORD )
    {
      print_error(ABSVCPC,"y0:");
      return 0;
    }
  else if( fabs(z0) > MAX_MOD_CPCOORD )
    {
      print_error(ABSVCPC,"z0:");
      return 0;
    }
  else if( (R < MIN_DIM)  || (R > MAX_DIM) )
    {
      print_error(WDIMV," R:");
      return 0;
    }
  else if( R < MIN_DIMOCPC * maxcpc )
    {
      fl_alert("The radius of the display\narea can not be less than\n %.12f", MIN_DIMOCPC*maxcpc);
      return 0;
    }
  else
    return 1;
}

static float max(float a,float b)
{
  return (a > b ? a : b);
}

// Callbacks bounded to the objects of the MpGUI class
void quit_cb(Fl_Window* w, void* v)
{ 
  CNV->Canvas->free_mem();
  w->hide();
}

void c2cb(Fl_Check_Button* b, void* v)
{
  obj_to_plot=C2D;
}

void c3cb(Fl_Check_Button* b, void* v)
{
  obj_to_plot=C3D;
}

void sfcb(Fl_Check_Button* b, void* v)
{
  obj_to_plot=SF;
}

void radcb(Fl_Check_Button* b, void* v)
{
  angle_m=1;
}

void degcb(Fl_Check_Button* b, void* v)
{
  angle_m=0;
}

void okcb(Fl_Return_Button* b, void* v)
{
  CNV->Canvas->listpareq(1); // Enable the lpareq flag inside CNV->Canvas
  switch(obj_to_plot)        // in case of future print operations.
    {
    case C2D:
      CNV->Canvas->set_type(C2D);
      FORMC2->prmbutt->value(CNV->Canvas->infos_status());
      FORMC2->show();
      break;
    case C3D:
      CNV->Canvas->set_type(C3D);
      FORMC3->prmbutt->value(CNV->Canvas->infos_status());
      FORMC3->show();
      break;
    case SF:
      CNV->Canvas->set_type(SF);
      FORMSF->prmbutt->value(CNV->Canvas->infos_status());
      FORMSF->show();
    }
}

void loadbuttcb(Fl_Button* b, void* v)
{
  CNV->Canvas->listpareq(0); // Disable the lpareq flag inside CNV->Canvas
  // in case of future print operations.
  MLOAD->prmbutt->value(CNV->Canvas->infos_status());
  switch(obj_to_plot)
    {
    case C2D:
      CNV->Canvas->set_type(C2D);
      MLOAD->z0_input->deactivate();
      MLOAD->R_input->deactivate();
      break;
    case C3D:
      CNV->Canvas->set_type(C3D);
      MLOAD->w_input->deactivate();
      MLOAD->h_input->deactivate();
      break;
    case SF:
      CNV->Canvas->set_type(SF);
      MLOAD->w_input->deactivate();
      MLOAD->h_input->deactivate();
    }
  MLOAD->show();
}

void author_cb(Fl_Menu_* m, void* v)
{
  fl_message_font(FL_COURIER_ITALIC,22);
  fl_message("Mplot++ by Ivano Primi\nivano.primi@tin.it\nCopyright 2001");
  fl_message_font(FL_HELVETICA,14);
}

void about_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/ABOUT"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/ABOUT");
  else
    MHELP->show();
}

void fltk_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/FLTK"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/FLTK");
  else
    MHELP->show();
}

void nowarranty_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/NOWARRANTY"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/NOWARRANTY");
  else
    MHELP->show();
}

void license_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/LICENSE"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/LICENSE");
  else
    MHELP->show();
}

void intro_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/introduction.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/introduction.txt");
  else
    MHELP->show();
}

void dispgraphs_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/visual.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/visual.txt");
  else
    MHELP->show();
}

void datafiles_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/lwdata.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/lwdata.txt");
  else
    MHELP->show();
}

void printing_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/print.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/print.txt");
  else
    MHELP->show();
}

void ctrlkeys_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/controlkeys.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/controlkeys.txt");
  else
    MHELP->show();
}

void operators_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/operators.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/operators.txt");
  else
    MHELP->show();
}

void functions_cb(Fl_Menu_* m, void* v)
{
  Fl_Browser* hb=MHELP->help_browser;

  if(!hb->load(DATADIR"/mathfunc.txt"))
    fprintf(stderr,"Error: Cannot open file \"%s\"\n",DATADIR"/mathfunc.txt");
  else
    MHELP->show();
}

void quitbuttcb(Fl_Button* b, void* v)
{
  MpGUI* w=(MpGUI*)v;

  CNV->Canvas->free_mem();
  w->hide();
}

// Callbacks bounded to the objects of MpLoad 
void exitloadcb(Fl_Window* w, void* v)
{
  if(obj_to_plot==C2D)
    {
      MLOAD->z0_input->activate();
      MLOAD->R_input->activate();
    }
  else
    {
      MLOAD->w_input->activate();
      MLOAD->h_input->activate();
    }
  w->hide();
}

void prmldcb(Fl_Round_Button *rb, void *v)
{
  CNV->Canvas->infos_on_off(rb->value());
}

void prefcb(Fl_Button * b, void * v)
{
  MPREF->show();
}

void opencb(Fl_Button* b, void* v)
{
  MpLoad* Form=(MpLoad*)v; 
  char *path,*errmsg;
  FILE* pf;
  struct Param2d p2d;
  struct Param3d p3d;
  int errcode;
  float M;

  if( CNV->Canvas->get_type()==C2D )
    {
      if( (errcode=stodbl(Form->x0_input->value(),&p2d.x0)) )
	{
	  print_error(errcode,"x0:");
	  return;
	}
      else if( (errcode=stodbl(Form->y0_input->value(),&p2d.y0)) )
	{
	  print_error(errcode,"y0:");
	  return;
	}
      else if( (errcode=stodbl(Form->w_input->value(),&p2d.w)) )
	{
	  print_error(errcode," w:");
	  return;
	}
      else if( (errcode=stodbl(Form->h_input->value(),&p2d.h)) )
	{
	  print_error(errcode," h:");
	  return;
	}
      else if( !good2darea(p2d.x0,p2d.y0,p2d.w,p2d.h) )
	return;
    }
  else
    {
      if( (errcode=stodbl(Form->x0_input->value(),&p3d.x0)) )
	{
	  print_error(errcode,"x0:");
	  return;
	}
      else if( (errcode=stodbl(Form->y0_input->value(),&p3d.y0)) )
	{
	  print_error(errcode,"y0:");
	  return;
	}
      else if( (errcode=stodbl(Form->z0_input->value(),&p3d.z0)) )
	{
	  print_error(errcode,"z0:");
	  return;
	}
      else if( (errcode=stodbl(Form->R_input->value(),&p3d.R)) )
	{
	  print_error(errcode," R:");
	  return;
	}
      else if( !good3darea(p3d.x0,p3d.y0,p3d.z0,p3d.R) )
	return;
    }
  path=fl_file_chooser("Select the data file","*.mpp",NULL);
  if( !(path) )
    return;
  if( !(pf=open_file(path,TO_READ,&errcode)) )
    {
      switch(errcode)
	{
	case FAILED_SCANDIR:
	  fl_alert("Error in directory scansion");
	  return;
	case TYPE_OR_PERMS:
	  fl_alert("Cannot open file:\nWrong type or permissions ?");
	  return;
	case CHECK_ERROR:
	  fl_alert("Error in file checking");
	  return;
	case UNEXIST_FILE:
	  fl_alert("The requested file does not exist");
	  return;
	case IDONOTLIKE_HDLNS:
	  fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
	  return;
	default: // Another type of error has happened
	  fl_alert("Cannot open file:\n%s",strerror(errcode));
	  return;
	}
    }
  // If you arrive here pf contains the address
  // of a file opened to be read.
  switch( CNV->Canvas->get_type() )
    {
    case C2D:
      if( (errmsg=LoadC2Dparams(pf, &p2d)) )
	{
	  if( (fclose(pf)) )
	    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	  else
	    fl_alert("%s",errmsg);
	  return;
	}
      else if( (CNV->Canvas->set_par2d(&p2d)) )
	{
	  fl_alert("Error in allocation of memory\ninside mycanvas->set_par2d()");
	  return;
	}
      else if( (errmsg=LoadC2(CNV->Canvas, pf, &p2d)) )
	{
	  if( (fclose(pf)) )
	    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	  else
	    fl_alert("%s",errmsg);
	  return;
	}
      else if( (fclose(pf)) )
	{
	  fl_alert("Error in closing file:\n%s",strerror(errno));
	  return;
	}
      else
	{
	  CNV->x0_output->value(p2d.x0);
	  CNV->y0_output->value(p2d.y0);
	  CNV->z0_output->deactivate();
	  CNV->longcount->deactivate();
	  CNV->latcount->deactivate();
	  CNV->vangcount->deactivate();
	  CNV->RUbutt->deactivate();
	  CNV->LDbutt->deactivate();
	  CNV->fogroll->deactivate();
	  CNV->Canvas->zoomreset();
	  M=max(p2d.h,p2d.w);
	  CNV->gstep->minimum(M/100);
	  CNV->gstep->value(M/100);
	  CNV->Canvas->set_gstep(CNV->gstep->value());
	  CNV->show();
	}
      break;
    case C3D:
      p3d.nnods=1; // Without these You would have a Segmentation fault
      if( (errmsg=LoadC3Dparams(pf, &p3d)) )
	{
	  if( (fclose(pf)) )
	    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	  else
	    fl_alert("%s",errmsg);
	  return;
	}
      else if( (CNV->Canvas->set_par3d(&p3d)) )
	{
	  fl_alert("Error in allocation of memory\ninside mycanvas->set_par3d()");
	  return;
	}
      else if( (errmsg=LoadC3(CNV->Canvas, pf, &p3d)) )
	{
	  if( (fclose(pf)) )
	    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	  else
	    fl_alert("%s",errmsg);
	  return;
	}
      else if( (fclose(pf)) )
	{
	  fl_alert("Error in closing file:\n%s",strerror(errno));
	  return;
	}
      else
	{
	  CNV->x0_output->value(p3d.x0);
	  CNV->y0_output->value(p3d.y0);
	  CNV->z0_output->value(p3d.z0);
	  CNV->Canvas->zoomreset();
	  CNV->gstep->minimum(p3d.R/20);
	  CNV->gstep->value(p3d.R/20);
	  CNV->Canvas->set_gstep(CNV->gstep->value());
	  CNV->show();
	}
      break;
    case SF:
      if( (errmsg=LoadSFparams(pf, &p3d)) )
	{
	  if( (fclose(pf)) )
	    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	  else
	    fl_alert("%s",errmsg);
	  return;
	}
      else if( (CNV->Canvas->set_par3d(&p3d)) )
	{
	  fl_alert("Error in allocation of memory\ninside mycanvas->set_par3d()");
	  return;
	}
      else if( (errmsg=LoadSF(CNV->Canvas, pf, &p3d)) )
	{
	  if( (fclose(pf)) )
	    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	  else
	    fl_alert("%s",errmsg);
	  return;
	}
      else if( (fclose(pf)) )
	{
	  fl_alert("Error in closing file:\n%s",strerror(errno));
	  return;
	}
      else
	{
	  CNV->x0_output->value(p3d.x0);
	  CNV->y0_output->value(p3d.y0);
	  CNV->z0_output->value(p3d.z0);
	  CNV->Canvas->zoomreset();
	  CNV->gstep->minimum(p3d.R/20);
	  CNV->gstep->value(p3d.R/20);
	  CNV->Canvas->set_gstep(CNV->gstep->value());
	  CNV->show();
	}
    } //end of switch( ... )
}

void closeldcb(Fl_Button* b, void* v)
{
  MpLoad* w=(MpLoad*)v;

  if(obj_to_plot==C2D)
    {
      MLOAD->z0_input->activate();
      MLOAD->R_input->activate();
    }
  else
    {
      MLOAD->w_input->activate();
      MLOAD->h_input->activate();
    }
  w->hide();
}

// Callbacks bounded to the objects of the MpFormC2
void exitc2cb(Fl_Window* w, void* v)
{
  w->hide();
}

void prmc2cb(Fl_Round_Button* rb, void* v)
{
  CNV->Canvas->infos_on_off(rb->value());
}

void prefc2cb(Fl_Button * b, void * v)
{
  MPREF->show();
}

extern char* WrtC2d(struct Param2d* par,FILE* pf,unsigned char ic);

#define MIN(a,b) ( ((a) < (b)) ? (a) : (b) )

void writec2cb(Fl_Button* b, void* v)
{
  MpFormC2* Form=(MpFormC2*)v;
  const char *p,*s;
  char *path,*oldp,*olds,*errmsg;
  FILE* pf;
  struct Param2d p2d;
  int nlx,nly,errcode,answer;
  unsigned int nchx,nchy;

  // Because I do not need these parameters for the next operations
  // I can set everyone to an arbitrary value.
  p2d.x0=p2d.y0=0;
  p2d.w=p2d.h=1;
  if( (errcode=stodbl(Form->lb_input->value(),&p2d.t1)) )
    print_error(errcode,"lb:");
  else if( (errcode=stodbl(Form->ub_input->value(),&p2d.t2)) )
    print_error(errcode,"ub:");
  else if( (p2d.t2-p2d.t1 <=0) || (p2d.t2-p2d.t1 > MAX_EXTVIPRM) )
    print_error(WIV,"ub-lb:");
  else if( (nlx=nlines(Form->x_input->value())) < 1 || nlx > MAX_NC)
    {
      fl_alert("X: no line or too many lines");
      return;
    }
  else if( (nly=nlines(Form->y_input->value())) < 1 || nly > MAX_NC)
    {
      fl_alert("Y: no line or too many lines");
      return;
    }
  else if( nlx != nly)
    {
      fl_alert("There is a different number of X-lines and Y-lines");
      return;
    }
  else if( (errcode=stoushr(Form->n_input->value(),&p2d.nnod)) )
    print_error(errcode," n:");
  else if( (p2d.nnod < 2) || (nlx* p2d.nnod > MAX_NNOD_C2D) )
    print_error(BADNNC2D," n:");
  else
    {
      p2d.nc=(unsigned char)nlx; // == nly
      path=fl_file_chooser("Choose the data file","*.mpp",NULL);
      if( (path) )
	{
	  if( !(pf=open_file(path,TO_WRITE,&errcode)) )
	    {
	      switch(errcode)
		{
		case FAILED_SCANDIR:
		  fl_alert("Error in directory scansion");
		  return;
		case TYPE_OR_PERMS:
		  fl_alert("Cannot open file:\nWrong type or permissions ?");
		  return;
		case CHECK_ERROR:
		  fl_alert("Error in file checking");
		  return;
		case IDONOTLIKE_HDLNS:
		  fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
		  return;
		case FILE_EXISTS:
		  answer=fl_ask("The requested file already exists,\nDo you want overwrite it \?");
		  if( (answer) ) // answer==1==YES
		    {
		      if( !(pf=open_file(path,TO_OVERWRITE,&errcode)) )
			{
			  switch(errcode)
			    {
			    case FAILED_SCANDIR:
			      fl_alert("Error in directory scansion");
			      break;
			    case TYPE_OR_PERMS:
			      fl_alert("Cannot open file:\nWrong type or permissions ?");
			      break;
			    case CHECK_ERROR:
			      fl_alert("Error in file checking");
			      break;
			    case IDONOTLIKE_HDLNS:
			      fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
			      break;
			    default: // Another type of error has happened
			      fl_alert("Cannot open file:\n%s",strerror(errcode));
			    }
			  return;
			}
		    } // end if((answer))
		  else
		    return;
		  break; // If you may open *path in TO_OVERWRITE mode
		  // you arrive here
		default: // Another type of error has happened
		  fl_alert("Cannot open file:\n%s",strerror(errcode));
		  return;
		}
	    }
	  // If you arrive here pf contains the address
	  // of a file opened to be written.
	  // Every time this function is called she must insert
	  // a line of parameters.
	  fprintf(pf,"% 20f\t% 20f\t% 10hu\t% 10u\n",p2d.t1,p2d.t2,p2d.nnod,p2d.nc); 
	  if( ferror(pf) || feof(pf) )
	    {
	      if( (fclose(pf)) )
		fl_alert("Error in output operation on file\n+ Error in closing file:\n%s",strerror(errno));
	      else
		fl_alert("Error in output operation on file");
	      return;
	    }
	  p=Form->x_input->value();
	  s=Form->y_input->value();
	  oldp=next_line(&p,&nchx);
	  olds=next_line(&s,&nchy); 
	  while( (oldp) && (olds) )
	    {
	      strncpy(prm[nly-nlx].x,oldp,MIN(nchx,MAX_LUN-1));
	      prm[nly-nlx].x[MIN(nchx,MAX_LUN-1)]=0;
	      strncpy(prm[nly-nlx].y,olds,MIN(nchy,MAX_LUN-1));
	      prm[nly-nlx].y[MIN(nchy,MAX_LUN-1)]=0;
	      if( (strchr(prm[nly-nlx].x,'!')) || (strchr(prm[nly-nlx].y,'!')) )
		{
		  if( (fclose(pf)) )
		    fl_alert("Unagreable reference to \"!\" parameter inside one\nof the equations + Error in closing\nfile: %s",strerror(errno));
		  else
		    fl_alert("Unagreable reference to \"!\" parameter\ninside one of the equations");
		  return;
		}
	      if( (errmsg=WrtC2d(&p2d,pf,nly-nlx)) )
		{
		  if( (fclose(pf)) )
		    fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
		  else
		    fl_alert("%s",errmsg);
		  return;
		}
	      oldp=next_line(&p,&nchx);
	      olds=next_line(&s,&nchy);
	      nlx--;
	    }
	  if( (fclose(pf)) )
	    fl_alert("Error in closing file:\n%s",strerror(errno));
	} // end of if( (path) )
      else
	return;
    } // end of the outest else
}

extern char* DsnC2d(mycanvas* cnv, unsigned char ic);

void displayc2cb(Fl_Return_Button* b, void* v)
{
  MpFormC2* Form=(MpFormC2*)v;
  int nlx,nly,errcode;
  unsigned int nchx,nchy;
  struct Param2d p2d;
  const char *p,*s;
  char *oldp,*olds,*errmsg;
  float M;

  if( (errcode=stodbl(Form->x0_input->value(),&p2d.x0)) )
    print_error(errcode,"x0:");
  else if( (errcode=stodbl(Form->y0_input->value(),&p2d.y0)) )
    print_error(errcode,"y0:");
  else if( (errcode=stodbl(Form->w_input->value(),&p2d.w)) )
    print_error(errcode," w:");
  else if( (errcode=stodbl(Form->h_input->value(),&p2d.h)) )
    print_error(errcode," h:");
  else if( !good2darea(p2d.x0,p2d.y0,p2d.w,p2d.h) )
    return;
  else if( (errcode=stodbl(Form->lb_input->value(),&p2d.t1)) )
    print_error(errcode,"lb:");
  else if( (errcode=stodbl(Form->ub_input->value(),&p2d.t2)) )
    print_error(errcode,"ub:");
  else if( (p2d.t2-p2d.t1<=0) || (p2d.t2-p2d.t1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub-lb:");
  else if( (nlx=nlines(Form->x_input->value())) < 1 || nlx > MAX_NC)
    {
      fl_alert("X: no line or too many lines");
      return;
    }
  else if( (nly=nlines(Form->y_input->value())) < 1 || nly > MAX_NC)
    {
      fl_alert("Y: no line or too many lines");
      return;
    }
  else if( nlx != nly)
    {
      fl_alert("There is a different number of X-lines and Y-lines");
      return;
    }
  else if( (errcode=stoushr(Form->n_input->value(),&p2d.nnod)) )
    print_error(errcode," n:");
  else if( (p2d.nnod < 2) || (nlx* p2d.nnod > MAX_NNOD_C2D) )
    print_error(BADNNC2D," n:");
  else
    {
      p2d.nc=(unsigned char)nlx; // == nly
      if( (CNV->Canvas->set_par2d(&p2d)) )
	{
	  fl_alert("Error in allocation of memory\ninside mycanvas->set_par2d()");
	  return;
	}
      p=Form->x_input->value();
      s=Form->y_input->value();
      oldp=next_line(&p,&nchx);
      olds=next_line(&s,&nchy); 
      while( (oldp) && (olds) )
	{
	  strncpy(prm[nly-nlx].x,oldp,MIN(nchx,MAX_LUN-1));
	  prm[nly-nlx].x[MIN(nchx,MAX_LUN-1)]=0;
	  strncpy(prm[nly-nlx].y,olds,MIN(nchy,MAX_LUN-1));
	  prm[nly-nlx].y[MIN(nchy,MAX_LUN-1)]=0;
	  if( (strchr(prm[nly-nlx].x,'!')) || (strchr(prm[nly-nlx].y,'!')) )
	    {
	      fl_alert("Unagreable reference to \"!\" parameter\ninside one of the equations");
	      return;
	    }
	  if( (errmsg=DsnC2d(CNV->Canvas,nly-nlx)) )
	    {
	      fl_alert("%s",errmsg);
	      return;
	    }
	  oldp=next_line(&p,&nchx);
	  olds=next_line(&s,&nchy);
	  nlx--;
	}
      CNV->x0_output->value(p2d.x0);
      CNV->y0_output->value(p2d.y0);
      CNV->z0_output->deactivate();
      CNV->longcount->deactivate();
      CNV->latcount->deactivate();
      CNV->vangcount->deactivate();
      CNV->RUbutt->deactivate();
      CNV->LDbutt->deactivate();
      CNV->fogroll->deactivate();
      CNV->Canvas->zoomreset();
      M=max(p2d.h,p2d.w);
      CNV->gstep->minimum(M/100);
      CNV->gstep->value(M/100);
      CNV->Canvas->set_gstep(CNV->gstep->value());
      CNV->show();
    }
}

void closec2cb(Fl_Button* b, void* v)
{
  MpFormC2* w=(MpFormC2*)v;

  w->hide();
}

// Callbacks bounded to the objects of the MpFormC3
void exitc3cb(Fl_Window* w, void* v)
{
  w->hide();
}

void prmc3cb(Fl_Round_Button* rb, void* v)
{
  CNV->Canvas->infos_on_off(rb->value());
}

extern char* WrtC3d(struct Param3d* par,FILE* pf);

void writec3cb(Fl_Button* b, void* v)
{
  MpFormC3* Form=(MpFormC3*)v;
  int answer,errcode;
  FILE* pf;
  struct Param3d p3d;
  char *path,*errmsg;

  // Because I do not need these parameters for the next operations
  // I can set everyone to an arbitrary value.
  p3d.x0=p3d.y0=p3d.z0=0;
  p3d.R=1;
  if( (errcode=stodbl(Form->lb_input->value(),&p3d.t1)) )
    print_error(errcode,"lb:");
  else if( (errcode=stodbl(Form->ub_input->value(),&p3d.t2)) )
    print_error(errcode,"ub:");
  else if( (p3d.t2-p3d.t1<=0) || (p3d.t2-p3d.t1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub-lb:");
  else if( (errcode=stoushr(Form->n_input->value(),&p3d.nnodt)) )
    print_error(errcode," n:");
  else if( (p3d.nnodt < 2) || (p3d.nnodt > MAX_NNOD_C3D) )
    print_error(BADNNC3D," n:");
  else
    {
      p3d.nnods=1;
      path=fl_file_chooser("Choose the data file","*.mpp",NULL);
      if( (path) )
	{
	  if( !(pf=open_file(path,TO_WRITE,&errcode)) )
	    {
	      switch(errcode)
		{
		case FAILED_SCANDIR:
		  fl_alert("Error in directory scansion");
		  return;
		case TYPE_OR_PERMS:
		  fl_alert("Cannot open file:\nWrong type or permissions ?");
		  return;
		case CHECK_ERROR:
		  fl_alert("Error in file checking");
		  return;
		case IDONOTLIKE_HDLNS:
		  fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
		  return;
		case FILE_EXISTS:
		  answer=fl_ask("The requested file already exists,\nDo you want overwrite it \?");
		  if( (answer) ) // answer==1==YES
		    {
		      if( !(pf=open_file(path,TO_OVERWRITE,&errcode)) )
			{
			  switch(errcode)
			    {
			    case FAILED_SCANDIR:
			      fl_alert("Error in directory scansion");
			      break;
			    case TYPE_OR_PERMS:
			      fl_alert("Cannot open file:\nWrong type or permissions ?");
			      break;
			    case CHECK_ERROR:
			      fl_alert("Error in file checking");
			      break;
			    case IDONOTLIKE_HDLNS:
			      fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
			      break;
			    default: // Another type of error has happened
			      fl_alert("Cannot open file:\n%s",strerror(errcode));
			    }
			  return;
			}
		    } // if( (answer) )
		  else
		    return;
		  break; // If you may open *path in TO_OVERWRITE mode
		  // you arrive here
		default: // Another type of error has happened
		  fl_alert("Cannot open file:\n%s",strerror(errcode));
		  return;
		}
	    }
	  // If you arrive here pf contains the address
	  // of a file opened to be written. 
	  // Every time this function is called she must insert
	  // a line of parameters followed by an empty line.
	  fprintf(pf,"% 20f\t% 20f\t% 20hu\n",p3d.t1,p3d.t2,p3d.nnodt); 
	  if( ferror(pf) || feof(pf) )
	    {
	      if( (fclose(pf)) )
		fl_alert("Error in output operation on file\n+ Error in closing file:\n%s",strerror(errno));
	      else
		fl_alert("Error in output operation on file");
	      return;
	    }
	  strncpy(prm[0].x,Form->x_input->value(),MAX_LUN-1);
	  prm[0].x[MAX_LUN-1]=0;
	  strncpy(prm[0].y,Form->y_input->value(),MAX_LUN-1);
	  prm[0].y[MAX_LUN-1]=0;
	  strncpy(prm[0].z,Form->z_input->value(),MAX_LUN-1);
	  prm[0].z[MAX_LUN-1]=0;
	  if( (strchr(prm[0].x,'!')) || (strchr(prm[0].y,'!')) || (strchr(prm[0].z,'!')) )
	    {
	      if( (fclose(pf)) )
		fl_alert("Unagreable reference to \"!\" parameter inside one\nof the equations + Error in closing\nfile: %s",strerror(errno));
	      else
		fl_alert("Unagreable reference to \"!\" parameter\ninside one of the equations");
	      return;
	    }
	  if( (errmsg=WrtC3d(&p3d,pf)) )
	    {
	      if( (fclose(pf)) )
		fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	      else
		fl_alert("%s",errmsg);
	      return;
	    }
	  if( (fclose(pf)) )
	    fl_alert("Error in closing file:\n%s",strerror(errno));
	} // end of if( (path) )
      else
	return;
    } // end of the outest else
}

extern char* DsnC3d(mycanvas* cnv);

void displayc3cb(Fl_Return_Button* b, void* v)
{
  MpFormC3* Form=(MpFormC3*)v;
  int errcode;
  struct Param3d p3d;
  char* s;

  if( (errcode=stodbl(Form->x0_input->value(),&p3d.x0)) )
    print_error(errcode,"x0:");
  else if( (errcode=stodbl(Form->y0_input->value(),&p3d.y0)) )
    print_error(errcode,"y0:");
  else if( (errcode=stodbl(Form->z0_input->value(),&p3d.z0)) )
    print_error(errcode,"z0:");
  else if( (errcode=stodbl(Form->R_input->value(),&p3d.R)) )
    print_error(errcode," R:");
  else if( !good3darea(p3d.x0,p3d.y0,p3d.z0,p3d.R) )
    return;
  else if( (errcode=stodbl(Form->lb_input->value(),&p3d.t1)) )
    print_error(errcode,"lb:");
  else if( (errcode=stodbl(Form->ub_input->value(),&p3d.t2)) )
    print_error(errcode,"ub:");
  else if( (p3d.t2-p3d.t1<=0) || (p3d.t2-p3d.t1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub-lb:");
  else if( (errcode=stoushr(Form->n_input->value(),&p3d.nnodt)) )
    print_error(errcode," n:");
  else if( (p3d.nnodt < 2) || (p3d.nnodt > MAX_NNOD_C3D) )
    print_error(BADNNC3D," n:");
  else
    {
      p3d.nnods=1;
      if( (CNV->Canvas->set_par3d(&p3d)) )
	{
	  fl_alert("Error in allocation of memory\ninside mycanvas->set_par3d()");
	  return;
	}
      strncpy(prm[0].x,Form->x_input->value(),MAX_LUN-1);
      prm[0].x[MAX_LUN-1]=0;
      strncpy(prm[0].y,Form->y_input->value(),MAX_LUN-1);
      prm[0].y[MAX_LUN-1]=0;
      strncpy(prm[0].z,Form->z_input->value(),MAX_LUN-1);
      prm[0].z[MAX_LUN-1]=0;
      if( (strchr(prm[0].x,'!')) || (strchr(prm[0].y,'!')) || (strchr(prm[0].z,'!')) )
	{
	  fl_alert("Unagreable reference to \"!\" parameter\ninside one of the equations");
	  return;
	}
      if( (s=DsnC3d(CNV->Canvas)) )
	{
	  fl_alert("%s",s);
	  return;
	}
      CNV->x0_output->value(p3d.x0);
      CNV->y0_output->value(p3d.y0);
      CNV->z0_output->value(p3d.z0);
      CNV->Canvas->zoomreset();
      CNV->gstep->minimum(p3d.R/20);
      CNV->gstep->value(p3d.R/20);
      CNV->Canvas->set_gstep(CNV->gstep->value());
      CNV->show();
    }
}

void closec3cb(Fl_Button* b, void* v)
{
  MpFormC3* w=(MpFormC3*)v;

  w->hide();
}

// Callbacks bounded to the objects of the MpFormSF
void exitsfcb(Fl_Window* w, void* v)
{
  w->hide();
}

void prmsfcb(Fl_Round_Button* rb, void* v)
{
  CNV->Canvas->infos_on_off(rb->value());
}

extern char* WrtSup(struct Param3d* par,FILE* pf);

void writesfcb(Fl_Button* b, void* v)
{
  MpFormSf* Form=(MpFormSf*)v;
  int answer,errcode;
  FILE* pf;
  struct Param3d p3d;
  char *path,*errmsg;

  // Because I do not need these parameters for the next operations
  // I can set everyone to an arbitrary value.
  p3d.x0=p3d.y0=p3d.z0=0;
  p3d.R=1;
  if( (errcode=stodbl(Form->lbs_input->value(),&p3d.s1)) )
    print_error(errcode,"lb?:");
  else if( (errcode=stodbl(Form->ubs_input->value(),&p3d.s2)) )
    print_error(errcode,"ub?:");
  else if( (p3d.s2-p3d.s1<=0) || (p3d.s2-p3d.s1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub?-lb?:");
  else if( (errcode=stodbl(Form->lbt_input->value(),&p3d.t1)) )
    print_error(errcode,"lb!:");
  else if( (errcode=stodbl(Form->ubt_input->value(),&p3d.t2)) )
    print_error(errcode,"ub!:");
  else if( (p3d.t2-p3d.t1<=0) || (p3d.t2-p3d.t1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub!-lb!:");
  else if( (errcode=stoushr(Form->n1_input->value(),&p3d.nnods)) )
    print_error(errcode,"n?:");
  else if( (p3d.nnods < 2) )
    print_error(BADNN,"n?:");
  else if( (errcode=stoushr(Form->n2_input->value(),&p3d.nnodt)) )
    print_error(errcode,"n!:");
  else if( (p3d.nnodt < 2) || (p3d.nnodt*p3d.nnods > MAX_NNOD_SF) )
    print_error(BADNNSF,"n! * n?:");
  else
    {
      path=fl_file_chooser("Choose the data file","*.mpp",NULL);
      if( (path) )
	{
	  if( !(pf=open_file(path,TO_WRITE,&errcode)) )
	    {
	      switch(errcode)
		{
		case FAILED_SCANDIR:
		  fl_alert("Error in directory scansion");
		  return;
		case TYPE_OR_PERMS:
		  fl_alert("Cannot open file:\nWrong type or permissions ?");
		  return;
		case CHECK_ERROR:
		  fl_alert("Error in file checking");
		  return;
		case IDONOTLIKE_HDLNS:
		  fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
		  return;
		case FILE_EXISTS:
		  answer=fl_ask("The requested file already exists,\nDo you want overwrite it \?");
		  if( (answer) ) // answer==1==YES
		    {
		      if( !(pf=open_file(path,TO_OVERWRITE,&errcode)) )
			{
			  switch(errcode)
			    {
			    case FAILED_SCANDIR:
			      fl_alert("Error in directory scansion");
			      break;
			    case TYPE_OR_PERMS:
			      fl_alert("Cannot open file:\nWrong type or permissions ?");
			      break;
			    case CHECK_ERROR:
			      fl_alert("Error in file checking");
			      break;
			    case IDONOTLIKE_HDLNS:
			      fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
			      break;
			    default: // Another type of error has happened
			      fl_alert("Cannot open file:\n%s",strerror(errcode));
			    }
			  return;
			}
		    } // end if( (answer) )
		  else
		    return;
		  break; // If you may open *path in TO_OVERWRITE mode
		  // you arrive here
		default: // Another type of error has happened
		  fl_alert("Cannot open file:\n%s",strerror(errcode));
		  return;
		}
	    }
	  // If you arrive here pf contains the address
	  // of a file opened to be written.
	  // Every time this function is called she must insert
	  // 2 lines of parameters.
	  fprintf(pf,"% 20f\t% 20f\t% 20hu\n",p3d.s1,p3d.s2,p3d.nnods);
	  if( ferror(pf) || feof(pf) )
	    {
	      if( (fclose(pf)) )
		fl_alert("Error in output operation on file\n+ Error in closing file:\n%s",strerror(errno));
	      else
		fl_alert("Error in output operation on file");
	      return;
	    }
	  fprintf(pf,"% 20f\t% 20f\t% 20hu\n",p3d.t1,p3d.t2,p3d.nnodt); 
	  if( ferror(pf) || feof(pf) )
	    {
	      if( (fclose(pf)) )
		fl_alert("Error in output operation on file\n+ Error in closing file:\n%s",strerror(errno));
	      else
		fl_alert("Error in output operation on file");
	      return;
	    }
	  strncpy(prm[0].x,Form->x_input->value(),MAX_LUN-1);
	  prm[0].x[MAX_LUN-1]=0;
	  strncpy(prm[0].y,Form->y_input->value(),MAX_LUN-1);
	  prm[0].y[MAX_LUN-1]=0;
	  strncpy(prm[0].z,Form->z_input->value(),MAX_LUN-1);
	  prm[0].z[MAX_LUN-1]=0;
	  if( (errmsg=WrtSup(&p3d,pf)) )
	    {
	      if( (fclose(pf)) )
		fl_alert("%s\n+ Error in closing file:\n%s",errmsg,strerror(errno));
	      else
		fl_alert("%s",errmsg);
	      return;
	    }
	  if( (fclose(pf)) )
	    fl_alert("Error in closing file:\n%s",strerror(errno));
	} // end of if( (path) )
      else
	return;
    } // end of the outest else
}

extern char* DsnSup(mycanvas* cnv);

void displaysfcb(Fl_Return_Button* b, void* v)
{
  MpFormSf* Form=(MpFormSf*)v;
  int errcode;
  struct Param3d p3d;
  char* s;

  if( (errcode=stodbl(Form->x0_input->value(),&p3d.x0)) )
    print_error(errcode,"x0:");
  else if( (errcode=stodbl(Form->y0_input->value(),&p3d.y0)) )
    print_error(errcode,"y0:");
  else if( (errcode=stodbl(Form->z0_input->value(),&p3d.z0)) )
    print_error(errcode,"z0:");
  else if( (errcode=stodbl(Form->R_input->value(),&p3d.R)) )
    print_error(errcode," R:");
  else if( !good3darea(p3d.x0,p3d.y0,p3d.z0,p3d.R) )
    return;
  else if( (errcode=stodbl(Form->lbs_input->value(),&p3d.s1)) )
    print_error(errcode,"lb?:");
  else if( (errcode=stodbl(Form->ubs_input->value(),&p3d.s2)) )
    print_error(errcode,"ub?:");
  else if( (p3d.s2-p3d.s1<= 0) || (p3d.s2-p3d.s1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub?-lb?:");
  else if( (errcode=stodbl(Form->lbt_input->value(),&p3d.t1)) )
    print_error(errcode,"lb!:");
  else if( (errcode=stodbl(Form->ubt_input->value(),&p3d.t2)) )
    print_error(errcode,"ub!:");
  else if( (p3d.t2-p3d.t1<= 0) || (p3d.t2-p3d.t1 >MAX_EXTVIPRM) )
    print_error(WIV,"ub!-lb!:");
  else if( (errcode=stoushr(Form->n1_input->value(),&p3d.nnods)) )
    print_error(errcode,"n?:");
  else if( (p3d.nnods < 2) )
    print_error(BADNN,"n?:");
  else if( (errcode=stoushr(Form->n2_input->value(),&p3d.nnodt)) )
    print_error(errcode,"n!:");
  else if( (p3d.nnodt < 2) || (p3d.nnodt*p3d.nnods > MAX_NNOD_SF) )
    print_error(BADNNSF,"n! * n?:");
  else
    {
      if( (CNV->Canvas->set_par3d(&p3d)) )
	{
	  fl_alert("Error in allocation of memory\ninside mycanvas->set_par3d()");
	  return;
	}
      strncpy(prm[0].x,Form->x_input->value(),MAX_LUN-1);
      prm[0].x[MAX_LUN-1]=0;
      strncpy(prm[0].y,Form->y_input->value(),MAX_LUN-1);
      prm[0].y[MAX_LUN-1]=0;
      strncpy(prm[0].z,Form->z_input->value(),MAX_LUN-1);
      prm[0].z[MAX_LUN-1]=0;
      if( (s=DsnSup(CNV->Canvas)) )
	{
	  fl_alert("%s",s);
	  return;
	}
      CNV->x0_output->value(p3d.x0);
      CNV->y0_output->value(p3d.y0);
      CNV->z0_output->value(p3d.z0);
      CNV->Canvas->zoomreset();      
      CNV->gstep->minimum(p3d.R/20);
      CNV->gstep->value(p3d.R/20);
      CNV->Canvas->set_gstep(CNV->gstep->value());
      CNV->show();
    }
}


void closesfcb(Fl_Button* b, void* v)
{
  MpFormSf* w=(MpFormSf*)v;

  w->hide();
}

//Callbacks bounded to the objects of the MpCanvas
void exitcanvcb(Fl_Double_Window* w, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  if(MpDisplay->Canvas->get_type()==C2D)
    {
      MpDisplay->z0_output->activate();
      MpDisplay->longcount->activate();
      MpDisplay->latcount->activate();
      MpDisplay->vangcount->activate();
      MpDisplay->RUbutt->activate();
      MpDisplay->LDbutt->activate();
      MpDisplay->fogroll->activate();
    }
  w->hide();
}

void long_cb(Fl_Counter* c, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_long(c->value());
  MpDisplay->Canvas->redraw();
}

void lat_cb(Fl_Counter* c, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_lat(c->value());
  MpDisplay->Canvas->redraw();
}

void vang_cb(Fl_Counter* c, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_vang(c->value());
  MpDisplay->Canvas->redraw();
}

void right_cb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_x0(MpDisplay->Canvas->get_x0()+1);
  MpDisplay->x0_output->value(MpDisplay->Canvas->get_x0());
  MpDisplay->Canvas->redraw();
}

void left_cb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_x0(MpDisplay->Canvas->get_x0()-1);
  MpDisplay->x0_output->value(MpDisplay->Canvas->get_x0());
  MpDisplay->Canvas->redraw();
}

void up_cb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  if(MpDisplay->Canvas->get_type()==C2D)
    {
      MpDisplay->Canvas->set_y0(MpDisplay->Canvas->get_y0()+1);
      MpDisplay->y0_output->value(MpDisplay->Canvas->get_y0());
    }
  else
    {
      MpDisplay->Canvas->set_z0(MpDisplay->Canvas->get_z0()+1);
      MpDisplay->z0_output->value(MpDisplay->Canvas->get_z0());
    }
  MpDisplay->Canvas->redraw();
}

void down_cb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  if(MpDisplay->Canvas->get_type()==C2D)
    {
      MpDisplay->Canvas->set_y0(MpDisplay->Canvas->get_y0()-1);
      MpDisplay->y0_output->value(MpDisplay->Canvas->get_y0());
    }
  else
    {
      MpDisplay->Canvas->set_z0(MpDisplay->Canvas->get_z0()-1);
      MpDisplay->z0_output->value(MpDisplay->Canvas->get_z0());
    }
  MpDisplay->Canvas->redraw();
}

void rightup_cb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_y0(MpDisplay->Canvas->get_y0()+1);
  MpDisplay->y0_output->value(MpDisplay->Canvas->get_y0());
  MpDisplay->Canvas->redraw();
}

void leftdown_cb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_y0(MpDisplay->Canvas->get_y0()-1);
  MpDisplay->y0_output->value(MpDisplay->Canvas->get_y0());
  MpDisplay->Canvas->redraw();
}

void x0cb(Fl_Value_Output* vo, void*v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_x0(vo->value());
  MpDisplay->Canvas->redraw();
}

void y0cb(Fl_Value_Output* vo, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_y0(vo->value());
  MpDisplay->Canvas->redraw();
}

void z0cb(Fl_Value_Output* vo, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_z0(vo->value());
  MpDisplay->Canvas->redraw();
}

void axiscb(Fl_Round_Button* rb, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->axes_on_off(rb->value());
  MpDisplay->Canvas->redraw();
}

void gridcb(Fl_Round_Button* rb, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;
  int gridon=rb->value();

  MpDisplay->Canvas->grid_on_off(gridon);
  if( (gridon) )
    MpDisplay->gstep->activate();
  else
    MpDisplay->gstep->deactivate();
  MpDisplay->Canvas->redraw();
}

void gstepcb(Fl_Value_Output* vo, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_gstep(vo->value());
  MpDisplay->Canvas->redraw();
}

void zoomoutcb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;
  struct Param2d* p2d=MpDisplay->Canvas->get_par2d();
  struct Param3d* p3d=MpDisplay->Canvas->get_par3d();
  float M;

  MpDisplay->Canvas->zoomfactor(-1);
  if( MpDisplay->Canvas->get_type()==C2D )
    {
      M=max(p2d->h,p2d->w);
      MpDisplay->gstep->minimum(M/100);
      MpDisplay->gstep->value(M/100);
    }
  else
    {
      MpDisplay->gstep->minimum(p3d->R/20);
      MpDisplay->gstep->value(p3d->R/20);
    }
  MpDisplay->Canvas->set_gstep(MpDisplay->gstep->value());
  MpDisplay->Canvas->redraw();
}

void zoomincb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;
  struct Param2d* p2d=MpDisplay->Canvas->get_par2d();
  struct Param3d* p3d=MpDisplay->Canvas->get_par3d();
  float M;

  MpDisplay->Canvas->zoomfactor(1);
  if( MpDisplay->Canvas->get_type()==C2D )
    {
      M=max(p2d->h,p2d->w);
      MpDisplay->gstep->minimum(M/100);
      MpDisplay->gstep->value(M/100);
    }
  else
    {
      MpDisplay->gstep->minimum(p3d->R/20);
      MpDisplay->gstep->value(p3d->R/20);
    }
  MpDisplay->Canvas->set_gstep(MpDisplay->gstep->value());
  MpDisplay->Canvas->redraw();
}

void fogcb(Fl_Roller* r, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->set_fog(r->value());
  MpDisplay->Canvas->redraw();
}

void redrawcb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  MpDisplay->Canvas->redraw();
}

void exportcb(Fl_Button* b, void* v)
{
  MPRINT->show();
}

void closecb(Fl_Button* b, void* v)
{
  MpCanvas* MpDisplay=(MpCanvas*)v;

  if(MpDisplay->Canvas->get_type()==C2D)
    {
      MpDisplay->z0_output->activate();
      MpDisplay->longcount->activate();
      MpDisplay->latcount->activate();
      MpDisplay->vangcount->activate();
      MpDisplay->RUbutt->activate();
      MpDisplay->LDbutt->activate();
      MpDisplay->fogroll->activate();
    }
  MpDisplay->hide();
}

//Callbacks bounded to the objects of the MpHelp
void help_buttOk_cb(Fl_Return_Button* b, void* v)
{
  MpHelp* hlp=(MpHelp*)v;
  Fl_Browser* hb=hlp->help_browser;

  hb->clear();
  hlp->hide();
}

void exithlpcb(Fl_Window* w, void* v) //TO REVIEW
{
  MpHelp* hlp=(MpHelp*)v;
  Fl_Browser* hb=hlp->help_browser;

  hb->clear();
  w->hide();
}

//Callbacks bounded to the objects of the MpPref
void exitprefcb(Fl_Window* w, void* v)
{
  w->hide();
}

void firstchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(0,style);
}

void secondchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(1,style);
}

void thirdchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(2,style);
}

void fourthchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(3,style);
}

void fifthchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(4,style);
}

void sixthchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(5,style);
}

void seventhchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(6,style);
}

void eighthchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(7,style);
}

void ninthchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(8,style);
}

void tenthchcb(Fl_Choice* c, void* v)
{
  unsigned char style=(unsigned char)c->value();

  CNV->Canvas->set_plotstyle(9,style);
}

void quitprefcb(Fl_Return_Button* b, void* v)
{
  MpPref* prf=(MpPref*)v;

  prf->hide();
}

//Callbacks bounded to the objects of MPRINT
void exitplot_cb(Fl_Window* w, void* v)
{
  w->hide();
}

void epsbutt_cb(Fl_Check_Button* b, void* v)
{
  MpPrint* wp=(MpPrint*)v; 

  CNV->Canvas->encflag(1);
  wp->mediumch->deactivate();
  wp->orientch->deactivate();
}

void psbutt_cb(Fl_Check_Button* b, void* v)
{
  MpPrint* wp=(MpPrint*)v; 

  CNV->Canvas->encflag(0);
  wp->mediumch->activate();
  wp->orientch->activate();
}

void rgbbutt_cb(Fl_Check_Button* b, void* v)
{
  CNV->Canvas->colorprint(1);
}

void graybutt_cb(Fl_Check_Button* b, void* v)
{
  CNV->Canvas->colorprint(0);
}

void medium_cb(Fl_Choice* c, void* v)
{
  switch( c->value() )
    {
    case 0:
      CNV->Canvas->setmedium(PS_A3);
      break;
    case 1:
      CNV->Canvas->setmedium(PS_A4);
      break;
    case 2:
      CNV->Canvas->setmedium(PS_B4);
      break;
    case 3:
      CNV->Canvas->setmedium(PS_LETTER);
      break;
    case 4:
      CNV->Canvas->setmedium(PS_LEGAL);
    }
}

void orient_cb(Fl_Choice* c, void* v)
{
  CNV->Canvas->setorient((int)c->value()+1);
}

// A simple utility function
static char* file_name(const char* path)
{
  char* psz;
  static char fname[256];

  if(!path || !*path)
    return NULL;
  else if( !(psz=strrchr(path,'.')) )
    {
      strncpy(fname,path,255);
      fname[255]='\0';
    }
  else
    {
      if( psz==path )
	return NULL;
      else
	{
	  strncpy(fname,path,psz-path);
	  fname[psz-path]='\0';
	}
    }
  return fname;
}
// *************************

void exp_okbutt_cb(Fl_Return_Button* b, void* v)
{
  const char* path;
  char *name;
  int answer,errcode;

  if( !(path=fl_file_chooser("File Name","*.{eps,ps}",NULL)) )
    return;
  name=file_name(path);
  if( (name) )
    {
      if( (errcode=CNV->Canvas->print(name,TO_WRITE)) )
	{
	  switch(errcode)
	    {
	    case WRITE_ERROR:
	      fl_alert("Error in writing document");
	      return;
	    case FAILED_SCANDIR:
	      fl_alert("Error in directory scansion");
	      return;
	    case TYPE_OR_PERMS:
	      fl_alert("Cannot open file:\nWrong type or permissions ?");
	      return;
	    case CHECK_ERROR:
	      fl_alert("Error in file checking");
	      return;
	    case IDONOTLIKE_HDLNS:
	      fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
	      return;
	    case FILE_EXISTS:
	      answer=fl_ask("The requested file already exists,\nDo you want overwrite it \?");
	      if( (answer) ) // answer==1==YES
		{
		  if( (errcode=CNV->Canvas->print(name,TO_OVERWRITE)) )
		    {
		      switch(errcode)
			{
			case WRITE_ERROR:
			  fl_alert("Error in writing document");
			  return;
			case FAILED_SCANDIR:
			  fl_alert("Error in directory scansion");
			  break;
			case TYPE_OR_PERMS:
			  fl_alert("Cannot open file:\nWrong type or permissions ?");
			  break;
			case CHECK_ERROR:
			  fl_alert("Error in file checking");
			  break;
			case IDONOTLIKE_HDLNS:
			  fl_alert("Some hard links have been made with the file you\n want to open, something strange there could be\n and it is better if you do some control (see \'man ln\' or \'info ln\')");
			  break;
			default: // Another type of error has happened
			  fl_alert("Cannot open file:\n%s",strerror(errcode));
			}
		      return;
		    }
		} // end if((answer))
	      else
		return;
	      break; // If you may open *name in TO_OVERWRITE mode
	      // you arrive here
	    default: // Another type of error has happened
	      fl_alert("Cannot open file:\n%s",strerror(errcode));
	      return;
	    } //end of  switch(errcode) 
	} // end of  if( (errcode=CNV->Canvas->print(...)) )
    } // end of  if( (name) )
}

void cancbutt_cb(Fl_Button* b, void* v)
{
  MpPrint* wp=(MpPrint*)v;

  wp->hide();
} 
