/*
g2jd.c Release 0.1.0  Gregorian to Julian Day Conversion
Copyright (C) 2004  dondalah@ripco.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This program is the outer routine for converting */
/* a calendar date to the scientific Julian Day. */
/* The calendar date is based on the Julian calendar, */
/* if prior to October 15, 1582.  If later than */
/* October 4, 1582, the date is based on the */
/* Gregorian calendar. */

/* In datagen, daterange is defined by a lower */
/* and upper Julian Day. This routine will help */
/* you to obtain the two Julian Days required by */
/* daterange. */

/* The subroutine called by this program, */
/* greg2jd(), is based on Peter Duffett-Smith */
/* Practical Astronomy with Your Calculator */
/* Third Edition */
/* Cambridge University Press */
/* ISBN 0 521 35629 6 hardback */
/* ISBN 0 521 35699 7 paperback */
/* Chapter 4: Julian Day Numbers */

/* In this implementation, */
/* dates prior to the Christian Era are true */
/* Julian dates.  Year 0 is treated as an error. */
/* Step 4 in the book has been modified */
/* to allow for true dates. */
/* December 31, -1 is immediatedly followed by */
/* January 1, 1. */
/* October 4, 1582 is immediatedly followed by */
/* October 15, 1582. */

/* Notice that the integer routine described */
/* in the book has been implemented as astrint() */
/* astrint(-3.914) = -3.0 */
/* astrint() gives the same result as the integer */
/* result from the modf() function. */

#include "datagen.h"

void g2jd_stx(char *pgm)
   {
   fprintf(stderr,"Usage: %s year month "
      "day hour\n", pgm);
   fprintf(stderr,"Where year  is -4713 to 9999\n");
   fprintf(stderr,"      month is 1 to 12\n");
   fprintf(stderr,"      day is 1 to 31 depending on month\n");
   fprintf(stderr,"      hour is 0.0 to 24.0\n");
   fprintf(stderr,"Dates prior to Oct 15, 1582\n");
   fprintf(stderr,"are based on the Julian calendar.\n");
   exit(1);
   } /* g2jd_stx */

int main(int argc,char **argv)
   {
   int year,month,day,lpyr;
   int sign;
   char *p;
   double hour;
   double frac;
   double jd;
   int monthtbl[13] = {0, 31, 28, 31, 30, 31, 30,
      31, 31, 30, 31, 30, 31};
   if (argc != 5) g2jd_stx(*argv);
   year = atoi(*(argv+1));
   if (year == 0 || year < -4713 || year > 9999)
      {
      fprintf(stderr,"Invalid year %s\n", *(argv+1));
      g2jd_stx(*argv);
      }
   month = atoi(*(argv+2));
   if (month < 1 || month > 12)
      {
      fprintf(stderr,"Invalid month %s\n", *(argv+2));
      g2jd_stx(*argv);
      }
   day = atoi(*(argv+3));
   if (year % 400 == 0) lpyr = 1;
   else if (year % 100 == 0) lpyr = 0;
   else if (year % 4 == 0) lpyr = 1;
   else lpyr = 0;
   if (lpyr) monthtbl[2] = 29;
   if (day < 1 || day > monthtbl[month])
      {
      fprintf(stderr,"Invalid day %s\n", *(argv+3));
      g2jd_stx(*argv);
      }
   /* code the atof logic */
   /* the compiler gives an error on atof */
   p = (char *) *(argv+4);
   sign = 0;
   if (*p == '-')
      {
      sign = 1;
      p++;
      } /* if negative */
   hour = frac = 0.0;
   while (*p)
      {
      if (*p == '.') break;
      if (*p >= '0' && *p <= '9')
         hour = (hour * 10.0) + (double) (*p - '0');
      else
         {
         fprintf(stderr,"g2jd: invalid char %02x "
            "in hour\n", *p);
         exit(1);
         } /* illegal character */
      p++;
      } /* for each integer digit in hour */
   if (*p == '.')
      {
      p++;
      while (*p)
         {
         if (*p >= '0' && *p <= '9')
            frac = (frac + (double) (*p - '0')) * 0.1;
         else
            {
            fprintf(stderr,"g2jd: invalid decimal "
               " char %02x in hour\n", *p);
            exit(1);
            } /* illegal character */
         p++;
         } /* for each decimal digit in hour */
      hour += frac;
      } /* if hour decimal exists */
   if (sign) hour = - hour;
   /* hour = (double) atof(p); */
   if (hour < 0.0 || hour > 24.0)
      {
      fprintf(stderr,"Invalid hour %s\n", *(argv+4));
      g2jd_stx(*argv);
      }
   if (year == 1582)
      {
      if (month == 10)
         {
         if (day > 4 && day < 15)
            {
            fprintf(stderr,"Invalid date %s %s %s\n",
               *(argv+1), *(argv+2), *(argv+3));
            g2jd_stx(*argv);
            } /* if between Julian and Gregorian */
         } /* if Oct 1582 */
      } /* if 1582 */
   jd = greg2jd(year,month,day,hour);
   printf("%20.10f\n", jd);
   return(0);
   } /* main */
