/******************************************************************************
 * program:     wp2latex                                                      *
 * function:    convert WordPerfect 5.0 or 5.1 files into LaTeX               *
 * modul:       pass1.cc                                                      *
 * description: This modul contains functions for first pass. In the first    *
 *              pass information of the WP binary file will splitted in two   *
 *              parts:                                                        *
 *              1) A text file containing the whole text. The special WP      *
 *                 characters are translated.                                 *
 *              2) A binary file which contains information about             *
 *                 environments, tabbings, line endings                       *
 ******************************************************************************/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <ctype.h>

#include <sets.h>
#include <lists.h>
#include <strings.h>
#include "wp2latex.h"

#ifdef __UNIX__
 #include <unistd.h>
#endif
#ifdef __BORLANDC__
 #include "cp_lib/cptran.h"
#else
 #include "cptran.h"
#endif


extern list UserLists;

set filter[4];
int set0[3]={0 ___ 0xFF};		//Everything
int set1[16]={0 ___ 0xB,0xD ___ 0xBF,0xC0,0xC1,0xC3,0xC4,0xD0,0xD1,0xD2,0xD4 ___ 0xFF}; //Header, Footer, Minipage Text
int set2[6]={0x20 ___ 0x7F,0xA0,0xA9,0xC0};  //Characters Only


/*(3,8,10,2,2,4,5,6,0,0,0,0,0,0,0,0);*/
static unsigned char SizesC0[0x10] = {
  4, 9, 11, 3, 3, 5, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0
  };

typedef struct
	{
	WORD   PacketType;
	DWORD  Length;
	DWORD  DataPointer;
	} NestedPacket;


typedef class TconvertedPass1_WP5:public TconvertedPass1
     {
public:CpTranslator *ConvertWP5;
     SBYTE DefColumns;
     NestedPacket Images;
     DWORD FontNames;		//offset of packet with font names
     DWORD FontList;
     };


void ProcessKey(TconvertedPass1_WP5 *cq);

#ifndef FINAL
void CrackObject(TconvertedPass1 *cq, DWORD end_of_code);
#endif

typedef struct
	{
	WORD  HeadPacketType;
	WORD  NoIndexes;
	WORD  IndexBlkSize;
	DWORD NextBlock;
	} Resource5;


void FinishSummaryItem(TconvertedPass1_WP5 *cq,int ItemNo)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#FinishSummaryItem() ");fflush(cq->log);
#endif

switch(ItemNo)
	{
	case 0:
	case 1:fprintf(cq->strip,"}");		break;
	case 2:break;
	case 3:break;
	case 4:fprintf(cq->strip,"}");		break;
	case 5:break;
	case 6:NewLine(cq);
	       fprintf(cq->strip,"\\end{abstract}");
	       break;
	}
NewLine(cq);
}

/*This procedure reads a document summary packet*/
static void DocumentSummary(TconvertedPass1_WP5 *cq,DWORD length)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#DocumentSummary(%lu) ",(unsigned long)length);fflush(cq->log);
#endif
DWORD SummaryEnd,SummaryStart;
int SummaryItem=0;
unsigned SummaryChars;
unsigned TotalSummary=0;


SummaryStart=cq->ActualPos;
SummaryEnd=cq->ActualPos+length;
//cq->ActualPos+=26;
fseek(cq->wpd,cq->ActualPos,SEEK_SET);

cq->flag = HeaderText;
InitAttr(cq->attr);
SummaryChars=0;


while(cq->ActualPos<SummaryEnd)
   {
   if(fread(&cq->by, 1, 1, cq->wpd) !=1 ) break; /*Error during read*/

   if(cq->ActualPos>=SummaryStart+26 && SummaryItem==0)	//End of first item
	{
	if(SummaryChars>0) FinishSummaryItem(cq,SummaryItem);
	SummaryItem=1;
	SummaryChars=0;
	}
   if(cq->ActualPos>=SummaryStart+26+68 && SummaryItem==1)//End of second item
	{
	if(SummaryChars>0) FinishSummaryItem(cq,SummaryItem);
	SummaryItem++;
	SummaryChars=0;
	}
   if(cq->by == 0)			//End of null terminated items
	{
	if(SummaryChars>0) FinishSummaryItem(cq,SummaryItem);
	SummaryItem++;
	SummaryChars=0;
	cq->ActualPos++;
	continue;
	}
   if(cq->by==0xFF) break;		//End of all items in summary


   if(SummaryChars==0)
	  {			// Rule out return before any character
	  if((cq->by==0x0A)||(cq->by==0x0D)) cq->by=' ';
	  }
   if(cq->by==0x0A) cq->by=0x0D; // Replace [HRt] by [SRt]

   if(isalnum(cq->by) || cq->by==0xC0)
	{
	if(SummaryChars==0)
	   switch(SummaryItem)
		{
		case 0:fprintf(cq->strip,"\\date{");
		       TotalSummary|=4;				break;
		case 1:fprintf(cq->strip,"\\title{");
		       TotalSummary|=4;                         break;
		case 2:fprintf(cq->strip,"%%Type: ");
		       TotalSummary|=1;				break;
		case 3:fprintf(cq->strip,"%%Subject: ");
		       TotalSummary|=1;				break;
		case 4:fprintf(cq->strip,"\\author{");
		       TotalSummary|=4;				break;
		case 5:fprintf(cq->strip,"%%Typist: ");
		       TotalSummary|=1;				break;
		case 6:if(TotalSummary>=4 && TotalSummary<8)
				{
				NewLine(cq);
				fprintf(cq->strip,"\\maketitle");
				NewLine(cq);
				TotalSummary|=8;
				}
		       fprintf(cq->strip,"\\begin{abstract}");
		       TotalSummary|=2;
		       NewLine(cq);
		       break;
		case 7:fprintf(cq->strip,"%%Account: ");
		       TotalSummary|=1;				break;
		case 8:fprintf(cq->strip,"%%Key Words: ");
		       TotalSummary|=1;				break;
		}
	SummaryChars++;
	cq->flag = HeaderText;
	}
   else {
	if(SummaryChars==0) cq->flag = Nothing;
	}
   ProcessKey(cq);
   }
if(SummaryChars>0) FinishSummaryItem(cq,SummaryItem);

if(TotalSummary>=4 && TotalSummary<8)
	{
	NewLine(cq);
	fprintf(cq->strip,"\\maketitle");
	NewLine(cq);
	}
if(TotalSummary>0) NewLine(cq);
}

static void WalkResources(TconvertedPass1_WP5 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#WalkResources() ");fflush(cq->log);
#endif
DWORD pos1;
Resource5 res;
NestedPacket packet;
int i;
int BlockCounter=0;

pos1=16;
while(pos1>=16 && pos1<=cq->DocumentStart)
	{
	BlockCounter++;
	fseek(cq->wpd,pos1,SEEK_SET);
	if(feof(cq->wpd)) return;

	Rd_word(cq->wpd, &res.HeadPacketType);
	if(res.HeadPacketType==0) break;		//End of prefix

	Rd_word(cq->wpd, &res.NoIndexes);
	Rd_word(cq->wpd, &res.IndexBlkSize);
	Rd_dword(cq->wpd, &res.NextBlock);

	if( (res.NextBlock<=pos1 && res.NextBlock!=0) || res.NextBlock>=cq->DocumentStart)
	   {
	   fprintf(cq->err,_("\nError: WP preffix area is corrupted (block #%d), attempting to ignore!"),BlockCounter);
	   return;
	   }

	if( res.IndexBlkSize != 10*res.NoIndexes)
	   {
	   fprintf(cq->err,_("\nWarning: Suspicious size of packet descriptors, skipping packet block %d!"),BlockCounter);
	   goto NextBlock;
	   }

	if(res.HeadPacketType==0xFFFB)	//valid block
	   {
	   for(i=0;i<res.NoIndexes;i++)
	      {
	      Rd_word(cq->wpd,  &packet.PacketType);
	      Rd_dword(cq->wpd, &packet.Length);
	      Rd_dword(cq->wpd, &packet.DataPointer);

	      if((packet.DataPointer>=res.NextBlock && res.NextBlock!=0) ||
		 (packet.DataPointer<=pos1 && pos1>=16) )
		 {
		 if(packet.PacketType==0 || packet.PacketType>=256 || packet.DataPointer==0xFFFFFFFF ||
		    packet.Length==0 ) continue;
		 fprintf(cq->err,_("\nWarning: Packet #%d inside block #%d has an invalid data pointer!"),i,BlockCounter);
		 continue;
		 }

	      switch(packet.PacketType)
		   {
		   case 1:pos1 = ftell(cq->wpd);
			  cq->ActualPos = packet.DataPointer;
			  fseek(cq->wpd,packet.DataPointer,SEEK_SET);
			  DocumentSummary(cq,packet.Length);
			  fseek(cq->wpd,pos1,SEEK_SET);
			  break;
		   case 7:cq->FontNames=packet.DataPointer;	//Font Name Pool
			  break;
		   case 8:cq->ResourceStart=packet.DataPointer; //Graphics Information
			  cq->Images=packet;
			  break;
		   case 15:cq->FontList=packet.DataPointer;	//List of Fonts
			  break;
		   }
	      }

	   }
NextBlock:
	pos1=res.NextBlock;
	}

}


DWORD SelectImageResource(TconvertedPass1 *cq, int ResourceNo)
{
WORD Resources;
int i;
DWORD StartResource,ResourceSize;

StartResource=cq->ResourceStart;
if(cq->ResourceStart==0) goto ResErr;

fseek(cq->wpd,cq->ResourceStart,SEEK_SET);
Rd_word(cq->wpd, &Resources);
if(ResourceNo >= Resources) goto ResErr;

StartResource+=(DWORD)4 * Resources + 2l;

Rd_dword(cq->wpd, &ResourceSize);
i=ResourceNo;
while(i>0)		//walk through a table of resource lengths
	{
	StartResource+=ResourceSize;
	i--;
	Rd_dword(cq->wpd, &ResourceSize);
	}

if(ResourceNo+1==Resources) // The size of last resource must be computed
	{		    // because it is not written in the table
	ResourceSize=cq->DocumentStart-StartResource-1;
	}

if(StartResource+ResourceSize>=cq->DocumentStart)
	{
ResErr:	cq->perc.Hide();
	fprintf(cq->err, _("\nError: Resource #%d cannot be read!"),(int)ResourceNo);
	fflush(stdout);
	return(0);
	}
fseek(cq->wpd,StartResource,SEEK_SET);
return(ResourceSize);
}


void DoCaption5(TconvertedPass1 *_cq,unsigned short CaptionSize)
{
#ifdef DEBUG
  fprintf(_cq->log,"\n#DoCaption5() ");fflush(_cq->log);
#endif

  TconvertedPass1_WP5 *cq;
  DWORD end_of_code;
  unsigned char OldFlag;
  char OldEnvir;
  attribute OldAttr;
  int CaptionChars;

  if (CaptionSize == 0)
    return;

  cq = (TconvertedPass1_WP5 *)_cq;
  cq->ActualPos = ftell(cq->wpd);
  end_of_code = cq->ActualPos + CaptionSize;
  OldFlag = cq->flag;
  OldEnvir = cq->envir;
  cq->recursion++;

  Close_All_Attr(cq->attr,cq->strip);
  OldAttr = cq->attr;

  cq->flag = HeaderText;
  InitAttr(cq->attr);
  CaptionChars=0;


  fprintf(cq->strip, "\\caption{");
  while (cq->ActualPos < end_of_code)
	{
	if(fread(&cq->by, 1, 1, cq->wpd) !=1 ) break; /*Error during read*/

	if(CaptionChars==0)
	  {			// Rule out return before any character
	  if((cq->by==0x0A)||(cq->by==0x0D)) cq->by=' ';
	  }
	if(cq->by==0x0A) cq->by=0x0D; // Replace [HRt] by [SRt]
	if(cq->by==0xD7 && cq->subby==8)
		{
		fread(&cq->subby, 1, 1, cq->wpd);
		fseek(cq->wpd,-1,SEEK_CUR);
		if(cq->subby==8)	//Ignore \label{} in caption
			{
			cq->flag = Nothing;
			ProcessKey(cq);
			cq->flag = HeaderText;
			continue;
			}
		}

	if(isalnum(cq->by)) CaptionChars++;

	ProcessKey(cq);
	}
  Close_All_Attr(cq->attr,cq->strip);
  fprintf(cq->strip, "}\n");

  cq->line_term = 's';   /* Soft return */
  Make_tableentry_envir_extra_end(cq);

  cq->recursion--;
  cq->flag = OldFlag;
  cq->envir = OldEnvir;
  cq->attr = OldAttr;
  cq->char_on_line = false;
  cq->nomore_valid_tabs = false;
  cq->rownum++;
  Make_tableentry_attr(cq);
  cq->latex_tabpos = 0;
}

void ExtractCaptionLabel5(TconvertedPass1 *_cq,const TBox & Box)
{
#ifdef DEBUG
  fprintf(_cq->log,"\n#ExtractCaptionLabel6() ");fflush(_cq->log);
#endif

  TconvertedPass1_WP5 *cq;
  DWORD end_of_code;
  unsigned char OldFlag;
  boolean FoundLabel=false;

  if (Box.CaptionSize == 0) return;

  cq = (TconvertedPass1_WP5 *)_cq;
  fseek(cq->wpd, Box.CaptionPos, SEEK_SET);
  cq->ActualPos = Box.CaptionPos;
  end_of_code = cq->ActualPos + Box.CaptionSize;
  OldFlag = cq->flag;
  cq->recursion++;

  cq->flag = Nothing;
  InitAttr(cq->attr);


  while (cq->ActualPos < end_of_code)
	{
	if(fread(&cq->by, 1, 1, cq->wpd) !=1 ) break; /*Error during read*/

	if(cq->by==0xD7)
		{
		if(fread(&cq->subby, 1, 1, cq->wpd)!=1) break;
		fseek(cq->wpd,-1,SEEK_CUR);
		if(cq->subby==8)	//convert \label{} only
			{
			cq->flag = HeaderText;
			ProcessKey(cq);
			NewLine(cq);
			FoundLabel=true;
			break;
			}
		}
	ProcessKey(cq);
	}

  cq->recursion--;
  cq->flag = OldFlag;
}


static void ReadFormulaStr(TconvertedPass1_WP5 *cq, string & StrBeg, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#ReadFormulaStr() ");fflush(cq->log);
#endif

BYTE chr_set, chr_code;
string StrEnd;

while (cq->ActualPos < end_of_code)
      {
      fread(&cq->by, 1, 1, cq->wpd);
      if (cq->by <= '\005' || cq->by == 0xEA) break;

      if (cq->by == 169) cq->by = '-';
      if (cq->by == '\n' || cq->by == '\015' || cq->by == '&' || cq->by == '~' ||
	  cq->by == '(' || cq->by == ')' || cq->by == '}' || cq->by == '{' ||
	  cq->by == '|' ||
	  (cq->by >= ' ' && cq->by <= 'z'))
	     {
	     if (cq->by == 10 || cq->by == 13) cq->by = ' ';
	     if (cq->by == '$') StrBeg += '\\';	 // $ is transformed to \$

	     StrBeg += char(cq->by);

	     cq->ActualPos++;
	     continue;
	     }

      if (cq->by == 0xc0)       /*extended character*/
	    {			/*why special wp characters are in equations?*/
	    fread(&chr_code, 1, 1, cq->wpd);
	    fread(&chr_set, 1, 1, cq->wpd);

            StrEnd = Ext_chr_str(chr_set, chr_code, cq);

	    fread(&cq->by, 1, 1, cq->wpd);
            if (StrEnd[0] == '$') StrEnd=copy(StrEnd,1,length(StrEnd)-2)+' ';
                             else {
                             	  if(chr_set==10) StrEnd="{\\cyr "+StrEnd+"}";
				             else StrEnd="{\\rm "+StrEnd+"}";
				  }
	    StrEnd=replacesubstring(StrEnd,"\\'","\2acute ");
	    StrEnd=replacesubstring(StrEnd,"\\=","\2bar ");
	    StrEnd=replacesubstring(StrEnd,"\\u{","\2breve {");
	    StrEnd=replacesubstring(StrEnd,"\\.","\2dot ");
	    StrEnd=replacesubstring(StrEnd,"\\`","\2grave ");
	    StrEnd=replacesubstring(StrEnd,"\\^","\2hat ");
	    StrEnd=replacesubstring(StrEnd,"\\~","\2tilde ");

            StrBeg+=StrEnd;
	    StrEnd="";
            cq->ActualPos+=3;
            continue;
	    }

      ProcessKey(cq);
      }
}

char *BoxNames[6]={"Figure","Table Box","Text Box","Usr Box","Equation","!Box?"};
void AnyBox(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#AnyBox() ");fflush(cq->log);
#endif
  char OldEnvir;
  string StrBeg,ImageName;

  WORD EquLen;
  BYTE OldFlag;
  TBox Box;
  attribute OldAttr;
  int i;
  char ch;
  boolean AllowCaption;
  char BoxType;

//  CrackObject(cq, end_of_code); //

  BoxType=cq->subby;
  if(BoxType>5) {		/* Unknown Box */
		strcpy(cq->ObjType, "!Box");
		return;
		}
  if(BoxType==4) FormulaNo++;
  Box.Type=BoxType;

  cq->ActualPos = ftell(cq->wpd);
  fseek(cq->wpd, cq->ActualPos+2L, SEEK_SET);
  fread(&Box.AnchorType, 1, 1, cq->wpd);
  Box.AnchorType &= 0x3;   /*0-Paragraph, 1-Page, 2-Character*/
  fread(&Box.HorizontalPos, 1, 1, cq->wpd);
  Box.HorizontalPos&=3;	   /*0-Left, 1-Right, 2-Center, 3-Full */
  Rd_word(cq->wpd,&Box.Width);


  fseek(cq->wpd, cq->ActualPos+48L, SEEK_SET);
  fread(&Box.Contents, 1, 1, cq->wpd);


  fread(&ch, 1, 1, cq->wpd);
  while(ch>=' ')
  	{
        ImageName+=ch;
        fread(&ch, 1, 1, cq->wpd);
	}
  ImageName=cutspaces(ImageName);


  				/* Any of section's attr cannot be opened */
  for(i=First_com_section;i<=Last_com_section;i++)
		     AttrOff(cq,i);

  OldEnvir = cq->envir;
  OldFlag = cq->flag;
  OldAttr = cq->attr;
  cq->recursion++;


  switch(Box.Contents)
    {
    case 0:sprintf(cq->ObjType,"%s:Empty",BoxNames[BoxType]);
	   goto UnknownEquType;
    case 0x08:Box.Contents = 4;	  //Content equation
              fseek(cq->wpd, cq->ActualPos+115L, SEEK_SET);
	      Rd_word(cq->wpd, &Box.CaptionSize);   /*total length of caption*/
	      Box.CaptionPos = ftell(cq->wpd);

              fseek(cq->wpd, Box.CaptionSize + 10L, SEEK_CUR);
              Rd_word(cq->wpd, &EquLen);   /*total length of equation*/
              StrBeg = "";

	      cq->flag = Nothing;
	      end_of_code -= 4;
              cq->ActualPos = ftell(cq->wpd);

              if (end_of_code > cq->ActualPos + EquLen)
                          end_of_code = cq->ActualPos + EquLen;

	      ReadFormulaStr(cq, StrBeg, end_of_code);

	      InitAttr(cq->attr);
	      OptimizeFormulaStr(cq, StrBeg);
	      if(StrBeg=="") goto LEqEmpty;

	      cq->attr = OldAttr;

	      PutFormula(cq,StrBeg(),Box,5);

  /*	     if(CaptionLen>0) then
			  begin
			  seek(cq.wpd^,CaptionPos);
			  DoCaption(cq,CaptionLen);
			  end;*/
              break;

    case 0x10:Box.Contents = 1;		//Content text
	      fseek(cq->wpd, cq->ActualPos+115L, SEEK_SET);
	      Rd_word(cq->wpd, &Box.CaptionSize);   /*total length of caption*/
	      Box.CaptionPos = ftell(cq->wpd);

	      fseek(cq->wpd, Box.CaptionSize, SEEK_CUR);
              cq->ActualPos = ftell(cq->wpd);
	      end_of_code -= 4;

                     
	      if(Box.HorizontalPos!=3)
                 if(Box.Width<=1 && cq->err != NULL)
                 	  fprintf(cq->err, _("\nWarning: The width of the Box is %d, are you sure?"),Box.Width);


	      if(cq->char_on_line == -20)    /* Left one enpty line for new enviroment */
		 {
		 fputc('%', cq->strip);
		 NewLine(cq);
		 cq->char_on_line = true;
		 }
	      if(cq->char_on_line==true)     /* make new line for leader of minipage */
		   {
                   NewLine(cq);
		   }

	      AllowCaption=BoxTexHeader(cq,Box);
	      cq->envir='!';			//Ignore enviroments Before
	      NewLine(cq);

              cq->char_on_line = -10;
              cq->envir=' ';
	      while (cq->ActualPos < end_of_code)
		       {
		       fread(&cq->by, 1, 1, cq->wpd);
               
                       ProcessKey(cq);
                       }
               
	      Close_All_Attr(cq->attr,cq->strip);

              cq->envir=' ';
              if (cq->char_on_line==true)
                   {
                   NewLine(cq);
		   }
              if(AllowCaption)
                      {
                      fseek(cq->wpd, Box.CaptionPos, SEEK_SET);
		      DoCaption5(cq, Box.CaptionSize);
                      }
	      BoxTexFoot(cq, Box);

              cq->envir='^';		//Ignore enviroments after minipage
	      NewLine(cq);
	      break;
    case 0x80:
    case 0x83:			//Image in resource
    case 0x02:                  //Image on disk
	      end_of_code -= 4;
	      Box.Contents = 3; // content image
	      Box.CaptionPos=cq->ActualPos+133L;
	      Box.CaptionSize=
		 (end_of_code<Box.CaptionPos)?0:(WORD)(end_of_code-Box.CaptionPos);
	      Image(cq,ImageName,Box,DoCaption5,5);
	      break;
    default:sprintf(cq->ObjType,"!%s",BoxNames[BoxType]);
	    goto UnknownEquType;

    }


LEqEmpty:
  strcpy(cq->ObjType, BoxNames[BoxType]);
UnknownEquType:
  cq->recursion--;
  if(cq->envir=='^') cq->char_on_line = -10;		// stronger false;
  cq->flag = OldFlag;
  cq->envir = OldEnvir;
  cq->attr = OldAttr;
}


/* This procedure expands extended WP5.x characters into LaTeX string */
static void ExtendedCharacter(TconvertedPass1_WP5 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#ExtendedCharacter() ");fflush(cq->log);
#endif
  unsigned char chr_code, chr_set;
  WORD WChar;
  char *ch;

  fread(&chr_code, 1, 1, cq->wpd);
  fread(&chr_set, 1, 1, cq->wpd);
  sprintf(cq->ObjType, "%d,%d",chr_set,chr_code);
  if(cq->ConvertWP5!=NULL)	/*Translate other WP5.x character sets (Brrr)*/
     {
     WChar=(*cq->ConvertWP5)[256*chr_set + chr_code];
     if(WChar>0)
	{
	chr_set = WChar / 256;
	chr_code= WChar % 256;
	}
     }
  ch=Ext_chr_str(chr_set, chr_code, cq);
  CharacterStr(cq,ch,cq->Font);
}


#define L1_5cm          709   /*length of 1.5 cm*/
#define L1_27cm         600   /*length of 1.27 cm*/

void WP_Default(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#WP_Default() ");fflush(cq->log);
#endif
  long j;

  cq->Font = 0;
  cq->FirstSection = 120;
  cq->recursion = 0;
  cq->flag = NormalText;

  cq->latex_tabpos = 0;
  cq->tab_type   = 0;
  cq->CentrePage = false;

  cq->indenting = false;
  cq->indent_end = false;
  cq->ind_text1 = false;
  cq->char_on_line = false;              //!!!!!!
  cq->ind_leftmargin = 0;
  cq->ind_rightmargin = 0;
  cq->Columns = Columns;


// Default sizes of the document
  cq->WP_sidemargin = 1200;
  cq->Lmar=1890;
  cq->Rmar=1200;

// Default tab sets
  cq->tabpos[0] = L1_27cm;   /* 1e WP-tab is kantlijn --> */
  for (j = 2; j <= 10; j++)   /* Size of tabs is 1,27  cm  */
	cq->tabpos[j-1] = cq->tabpos[j-2] + L1_27cm;
  for (j = 10; j <= 39; j++)   /* ($02c5 wpu) verder        */
	cq->tabpos[j] = 0xffffL;

  cq->num_of_tabs = 10;
}

#undef L1_5cm
#undef L1_27cm


/* Internal loop for translating text for labels and references */
static void LabelInside(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
char *characterStr,str2[2];
unsigned char chr_code,chr_set;

str2[1]=0;
while (cq->ActualPos < end_of_code)
       {
       fread(&cq->by, sizeof(unsigned char), 1, cq->wpd);
       if (cq->by == 0) break;
       if (cq->by >= 0x20 && cq->by <= 0x7f)
       		{
                str2[0]=cq->by;		/* Normal_char  */
                characterStr=str2;
                cq->ActualPos++;
                goto ProcessCharacterSTR;
		}
       if (cq->by == 0xc0)       /*extended character*/
            {			 /*Special Caracters cannot be used in labels*/
            fread(&chr_code, 1, 1, cq->wpd);
            fread(&chr_set, 1, 1, cq->wpd);

            characterStr = Ext_chr_str(chr_set, chr_code, cq);

            fread(&cq->by, 1, 1, cq->wpd);
            cq->ActualPos+=3;

ProcessCharacterSTR:
            while(*characterStr!=0)
            	{
                switch(*characterStr)
                	{
                        case 0: break;
                        case '\\':fputs("_bsl", cq->strip); break;
                        case '$' :fputs("_dol", cq->strip); break;
                        case '{' :fputs("_opb", cq->strip); break;
                        case '}' :fputs("_clb", cq->strip); break;
                        case '%' :fputs("_prc", cq->strip); break;
                        case '_' :fputs("_und", cq->strip); break;
                        default:fputc(*characterStr, cq->strip); break;
                        }
                characterStr++;
                }


            continue;
	    }

       ProcessKey(cq);
       }
}

static void MakeIndex(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#MakeIndex() ");fflush(cq->log);
#endif
  unsigned char OldFlag;

  OldFlag = cq->flag;
  cq->flag = HeaderText;
  end_of_code -= 4;

  cq->ActualPos = ftell(cq->wpd);
  fprintf(cq->strip, "\\index{");
  LabelInside(cq, end_of_code);
  putc('}', cq->strip);
  Index=true;

  cq->flag = OldFlag;
  cq->char_on_line = true;
  strcpy(cq->ObjType, "Index");
}



static void MakeLabel(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#MakeLabel() ");fflush(cq->log);
#endif
  unsigned char OldFlag;

  OldFlag = cq->flag;
  cq->flag = CharsOnly;
  end_of_code -= 4;

  cq->ActualPos = ftell(cq->wpd);
  fprintf(cq->strip, "%s\\label{",cq->char_on_line>0?" ":"");
  LabelInside(cq, end_of_code);
  putc('}', cq->strip);

  cq->flag = OldFlag;
  if(cq->char_on_line == false)
	cq->char_on_line = -1;
  strcpy(cq->ObjType, "Label");
}


static void MakeRef(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#MakeRef() ");fflush(cq->log);
#endif
  unsigned char OldFlag, TypeRef;

  OldFlag = cq->flag;
  cq->flag = HeaderText;
  end_of_code -= 4;

  fread(&TypeRef, 1, 1, cq->wpd);   /*Type of reference*/

  cq->ActualPos = ftell(cq->wpd);
  if(TypeRef==0) fprintf(cq->strip, " \\pageref{");
  	    else fprintf(cq->strip, " \\ref{");
  LabelInside(cq, end_of_code);
  putc('}', cq->strip);

  cq->flag = OldFlag;
  cq->char_on_line = true;
  strcpy(cq->ObjType, "Reference");
}



//***********************************************************************//
//*** Alphabetically ordered functions for translating WP5.x features ***//
//***********************************************************************//

static void Advance(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Advance() ");fflush(cq->log);
#endif

  SWORD w;
  unsigned char b;
  char *what;

  fread(&b, sizeof(unsigned char), 1, cq->wpd);
  fseek(cq->wpd, 2L, SEEK_CUR);
  Rd_word(cq->wpd, (WORD *)&w);

  what="";
  if (b == 0)   /*Up/Down*/
  	{
        if(w>0) what="Up";
        if(w<0) what="Down";
	fprintf(cq->strip, "\\vspace{%2.2fcm}", float(w)/470.0);
        }
  if (b == 2)   /*Left/Right*/
  	{
	if(w>0) what="Right";
        if(w<0) what="Left";
    	fprintf(cq->strip, "{\\hskip %2.2fcm}", float(w)/470.0);
        }
  sprintf(cq->ObjType, "Advance %s %2.2fcm",what,float(w)/470.0);
}


static void End_Align(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#End_Align() ");fflush(cq->log);
#endif

/*TabType = 0-Left, 1-Full, 2-Center, 3-Right, 4-Aligned */
  switch(cq->tab_type)
	{
	case 4:Close_All_Attr(cq->attr,cq->strip);	//aligned
	       fputs("\\'",cq->strip);
	       cq->tab_type = 0;
	       Open_All_Attr(cq->attr,cq->strip);
	       break;

	case 3:Close_All_Attr(cq->attr,cq->strip);	//right
	       fprintf(cq->strip, "\\'");
	       cq->tab_type = 0;
	       Open_All_Attr(cq->attr,cq->strip);
	       break;

	case 2:Close_All_Attr(cq->attr,cq->strip);	//center
	       putc('}', cq->strip);
	       cq->tab_type = 0;
	       Open_All_Attr(cq->attr,cq->strip);
	       break;
	}

strcpy(cq->ObjType, "End Align");
}


static void ColDef5(TconvertedPass1_WP5 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#ColDef5() ");fflush(cq->log);
#endif

  fseek(cq->wpd, 97L, SEEK_CUR);
  fread(&cq->DefColumns, 1, 1, cq->wpd);   /* new number of columns */

  sprintf(cq->ObjType, "ColDef:%d",(int)cq->DefColumns);
  if(Columns==0 && cq->DefColumns>2) cq->DefColumns=2;
}


/*This function converts commented text to proper LaTeX comment.*/
static void Comment5(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Comment5() ");fflush(cq->log);
#endif
  boolean Old_char_on_line;
  unsigned char OldFlag;
  WORD something;
  attribute OldAttr;


  OldFlag = cq->flag;
  OldAttr = cq->attr;
  Old_char_on_line = cq->char_on_line;
  cq->flag = CharsOnly;
  cq->recursion++;
  InitAttr(cq->attr);		//Turn all attributes in the comment off
  end_of_code -= 4;

  Rd_word(cq->wpd, &something);


  fputc('%',cq->strip);
  cq->ActualPos = ftell(cq->wpd);
  while (cq->ActualPos < end_of_code)
       {
       fread(&cq->by, sizeof(unsigned char), 1, cq->wpd);
       if(cq->by==0xa || cq->by==0xd)	//New comment line
       		{
                cq->line_term = 's';    	//Soft return
		Make_tableentry_envir_extra_end(cq);
		fprintf(cq->strip, "\n");
		cq->rownum++;
		Make_tableentry_attr(cq);

                fputc('%',cq->strip);
                }

       ProcessKey(cq);
       continue;
       }

  cq->line_term = 's';    	//Soft return
  Make_tableentry_envir_extra_end(cq);
  fprintf(cq->strip, "\n");
  cq->rownum++;
  Make_tableentry_attr(cq);


  cq->recursion--;
  strcpy(cq->ObjType, "Comment");
  cq->attr = OldAttr;
  cq->flag = OldFlag;
  cq->char_on_line = Old_char_on_line;
  cq->char_on_line = false;
}


void DateCode(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#DateCode() ");fflush(cq->log);
#endif

  fprintf(cq->strip, "\\today{}");

  strcpy(cq->ObjType, "Date");
  cq->char_on_line = true;
}


/*This procedure finishes table that is beeing converted.*/
void EndTable(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#EndTable() ");fflush(cq->log);
#endif

  if (cq->envir == 'B')
	{
	Close_All_Attr(cq->attr,cq->strip);
	fprintf(cq->strip, " \\\\ \\hline\n");
        }
	else fprintf(cq->strip, " \n");

  cq->line_term = 'h';   /* Hard return */
  Make_tableentry_envir_extra_end(cq);

  if (cq->envir == 'B')
	{
        cq->envir = ' ';
    /*cq.line_term := 's';
	    writeln(cq.strip^,);
            Make_tableentry_envir_extra_end(cq);*/
        }
  cq->char_on_line = false;
  cq->nomore_valid_tabs = false;
  cq->rownum++;
  Make_tableentry_attr(cq);
  cq->latex_tabpos = 0;
  strcpy(cq->ObjType, "End Table");
}


static void Endnote(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Endnote() ");fflush(cq->log);
#endif
  attribute OldAttr;
  unsigned char OldFlag;


  end_of_code -= 4;
  OldFlag = cq->flag;
  cq->flag = HeaderText;
  cq->recursion++;

  if(!EndNotes) EndNotes=true;		/* set up endnotes */
  fseek(cq->wpd, 7, SEEK_CUR);


  Close_All_Attr(cq->attr,cq->strip);
  OldAttr=cq->attr;
  InitAttr(cq->attr);


  fprintf(cq->strip, "\\endnote{");

  cq->ActualPos = ftell(cq->wpd);
  while (cq->ActualPos < end_of_code)
  	{
        fread(&cq->by, sizeof(unsigned char), 1, cq->wpd);

        switch (cq->by)
	     {
             case 0xa:
             case 0xc:fprintf(cq->strip, "\\\\ ");
                      break;

             case 0xb:
	     case 0xd:putc(' ', cq->strip);
                      break;

             default: ProcessKey(cq);
	              break;
             }

           cq->ActualPos = ftell(cq->wpd);
	   }


  Close_All_Attr(cq->attr,cq->strip);   /* Echt nodig ? */
  putc('}', cq->strip);

  cq->attr=OldAttr;
  Open_All_Attr(cq->attr,cq->strip);

  cq->recursion--;
  strcpy(cq->ObjType, "Endnote");
  cq->flag = OldFlag;
}


void Flush_right(TconvertedPass1 *cq,char type)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Flush_right() ");fflush(cq->log);
#endif

  if(cq->envir == 'T')
      {
      Close_All_Attr(cq->attr,cq->strip);
      fprintf(cq->strip, "\\`");
      Open_All_Attr(cq->attr,cq->strip);

      cq->nomore_valid_tabs = true;

//    cq->envir = 'T';
      }
      else {
	   if(cq->char_on_line!=true) fprintf(cq->strip, "\\strut");
           cq->char_on_line = true;   /*Something is placed on line */
           if(type==1) fprintf(cq->strip, "\\dotfill ");
		  else fprintf(cq->strip, "\\hfill ");
           }

  sprintf(cq->ObjType, "Flsh Rtg%s",type==1?" (dot)":"");
}


static void Footnote(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Footnote() ");fflush(cq->log);
#endif
  attribute OldAttr;
  unsigned char flags, num_of_pages;
  WORD fn_num;
  unsigned char OldFlag;


  end_of_code -= 4;
  OldFlag = cq->flag;
  cq->flag = HeaderText;
  cq->recursion++;


  fread(&flags, sizeof(unsigned char), 1, cq->wpd);

  Rd_word(cq->wpd, &fn_num);

  fread(&num_of_pages, sizeof(unsigned char), 1, cq->wpd);   /* Skip all the shit */
  fseek(cq->wpd, (num_of_pages + 1L) * 2 + 9, SEEK_CUR);

  Close_All_Attr(cq->attr,cq->strip);
  OldAttr=cq->attr;
  InitAttr(cq->attr);


  fprintf(cq->strip, "\\footnote" );
  if(ExactFnNumbers) fprintf(cq->strip, "[%u]", fn_num);
  fputc('{',cq->strip);

  cq->ActualPos = ftell(cq->wpd);
  while (cq->ActualPos < end_of_code)
	{
	fread(&cq->by, sizeof(unsigned char), 1, cq->wpd);

	switch (cq->by)
	     {
	     case 0xa:
	     case 0xc:fprintf(cq->strip, "\\\\ ");
		      break;

	     case 0xb:
	     case 0xd:putc(' ', cq->strip);
		      break;

	     default: ProcessKey(cq);
		      break;
	     }

	   cq->ActualPos = ftell(cq->wpd);
	   }


  Close_All_Attr(cq->attr,cq->strip);   /* Echt nodig ? */
  putc('}', cq->strip);

  cq->attr=OldAttr;

  cq->recursion--;
  strcpy(cq->ObjType, "Footnote");
  cq->flag = OldFlag;
}


static void Header_Footer(TconvertedPass1_WP5 *cq,DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Header_Footer() ");fflush(cq->log);
#endif
  unsigned char occurance;
  attribute OldAttr;
  string s;
  unsigned char OldFlag,OldEnvir;
  int i;


  if (end_of_code - ftell(cq->wpd) <= 18) goto ExitHeaderFooter;

  OldFlag = cq->flag;
  OldEnvir= cq->envir;

  cq->recursion++;
  end_of_code -= 4;


  cq->line_term = 's';    	//Soft return
  if(cq->char_on_line == -20)    /* Left one enpty line for new enviroment */
      {
      fputc('%', cq->strip);
      NewLine(cq);
      }
  if(cq->char_on_line==true)
      {
      NewLine(cq);
      }


  fseek(cq->wpd, 7L, SEEK_CUR);
  fread(&occurance, 1, 1, cq->wpd);

  fseek(cq->wpd, 10L, SEEK_CUR);

  Close_All_Attr(cq->attr,cq->strip);
  OldAttr=cq->attr;			/* Backup all attributes */

  				/* Any of section's attr cannot be opened */
  for(i=First_com_section;i<=Last_com_section;i++)
		  _AttrOff(cq->attr,i,s); // !!!!!! toto predelat!

  InitHeaderFooter(cq,cq->subby,occurance);
  occurance=(occurance << 4) | (cq->subby & 0xF);


  cq->envir='!';		//Ignore enviroments after header/footer
  NewLine(cq);

  InitAttr(cq->attr);		//Turn all attributes in the header/footer off

  cq->flag = HeaderText;
  cq->envir = ' ';
  cq->ActualPos = ftell(cq->wpd);
  cq->char_on_line = -10;
  while (cq->ActualPos < end_of_code)
        {
        fread(&cq->by, 1, 1, cq->wpd);

        ProcessKey(cq);
        }

  Close_All_Attr(cq->attr,cq->strip);
  if(cq->char_on_line==true)
     {
     cq->line_term = 's';    	//Soft return
     NewLine(cq);
     }
  putc('}', cq->strip);

  cq->line_term = 's';    	//Soft return
  cq->envir='^';		//Ignore enviroments after header/footer
  NewLine(cq);

  cq->attr=OldAttr;			/* Restore backuped attributes */
  cq->flag = OldFlag;
  cq->envir = OldEnvir;
  cq->char_on_line = -10;		// stronger false;
  cq->recursion--;

ExitHeaderFooter:
  strcpy(cq->ObjType,(occurance & 3)<=1?"Header":"Footer");
}


static void HLine(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#HLine() ");fflush(cq->log);
#endif
  unsigned char AnchorType,HorizontalPos,VerticalPos,Shading;
  WORD Width,Length,Raise;

  cq->ActualPos = ftell(cq->wpd);
  fseek(cq->wpd, cq->ActualPos+2L, SEEK_SET);
  fread(&AnchorType, 1, 1, cq->wpd);
  VerticalPos = AnchorType >> 4;
  AnchorType &= 0x3;	   /*0-Paragraph, 1-Page, 2-Character*/
  fread(&HorizontalPos, 1, 1, cq->wpd);
  HorizontalPos&=3;	   /*0-Left, 1-Right, 2-Center, 3-Full */
  Rd_word(cq->wpd,&Length);
  Rd_word(cq->wpd,&Width);
  fseek(cq->wpd, 2L, SEEK_CUR);
  Rd_word(cq->wpd,&Raise);
  fseek(cq->wpd, 20L, SEEK_CUR);
  fread(&Shading, 1, 1, cq->wpd);
  if(Shading>100) Shading=100;


  fprintf(cq->strip,"\\Xvadjust{");
  if(HorizontalPos==1 || HorizontalPos==2) fprintf(cq->strip,"\\hss");
  fprintf(cq->strip,"\\rule");
  if(VerticalPos==1) fprintf(cq->strip,"[%2.2fcm]",(float)Raise/470);
  if(HorizontalPos==3) fprintf(cq->strip,"{\\textwidth}{%2.2fcm}",(float)Width/470);
		  else fprintf(cq->strip,"{%2.2fcm}{%2.2fcm}",(float)Length/470,(float)Width/470);
  if(HorizontalPos==0 || HorizontalPos==2) fprintf(cq->strip,"\\hss");
  fprintf(cq->strip,"}");

  strcpy(cq->ObjType, "HLine");
}



void Hyphenation(TconvertedPass1 *cq,char state)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Hyphenation() ");fflush(cq->log);
#endif

switch(state)
   {
   case true:fprintf(cq->strip, "\\hyphenpenalty 45 ");
	     strcpy(cq->ObjType, "Hyph On");
	     break;
   case false:fprintf(cq->strip, "\\hyphenpenalty 10000 ");
	      strcpy(cq->ObjType, "Hyph Off");
	      break;
   default:strcpy(cq->ObjType, "!Hyph ?");
   }
}


void InvisibleSoftReturn(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#InvisibleSoftReturn() ");fflush(cq->log);
#endif

  if(cq->char_on_line==true) /* Ignore misplaced ISRt */
     {
     fprintf(cq->strip, "{\\penalty0}");
     }

  strcpy(cq->ObjType, "ISRt");
}


static void LineSpacing(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#LineSpacing() ");fflush(cq->log);
#endif
WORD LastSpacing;
WORD CurrentSpacing;
char b;

  Rd_word(cq->wpd,&LastSpacing);
  Rd_word(cq->wpd,&CurrentSpacing);

  b = 'l';
  fwrite(&b, 1, 1, cq->table);
  Wr_word(cq->table,CurrentSpacing);

  sprintf(cq->ObjType, "Line Spacing %2.2f",float(CurrentSpacing)/128);
}


void MarkedTextBegin(TconvertedPass1_WP5 *cq,DWORD & NewPos)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#MarkedTextBegin() ");fflush(cq->log);
#endif
  signed char b;
  string ListName;


  fread(&b, sizeof(unsigned char), 1, cq->wpd);   /* read flags */
  if(b<0x10)	//This is a MarkToContents
	{
	StartSection(cq, -b);
	return;
	}

  sprintf(cq->ObjType, "Start MarkToL:%d",(int)(b&0xDF) + 1);
  if(!(b&0x20)) return;


  ListName=_("List ");
  ListName+=cq->ObjType+14;
  b=ListName() IN UserLists;
  if(b) b--;
   else b=UserLists+=ListName();

  fprintf(cq->strip, "\\UserList{l%c}{",b+'a');
  fseek(cq->wpd,cq->ActualPos=NewPos,SEEK_SET);
  cq->by=fgetc(cq->wpd);
  cq->recursion++;
  while(filter[CharsOnly][cq->by])
	{
	ProcessKey(cq);
	NewPos=cq->ActualPos;
	cq->by=fgetc(cq->wpd);
	}
  fputc('}',cq->strip);
  cq->recursion--;
}


/*This function converts overstriked characters.*/
static void Overstrike(TconvertedPass1_WP5 *cq, DWORD end_of_code)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Overstrike() ");fflush(cq->log);
#endif
  boolean first_char_os = true;
  WORD char_width_os;
  attribute OldAttr;
  string s;

  unsigned char OldFlag;

  OldFlag = cq->flag;
  cq->flag = CharsOnly;
  cq->recursion++;
  end_of_code -= 4;

  Rd_word(cq->wpd, &char_width_os);

  OldAttr=cq->attr;
  cq->ActualPos = ftell(cq->wpd);
  while (cq->ActualPos < end_of_code)
       {
       fread(&cq->by, 1, 1, cq->wpd);

       if (cq->by == 0xC3 || cq->by == 0xC4)
	  {
	  cq->flag = OldFlag;
	  ProcessKey(cq);
	  cq->flag = CharsOnly;
	  continue;
	  }

       if (cq->by != 0xc0 && cq->by != 0xa9 && (cq->by < 0x20 || cq->by > 0x7f))
          {
          ProcessKey(cq);
          continue;
          }

      if (first_char_os)
	 {
	 ProcessKey(cq);
         first_char_os = false;
         }
      else {
	   Open_All_Attr(cq->attr, cq->strip);
	   if(cq->attr.Math_Depth>0) cq->attr.Math_Depth=0;//Math might be nested inside \llap
           fprintf(cq->strip, "\\llap{");
           ProcessKey(cq);
           putc('}', cq->strip);
           }
      }

  AttrFit(cq->attr,OldAttr,s);
  if(s!="") fputs(s(),cq->strip);

  cq->recursion--;
  strcpy(cq->ObjType, "Overstrike");
  cq->flag = OldFlag;
  cq->char_on_line = true;
}


static void SetFont5(TconvertedPass1_WP5 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#SetFont5() ");fflush(cq->log);
#endif
WORD PointSize;
WORD NamePtr;
char *ig="!";
string FontName;
BYTE FontNo;
char ch;

  fseek(cq->wpd, 25l, SEEK_CUR);
  FontNo=fgetc(cq->wpd);

  fseek(cq->wpd, 2l, SEEK_CUR);
  Rd_word(cq->wpd,&PointSize);

  if(cq->FontList > 0 && cq->FontNames >0)
    {
    fseek(cq->wpd, cq->FontList + (long)FontNo*86 + 18, SEEK_SET);
    Rd_word(cq->wpd,&NamePtr);
    fseek(cq->wpd, cq->FontNames + NamePtr, SEEK_SET);
    ch=fgetc(cq->wpd);
    while(ch!=0)
	{
	FontName+=ch;
	ch=fgetc(cq->wpd);
	}
    }

  SetFont(cq,PointSize,FontName());
  if(NFSS) ig="";

  sprintf(cq->ObjType, "%sFont %2.2f%.20s",ig,float(PointSize)/50,FontName());
}



typedef struct WPTab {
  WORD len;
  WORD attribute;
  BYTE alignment;
} WPTab;

static void StartTable(TconvertedPass1 *cq, unsigned short len)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#StartTable() ");fflush(cq->log);
#endif
  short columns = 0;
  WORD i;
  unsigned char b;
  WPTab Atbl[100];
  string TexAttrib;

  fseek(cq->wpd, 2L, SEEK_CUR);		//OldFlag + OldShading
  fread(&b, sizeof(unsigned char), 1, cq->wpd); //OldColumns
  i = b * 5 + 48;
  fseek(cq->wpd, i - 3L, SEEK_CUR);
  if (b & 1) i += 6;
	else i++;
  columns = (len - i) / 5;
  if ((len - i) % 5 != 0)
	    return;
  if (columns > 100 || columns <= 0)
	    return;

  for (i = 0; i <= columns - 1; i++)
	Rd_word(cq->wpd, &Atbl[i].len);
  for (i = 0; i <= columns - 1; i++)
	Rd_word(cq->wpd, &Atbl[i].attribute);
  for (i = 0; i <= columns - 1; i++)
	fread(&Atbl[i].alignment, 1, 1, cq->wpd);


  cq->line_term = 's';   /* Soft return */
  if(cq->char_on_line>=-1)
	{
	NewLine(cq);
	}
  cq->envir = 'B';

  fprintf(cq->strip, "{|");
  for (i = 0; i <= columns - 1; i++)
    {
    TexAttrib = "";			//Solve attributes for columns
    if(Atbl[i].attribute &  1) TexAttrib+="\\LARGE";
    if(Atbl[i].attribute &  2) TexAttrib+="\\Large";
    if(Atbl[i].attribute &  4) TexAttrib+="\\large";
    if(Atbl[i].attribute &  8) TexAttrib+="\\small";
    if(Atbl[i].attribute & 16) TexAttrib+="\\footnotesize";
    //32 supscript
    //64 subscript
    //128 outline
    if(Atbl[i].attribute & 256)  TexAttrib+="\\it";
    if(Atbl[i].attribute & 1024 && ((LaTeX_Version & 0xFF00) >= 0x300) && (colors>=0))
			{
			TexAttrib+="\\textcolor{red}";
			colors=true;
			}
    if(Atbl[i].attribute & 4096) TexAttrib+="\\bf";
    if(Atbl[i].attribute & 32768)TexAttrib+="\\sc";

    if(TexAttrib != "")
	{
	fprintf(cq->strip,"@{%s}",TexAttrib());
	}

    switch (Atbl[i].alignment & 0xf)
      {
      case 0:fprintf(cq->strip, "l|");   /*left*/
	     break;
      case 1:fprintf(cq->strip, "c|");   /*full*/
	     break;
      case 2:fprintf(cq->strip, "c|");   /*center*/
	     break;
      case 3:fprintf(cq->strip, "r|");   /*right*/
	     break;

      default:fprintf(cq->strip, "c|");   /*I don`t know*/
	      break;
      }
  }
  putc('}', cq->strip);

  cq->char_on_line = false;
  cq->nomore_valid_tabs = false;
  cq->rownum++;
  Make_tableentry_attr(cq);
  cq->latex_tabpos = 0;
  strcpy(cq->ObjType, "Table Start");
}


void TableOfContents(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#TableOfContents() ");fflush(cq->log);
#endif
  char ListType,ListSubType,b;
  string ListName;

  if ((cq->char_on_line==true)||(cq->attr.Opened_Depth>0))
    {
    Close_All_Attr(cq->attr,cq->strip);

    NewLine(cq);
    }

  fread(&ListType, 1, 1, cq->wpd);
  switch(ListType>>4)
	{
	case  0:fprintf(cq->strip, "\\tableofcontents ");
		cq->line_term = 's';   /* Soft return */
		NewLine(cq);
        	strcpy(cq->ObjType, "Def Mark:ToC");
                break;
	case  1:PlaceIndex(cq);
		break;
	case  2:fread(&ListSubType, 1, 1, cq->wpd);
		ListType = (ListType & 0xF) + 1;
		sprintf(cq->ObjType, "%d",ListType);
		ListName=_("List ");
		ListName+=cq->ObjType;
		b=ListName() IN UserLists;
		if(b) b--;
		   else b=UserLists+=ListName();

		fprintf(cq->strip, "\\ShowUserList{l%c}{%s}",b+'a',ListName());
		cq->line_term = 's';   /* Soft return */
		NewLine(cq);
		sprintf(cq->ObjType, "Def Mark:List,%d:%d",(int)ListType,(int)ListSubType+1);
		break;
	case  3:sprintf(cq->ObjType, "!Def Mark:ToA,%d",(int)(ListType & 0xF +1));
// !!!!!!!!! The tables of autorities are not converted yet
	        break;

        default: strcpy(cq->ObjType, "Def Mark:?Unknown");
	}

}


static void Tabset(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Tabset() ");fflush(cq->log);
#endif
  int j,SideMargin;
  WORD w,Absolute;
  long pos;


  pos = ftell(cq->wpd);
  fseek(cq->wpd, pos+202L, SEEK_SET);   /* Ga naar TAB-info */
  Rd_word(cq->wpd, &Absolute);
  if(Absolute!=0xFFFF)
	{
	SideMargin=Absolute+cq->WP_sidemargin-cq->Lmar;//Relative Tab
	}
	else SideMargin=cq->WP_sidemargin;	//Absolute Tab

  fseek(cq->wpd, pos+100L, SEEK_SET);   /* Ga naar TAB-info */
  cq->num_of_tabs = 0;

  for (j = 1; j <= 40; j++)
      {
      Rd_word(cq->wpd, &w);
      if (w > cq->WP_sidemargin && w != 0xffffL)
	    {
	    cq->num_of_tabs++;
	    cq->tabpos[cq->num_of_tabs - 1] = w - SideMargin;
	    }
      }

  Make_tableentry_tabset(cq);
  sprintf(cq->ObjType, "TabSet:%s",Absolute==0xFFFFL?"Abs":"Rel");
}

static void LRMargin(TconvertedPass1 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#LRMargin() ");fflush(cq->log);
#endif
  WORD w;

  fseek(cq->wpd, 4L, SEEK_CUR);

  Rd_word(cq->wpd, &w);
  cq->Lmar=w;
  Rd_word(cq->wpd, &w);
  cq->Rmar=w;

  strcpy(cq->ObjType, "L/R Mar");
}


void WidowOrphan(TconvertedPass1 *cq,signed char type) /*type 0:off; 1:on; 6:Wp6.0*/
{
#ifdef DEBUG
  fprintf(cq->log,"\n#WidowOrphan(%d) ",(int)type);fflush(cq->log);
#endif

if(type==6)
	{
	fseek(cq->wpd,3l,SEEK_CUR);
	fread(&type,1,1,cq->wpd);
	}

switch(type)
     {
     case 0:fprintf(cq->strip, "\\clubpenalty 0 \\widowpenalty 0 ");
	    strcpy(cq->ObjType, "Wid/Orph:Off");
	    break;
     case 1:fprintf(cq->strip, "\\clubpenalty 150 \\widowpenalty 150 ");
	    strcpy(cq->ObjType, "Wid/Orph:On");
	    break;
     default:strcpy(cq->ObjType, "!Wid/Orph:?");
     }
}



////////////////////////////////////////////////////////////////
static boolean CheckConzistency(TconvertedPass1 *cq, long NewPos)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#CheckConzistency() ");fflush(cq->log);
#endif

  boolean Result = true;
  unsigned char TestedBy,TestedSubBy;
  long Pos;

  Pos = ftell(cq->wpd);

  TestedSubBy=cq->subby;
  if (cq->by >= 0xd0)			//subby will be also tested
  		 {
                 fseek(cq->wpd, NewPos-2 , 0);
		 fread(&TestedSubBy, 1, 1, cq->wpd);
		 }
  	    else fseek(cq->wpd, NewPos-1 , 0);
  fread(&TestedBy, 1, 1, cq->wpd);
  if ((TestedBy != cq->by)||(TestedSubBy != cq->subby))
  	{
        if (cq->err != NULL)
          {
//        if(CorruptedObjects<19)
	    cq->perc.Hide();
            fprintf(cq->err,
              _("\nError: Object %lX:%X consistency check failed. Trying to ignore."),Pos,(int)cq->by);
/*        if(CorruptedObjects==19)
	    fprintf(cq->err,
	      _("\nError: Overwhelmed by too many currupted objects, switching to the silent mode.);*/
          }
        CorruptedObjects++;
        Result = false;
	/* asm int 3; end;*/
	}

  fseek(cq->wpd, Pos, 0);
  return Result;
}


int ixxx=0;

/*This is main procedure for processing one key. It is recursivelly called.*/
void ProcessKey(TconvertedPass1_WP5 *cq)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#ProcessKey() ");fflush(cq->log);
#endif
  WORD w;
  DWORD NewPos = 0;
  unsigned char by, subby;

  if (cq->by == 0)
    fread(&cq->by, 1, 1, cq->wpd);

  *cq->ObjType = '\0';
  w=1;
  cq->subby=0;


  /*Guessing end position of the object*/

  if (cq->by > 0xbf)
      {
      if (cq->by >= 0xc0 && cq->by <= 0xcf)
	  {	/*these functions has fixed size - see table SizesC0*/
	  if (SizesC0[cq->by - 0xc0] != 0)
		NewPos = SizesC0[cq->by - 0xc0] + cq->ActualPos;
          cq->subby = 0xff;   /*ignored subcommand*/
          }
      else if (cq->by >= 0xd0)
         {
         fread(&cq->subby, 1, 1, cq->wpd);
         Rd_word(cq->wpd, &w);
         NewPos = cq->ActualPos + w + 4;
         cq->Linebegin = false;
	 }
      }	/*other functions has size only 1 byte - all OK*/
      
  by = cq->by;
  subby = cq->subby;

  if (ExtendedCheck && NewPos != 0)
    if (!CheckConzistency(cq, NewPos))
      {
      NewPos = 0;
      strcpy(cq->ObjType, "Corrupted!!");
      goto _LObjectError;
      }

/*  if((by==0xd2)&&(subby==0xb))
               fprintf(cq->strip, " ****Start Of Table:%d****",cq->flag);*//**/

  if( filter[cq->flag][cq->by] )
     {
     switch (by)
	{
	case 0x02:PageNumber(cq);	break;		//^B
	case 0x0a:HardReturn(cq);	break;		// Hard return
	case 0x0b:Terminate_Line(cq,'p');strcpy(cq->ObjType, "SRt SoftPg");break;// Soft page
	case 0x0c:HardPage(cq);		break;	// Hard page
	case 0x0d:SoftReturn(cq);	break;		// Soft return
	case 0x20:putc(' ', cq->strip);   break;	/*soft space ' '*/

	case 0x80:strcpy(cq->ObjType, "NOP");		  break;	/* NOP */
	case 0x81:Justification(cq, 1 | 0x80);		  break;	/* Full justification */
	case 0x82:Justification(cq, 0 | 0x80);		  break;	/* Left justification - Ragged right */
	case 0x83:End_Align(cq);			  break;
//	case 0x84:break;						/* Reserved */
//	case 0x85:break;						/* Place Saver */
	case 0x86:CenterPage(cq);			  break;
	case 0x87:Column(cq,cq->DefColumns);		  break;	/* Col On */
	case 0x88:Column(cq,1);				  break;	/* Col Off */
//	case 0x89:break;						/* Reserved */
	case 0x8A:WidowOrphan(cq,true);			  break;
	case 0x8B:WidowOrphan(cq,false);		  break;
	case 0x8C:if(cq->char_on_line>=0) Terminate_Line(cq,'h'); // Hard return
				     else Terminate_Line(cq,'s');   // hard return mustn't occur here, fix it
		  break;
	case 0x8D:strcpy(cq->ObjType, "!Note Num");	  break;
	case 0x8E:strcpy(cq->ObjType, "!Fig Num");	  break;
	case 0x8F:if(toupper(cq->envir)=='C') Justification(cq, 0x83);
		  strcpy(cq->ObjType, "~Center");	  break;

	case 0x93:InvisibleSoftReturn(cq);		  break;

	case 0x96:strcpy(cq->ObjType, "!Block On");	  break;
	case 0x97:strcpy(cq->ObjType, "!Block Off");	  break;

	case 0x99:if(cq->envir != 'B' && cq->char_on_line > 0)	// Dormant Hard return
			     Terminate_Line(cq,'h');
			else fputc(' ', cq->strip);
		  strcpy(cq->ObjType, "Dorm HRt");
		  break;
	case 0x9A:CancelHyph(cq);			break;
	case 0x9B:strcpy(cq->ObjType, "End Def");	break; //End of automatically generated definition
	case 0x9E:Hyphenation(cq,false);		break;
	case 0x9F:Hyphenation(cq,true);			break;
	case 0xA0:fputc('~', cq->strip);strcpy(cq->ObjType, " ");
		  break;

	case 0xA9:
	case 0xAA:
	case 0xAB:CharacterStr(cq,"-",cq->Font);		/* Special_char */
		  strcpy(cq->ObjType, "-");
		  break;
	case 0xAC:
	case 0xAD:
	case 0xAE:SoftHyphen(cq);	  		break;
	case 0xAF:Column(cq,false);			break;
	case 0xB0:Column(cq,false);			break;

	case 0xC0:ExtendedCharacter(cq);	break;		/* Special character */
	case 0xC1:fread(&cq->subby, sizeof(unsigned char), 1, cq->wpd);
		  switch (cq->subby & 0xE8)
			 {
			 case 0:
			 case 0xc8:
			 case 0x48:
			 case 0x40:Tab(cq,5);                	        break;
			 case 0x60:if(cq->subby!=0x70) Flush_right(cq,0);
						  else Flush_right(cq,1);
				   break;
			 case 0xe0:Center(cq);            	        break;
			 }
		  break;

	case 0xC2:Indent(cq,5);		 break;
	case 0xC3:Attr_ON(cq);           break;
	case 0xC4:Attr_OFF(cq);          break;
	case 0xC6:End_of_indent(cq);     break;

	case 0xd0:switch (subby)
		   {
		   case 0x01:LRMargin(cq);		break;
		   case 0x02:LineSpacing(cq);           break;
		   case 0x04:Tabset(cq);		break;
		   case 0x06:Justification(cq,5);	break;
		   case 0x07:Suppress(cq,5);		break;
		   case 0x08:Page_number_position(cq,5);break;
		   }
		  break;
	case 0xD1:if(subby == 0) Color(cq);
		  if(subby == 1) SetFont5(cq);
		  break;
	case 0xd2:if(subby == 1) ColDef5(cq);
		  if(subby == 0xb) StartTable(cq, w);
		  break;
	case 0xd3:switch (subby)
		   {
		   case 0x02:SetFootnoteNum(cq,5);	break;
		   case 0x03:SetEndnoteNum(cq,5);	break;
                   case 0x04:SetPgNum(cq,5);		break;
	           case 0x06:Advance(cq);		break;
                   case 0x11:Language(cq, false);	break;
                   }
                  break;
	case 0xd4:if(cq->subby==3) strcpy(cq->ObjType,"!Repositioning Marker");
		  break;
        case 0xd5:Header_Footer(cq, NewPos);
                  break;
	case 0xD6:switch (subby)
		     {
		     case 0: Footnote(cq, NewPos);         break;
		     case 1: Endnote(cq, NewPos);          break;
		     }
		  break;

	case 0xD7: switch (subby)
		      {
		      case 0:   MarkedTextBegin(cq,NewPos);break;
		      case 0x1: EndSection(cq,5);	   break;
		      case 0x2: TableOfContents(cq);	   break;
		      case 0x3: MakeIndex(cq, NewPos);	   break;
		      case 0x5: PlaceEndnotes(cq);	   break;
		      case 0x7: MakeRef(cq, NewPos);       break;
		      case 0x8: MakeLabel(cq, NewPos);	   break;
//		      case 0xA: Start Included SubDocument
//		      case 0xB: End Included SubDocument
                      }
                   break;

        case 0xd8:switch (subby)
                    {
		    case 0x0: DateCode(cq);                break;
                    case 0x1: ParagraphNumber(cq);	   break;
		    case 0x2: Overstrike(cq, NewPos);      break;
                    case 0x3: PageNumber(cq);		   break;
		    }
                  break;

	case 0xd9:if (subby == 2) Comment5(cq, NewPos);
                  break;


	case 0xda:if (subby <= 4) AnyBox(cq, NewPos);
		  if (subby == 5) HLine(cq);
		  if (subby == 6) strcpy(cq->ObjType, "!VLine");;
                  break;

        /*      $DB:if subby=0 then StyleInsides(cq,NewPos);*/
        case 0xdc:switch (subby)	/* Normal table items */
                   {
                   case   0:CellTable(cq);	break;
		   case 0x1:RowTable(cq);	break;
                   case 0x2:EndTable(cq);	break;
                   }
                 break;
	 case 0xdd:switch (subby)	/* Table items on soft page break */
                   {
                   case 0x1:RowTable(cq);      break;
                   case 0x2:EndTable(cq);      break;
		   }
		 break;

	default:if (by >= 0x01 && by <= 0x7f)
		     {
		     CharacterStr(cq,lat[cq->by]);   /* Normal_char  */
		     }
		break;
	}
  }

_LObjectError:
  if (cq->log != NULL)
    {   /**/
    if (by==0xC0) fprintf(cq->log, " [ExtChr %s] ", cq->ObjType);
    else if ((by > 0x80)||(*cq->ObjType != '\0'))
	{
	fprintf(cq->log, _("\n%*sObject type:%3X subtype:%3X length:%4u"),
		  cq->recursion * 2, "", by, subby, w);
	if (*cq->ObjType != '\0')
		  fprintf(cq->log, " [%s] ", cq->ObjType);
	     else fprintf(cq->log, "    ");
	if(*cq->ObjType==0) UnknownObjects++;
	}
    else if (by >= ' ' && by <= 'z')
	         putc(by, cq->log);
    }

  if (NewPos == 0)
	{
	if(by<0xC0) cq->ActualPos++;	//Only one byte read - simple guess of new position
	       else cq->ActualPos = ftell(cq->wpd);
	return;
	}
  cq->ActualPos = ftell(cq->wpd);
  if (NewPos == cq->ActualPos) return;
  fseek(cq->wpd, NewPos, 0);
  cq->ActualPos = NewPos;
  NewPos = 0;
}

void InitFilter(void)
{
 filter[0]=set(set0,sizeof(set0)/sizeof(int));
 filter[1]=set(set1,sizeof(set1)/sizeof(int));
 filter[2]=set(set2,sizeof(set2)/sizeof(int));
 filter[3]=set();
}


/*******************************************************************/
/* This procedure provides all needed processing for the first pass*/
void Convert_pass1_WP5(FILE *FileIn,FILE  *table, FILE *StripOut,FILE *LogFile,FILE *ErrorFile)
{
#ifdef DEBUG
  fprintf(LogFile,"\n#Convert_pass1_WP5() ");fflush(LogFile);
#endif
  DWORD fsize;
  TconvertedPass1_WP5 cq;


  InitFilter();
  Table_Init(lat);

  InitAttr(cq.attr);
  cq.ConvertWP5=NULL;
  if(WPcharset == 2) cq.ConvertWP5 = GetTranslator("wp5_czTOwp5");

  cq.wpd   = FileIn;
  cq.table = table;
  cq.strip = StripOut;
  cq.log   = LogFile;
  cq.err   = ErrorFile;

  WP_Default(&cq);
  cq.DefColumns=2;


  cq.DocumentStart=ftell(cq.wpd);
  cq.ResourceStart=0;
  cq.Images.Length=0;
  cq.Images.DataPointer=0;
  cq.FontNames=0;
  cq.FontList=0;

  if (Verbosing >= 1)
	{
	printf(_("First pass WP 5.x:\n"));
	printf(_("Converting-percentage: "));
	}

  if(cq.DocumentStart>0x10)	//This can be omitted
    {
    WalkResources(&cq);
    fseek(cq.wpd,cq.DocumentStart,SEEK_SET);
    }


  cq.envir = ' ';
  cq.nomore_valid_tabs = false;

  fsize=filesize(cq.wpd);
  cq.perc.Init(ftell(cq.wpd), fsize );

  cq.rownum = 0;

  Make_tableentry_attr(&cq);   /* attribuut instelling */
  Make_tableentry_tabset(&cq);   /* Geef the defaulttabinstelling door */
  /* tied the 2e pass */

  cq.ActualPos = ftell(cq.wpd);
  while (cq.ActualPos < fsize)
      {
      if(Verbosing >= 1)		//actualise a procentage counter
	      cq.perc.Actualise(cq.ActualPos);

      cq.by = 0;
      ProcessKey(&cq);
      }


  cq.line_term = 's';			   /* Soft return */
  Make_tableentry_envir_extra_end(&cq);
  if(cq.CentrePage)    /* finish page centering */
  	{
	fprintf(cq.strip, "\\vfil ");
        cq.CentrePage=false;
	}

  num_of_lines_stripfile = cq.rownum;

  if(Verbosing >= 1)		//finishing a procentage counter
     {
     cq.perc.SetPercent(100);
     putchar('\n');
     }
}



extern string wpd_password,wpd_filename;
void StrGets(string & s);


WORD CheckSum(char *PassWord)
{
WORD csum;

csum=0;
if(PassWord==NULL) return(0);

while(*PassWord!=0)
	{
	if( *PassWord>='a' && *PassWord<='z')    /* convert to upper case */
			*PassWord -= 'a'-'A';

	csum = ( (csum >> 1) | ( csum << 15) ) ^ ( ((WORD)*PassWord)<<8 );
	PassWord++;
	}
return(csum);
}


int ObtainPassword(string & password, WORD encryption)
{
char c;

TryAgain:

  printf(_("\nEnter Password: "));
  StrGets(password);

if(CheckSum(password())!=encryption)
     {
     printf( _("\nWarning! Password checksum does not match file checksum! Again Y/n/i?"));

NextChar:c=getchar();
//	 putchar('\n');
	 switch(c)
	       {
	       case 'A':
	       case 'a':
	       case 'Y':
	       case 'y':getchar();
	       case '\n':goto TryAgain;
	       case 'n':
	       case 'N':getchar();return(false);
	       case 'I':
	       case 'i':getchar();
			ExtendedCheck=true;
			return(true);		// ignore=run anyway
	       default:goto NextChar;
	       }
	   }

return(true);
}

void Decrypt(FILE *Encrypted,FILE *Decrypted,const string & PassWord,FILE *ErrorFile)
{
char *buf;
WORD idx;                     /* index into buffer */
WORD pidx;                    /* index down password */
BYTE xmask;                   /* XOR mask */
DWORD len;
DWORD pos=ftell(Encrypted);
DWORD fsize=filesize(Encrypted);

 if(ErrorFile==NULL) ErrorFile=stderr;
 if((buf=(char *)malloc(4096))==NULL) return;


 pidx = 0;                       /* start of password */
 xmask = length(wpd_password)+1; /* start at password length+1 */

 if (Verbosing >= 1) printf(_("\nDecrypting ...\n\n"));

 while( pos<fsize )
	 {
	 if( fsize-pos > 4096 ) len = 4096;
			   else len = fsize-pos;

	 if( fread( buf, 1, len, Encrypted ) !=len )
		 {
		 fprintf(ErrorFile,_("\nError:Unexpected end of the input file."));
		 break;
		 }

	  for( idx=0; idx<len; idx++ )	/* now decrypt */
		 {
		 buf[idx] ^= PassWord()[pidx++] ^ xmask++;
		 if( pidx == length(PassWord) ) pidx = 0;
		 }

	   if( Decrypted )
		 {
		 if( fwrite( buf, 1, len, Decrypted ) !=len )
			 {
			 fprintf(ErrorFile,_("\nError:Couldn't write %lu bytes to the temporary file.\n"), (long unsigned)len);
			 break;
			 }
		 }
		 pos += len;
	 } /* while */

 if(buf!=NULL) free(buf);
}


void Convert_pass1_WP5_encrypted(FILE *FileIn,FILE  *table, FILE *StripOut,FILE *LogFile,FILE *ErrorFile)
{
#ifdef DEBUG
  fprintf(LogFile,"\n#Convert_pass1_WP5_encrypted() ");fflush(LogFile);
#endif
  WORD encryption;
  string temp_filename;
  DWORD DocumentStart;	/* position in file */
  FILE *DecryptedFile=NULL;
  BYTE SmallBuf[16];

  DocumentStart=ftell(FileIn);
  fseek(FileIn,12,SEEK_SET);	//check for encrypted documents

  RdWORD_HiEnd(&encryption, FileIn);
  if(encryption!=0)
	{
	if(CheckSum(wpd_password())!=encryption)
	   {
	   if(!ObtainPassword(wpd_password,encryption)) return;
	   }

	temp_filename = copy(wpd_filename,0,
		GetExtension(wpd_filename())- wpd_filename()) + ".$$$";

	if((DecryptedFile=fopen(temp_filename,"wb"))==NULL)
			RunError(6,temp_filename);


	fseek(FileIn,0l,SEEK_SET);
	fread( SmallBuf, 1, 16, FileIn );
	SmallBuf[12] = SmallBuf[13] = 0;        /* write out first part of WP5.1 file */
	fwrite( SmallBuf, 1, 16, DecryptedFile );

	Decrypt(FileIn,DecryptedFile,wpd_password,ErrorFile);
	} /* ver5 */


  if(DecryptedFile!=NULL) /* Now perform normal conversion */
    {
    fclose(DecryptedFile);
    if((DecryptedFile=fopen(temp_filename,"rb"))==NULL) return;

    fseek(DecryptedFile,DocumentStart,SEEK_SET);
    Convert_pass1_WP5(DecryptedFile, table, StripOut, LogFile, ErrorFile);
    fclose(DecryptedFile);

#ifndef DEBUG
    unlink(temp_filename);
#endif		// DEBUG
    }

}


/*--------------------End of PASS1--------------------*/





/* End. */
