/******************************************************************************
 * program:     wp2latex                                                      *
 * function:    convert WordPerfect 5.x or 6.x files into LaTeX               *
 * modul:       formulas.cc                                                   *
 * description: Procedures for converting WP formulas into LaTeX equivalents. *
 ******************************************************************************/

/* Output from p2c 1.21alpha-07.Dec.93, the Pascal-to-C translator */
/* From input file "formulas.pas" */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>

#include <strings.h>
#include "wp2latex.h"


void CrackObject(TconvertedPass1 *cq, DWORD end_of_code);
void DoCaption5(TconvertedPass1 *_cq,unsigned short CaptionSize);
void ExtractCaptionLabel5(TconvertedPass1 *_cq,const TBox & Box);
void ExtractCaptionLabel6(TconvertedPass1 *_cq,const TBox & Box);

int FormulaNo=0;


void WPU2cm(string & Numeral)
{
float f;
char array[32];

 f=atof(Numeral());
 sprintf(array,"%2.2fcm",f/470);
 Numeral=array;
}

#define WP2LATEX 1
#define AMSSYMB  2
#define LATEXSYM 4

struct FmlTranslationArray
    {
    char *szWP;               /* original WordPerfect term */
    char *szTeX;              /* translated LATEX term     */
    char styles;	      /* styles required for this glyph */
    };

/* special meaning of digits in the LATEX term:
** 0: previous argument,
** 1: next argument,
** 2: next next argument,
** 5: same word as WordPerfect, with "\" before, lower case,
** 6: same word as WordPerfect, with "\" before, right case
** 7: count Ampersands and write count times the following char
** 8:
** 9: advanced next argument with all xx_yy^zz stuffs (similiarly to 1)
**
** #: converts following argument from number in WPUnits to pt
**
** lower case chars in the WP-term match only lower case chars in
** the WordPerfect formula,
** upper case chars match both cases
**
** note: the order of words is sometimes essential !
** first the more specialized and longer words
** then the more common and shorter words.
*/

struct FmlTranslationArray FmlTransTable[] =
{
/* Hard FIX for accents in the math mode */
   {"\2acute","{\\acute{1}}",0},
   {"\2bar","{\\bar{1}}",0},
   {"\2breve","{\\breve{1}}",0},
   {"\2dot","{\\dot{1}}",0},
   {"\2grave","{\\grave{1}}",0},
   {"\2hat","{\\hat{1}}",0},
   {"\2tilde","{\\tilde{1}}",0},

/*High priority symbols*/
   { "LEFT", "5", 0 },
   { "RIGHT", "5", 0 },


/* 11 char words */
   { "SMALLCOPROD", "\\coprod", 0 },	/* bad translation */
/* 10 char words */
   { "STACKALIGN", "\\begin{array}{r@{}l}1\\end{array}", 0 },
   { "varepsilon", "5", 0 },
/* 9 char words */
   { "BIGOMINUS", "\\ominus", 0 },		/* bad translation */
   { "BIGOTIMES", "5", 0 },
   { "IDENTICAL", "5", 0 },		/* in WP2LaTeX style */
   { "LINESPACE", "\\vspace{#1}", 0 },	/* very bad translation */
   { "MINUSPLUS", "\\mp", 0 },
   { "PLUSMINUS", "\\pm", 0 },
   { "SMALLOINT", "\\oint", 0 },		/* bad translation */
   { "SMALLPROD", "\\prod", 0 },          /* bad translation */
   { "THEREFORE", "5", 0 },		/* in WP2LaTeX style */
   { "UNDERLINE", "5{1}", 0 },
/* 8 char words */
   { "ANGSTROM", "\\AA", 0 },
   { "SCALESYM", "scalesym1",0 },		/* no translation known */
   { "BIGOPLUS", "5", 0 },
   { "BIGSQCAP", "\\sqcap", 0 },		/* bad translation */
   { "BIGSQCUP", "5", 0 },
   { "BIGUPLUS", "5", 0 },
   { "BIGWEDGE", "5", 0 },
   { "DOTSAXIS", "\\cdots", 0 },
   { "DOTSDIAG", "\\ddots", 0 },
   { "DOTSVERT", "\\vdots", 0 },
   { "EMPTYSET", "5", 0 },
   { "INFINITY", "\\infty", 0 },
   { "LONGDIVS", "\\div", 0 },		/* very bad translation */
   { "OVERLINE", "5{1}", 0 },
   { "PARALLEL", "5", 0 },
   { "SETMINUS", "\\\\", 0 },
   { "SMALLINT", "\\int", 0 },		/* very bad translation */
   { "SMALLSUM", "\\sum", 0 },            /* very bad translation */
   { "SQSUBSET", "5", LATEXSYM },
   { "SQSUPSET", "5", LATEXSYM },
   { "varsigma", "5", 0 },
   { "vartheta", "5", 0 },
/* 7 char words */
   { "BECAUSE", "5", 0 },			/* in WP2LaTeX style */
   { "BETWEEN", "\\between ", 0 },		/* in WP2LaTeX style */
   { "BIGODIV", "\\div", 0 },		/* very bad translation */
   { "BIGODOT", "5", 0 },
   { "BINOMSM", "{\\small {1 \\choose 2}}", 0 },
   { "DOTSLOW", "\\ldots", 0 },
   { "epsilon", "5", 0 },
   { "EPSILON", "\\epsilon", 0 },
   { "LDBRACK", "[", 0 },			/* very bad translation */
   { "LONGDIV", "\\div", 0 },			/* very bad translation */
   { "omikron", "o", 0 },			/* very bad translation */
   { "OMIKRON", "O", 0 },			/* very bad translation */
   { "MASSERT", "\\dashv", 0 },
   { "MATFORM", "", 0 },			/* no translation kwown */
   { "MSANGLE", "\\measuredangle", AMSSYMB },
   { "NASYMEQ", "\\not\\asymp", 0 },
   { "PARTIAL", "5", 0 },
   { "PHANTOM", "5{1}", 0 },
   { "RDBRACK", "]", 0 },			/* very bad translation */
   { "RTANGLE", "?", 0 },			/* no translation known */
   { "upsilon", "\\upsilon", 0 },
   { "UPSILON", "\\Upsilon", 0 },
/* 6 char words */
   { "ALIGNC", "", 0 },			/* no translation known */
   { "ALIGNL", "", 0 },
   { "ALINGR", "", 0 },
   { "APPROX", "5", 0 },
   { "arccos", "5", 0 },
   { "arcsin", "5", 0 },
   { "arctan", "5", 0 },
   { "ASSERT", "\\vdash", 0 },
   { "ASYMEQ", "\\asymp", 0 },
   { "BIGCAP", "5", 0 },
   { "BIGCUP", "5", 0 },
   { "BIGVEE", "5", 0 },
   { "CIRCLE", "\\dot{0}", 0 },		/* no translation known */
   { "COPROD", "5", 0 },
   { "EXISTS", "5", 0 },
   { "FORALL", "5", 0 },
   { "lambda", "5", 0 },
   { "LAMBDA", "\\Lambda", 0 },
   { "LANGLE", "5", 0 },
   { "LBRACE", "\\{", 0 },
   { "LFLOOR", "5", 0 },
   { "liminf", "5", 0 },
   { "limsup", "5", 0 },
   { "MATRIX", "\\begin{array}{7}1\\end{array}", 0 },
   { "MODELS", "5", 0 },
   { "NEQUIV", "\\not\\equiv", 0 },
   { "OMINUS", "5", 0 },
   { "OTIMES", "5", 0 },
   { "OVERSM", "{\\mathsmaller{\\frac{0}{1}}}", 0 }, /* in WP2LaTeX style */
   { "PRECEQ", "5", 0 },
   { "PROPTO", "5", 0 },
   { "QEQUAL", "\\stackrel{?}{=}", 0 },
   { "RANGLE", "5", 0 },
   { "RBRACE", "\\}", 0 },
   { "RFLOOR", "5", 0 },
   { "RIMAGE", "5", 0 },			/* in WP2LaTeX style */
   { "SANGLE", "\\sphericalangle", AMSSYMB },
   { "SUBSET", "5", 0 },
   { "SUCCEQ", "5", 0 },
   { "SUPSET", "5", 0 },
   { "varphi", "5", 0 },
   { "varrho", "5", 0 },
   { "WREATH", "\\wr", 0 },
/* 5 char words */
   { "ACUTE", "5{0}", 0 },
   { "alpha", "5", 0 },
   { "ALPHA", "A", 0 },
   { "ANGLE", "5", 0 },
   { "BINOM", "{1 \\choose 2}", 0 },
   { "BREVE", "5{0}", 0 },
   { "cosec", "\\csc", 0 },
   { "DDDOT", "\\ddot{0}", 0 },
   { "delta", "5", 0 },
   { "DELTA", "\\Delta", 0 },
   { "DLINE", "\\|", 0 },
   { "DOTEQ", "5", 0 },
   { "EQUIV", "5", 0 },
   { "FROWN", "5", 0 },
   { "gamma", "5", 0 },
   { "GAMMA", "\\Gamma", 0 },
   { "GRAVE", "5{0}", 0 },
   { "IMAGE", "5", 0 },			/* in WP2LeTeX style */
   { "kappa", "5", 0 },
   { "KAPPA", "\\Kappa", 0 },
   { "LCEIL", "5", 0 },
   { "NOTIN", "5", 0 },
   { "NROOT", " \\sqrt[1]{2}", 0 },
   { "omega", "\\omega", 0 },
   { "OMEGA", "\\Omega", 0 },
   { "OPLUS", "5", 0 },
   { "RCEIL", "5", 0 },
// { "RIGHT", "5", 0 }, moved up
   { "sigma", "5", 0 },
   { "SIGMA", "\\Sigma", 0 },
   { "SIMEQ", "5", 0 },
   { "SMILE", "5", 0 },
   { "SQCAP", "5", 0 },
   { "SQCUP", "5", 0 },
   { "STACK", "\\begin{array}{c}1\\end{array}", 0 },
   { "theta", "5", 0 },
   { "THETA", "\\Theta", 0 },
   { "TILDE", "{5{0}}", 0 },
   { "TIMES", "5", 0 },
   { "UPLUS", "5", 0 },
   { "varpi", "5", 0 },
   { "VARPI", "6", 0 },
/* Words with 4 chars */
   { "beta", "5", 0 },
   { "BETA", "B", 0 },
   { "BOLD", "{\\bf 1}", 0 },
   { "CDOT", "5", 0 },
   { "CONG", "5", 0 },
   { "cosh", "5", 0 },
   { "coth", "5", 0 },
   { "DDOT", "5{0}", 0 },
   { "DSUM", "\\dot{+}?", 0 },
   { "DYAD", "{\\bar{0}}", 0 },
   { "FROM", "_{9}", 0 },
   { "FUNC", "{\\rm 1}", 0 },
   { "GRAD", "\\nabla", 0 },
   { "HORZ", "\\hspace{#1}", 0 },
   { "IMAG", "\\Im", 0 },
   { "iota", "5", 0 },
   { "IOTA", "I", 0 },
   { "ITAL", "{\\it 1}", 0 },
// { "LEFT", "5", 0 }, 		moved up
   { "LINE", "|", 0 },
   { "NISO", "\\not\\Bumpeq", AMSSYMB },	/* translation only with amssymb.sty */
   { "ODIV", "\\div", 0 },
   { "ODOT", "5", 0 },
   { "OINT", "5", 0 },
   { "OVER", "{\\frac{0}{1}}", 0 },
   { "OWNS", "\\ni", 0 },
   { "PERP", "5", 0 },
   { "PREC", "5", 0 },
   { "PROD", "5", 0 },
   { "REAL", "\\Re", 0 },
   { "sinh", "5", 0 },
   { "SQRT", "5{1}", 0 },
   { "SUCC", "5", 0 },
   { "tanh", "5", 0 },
   { "VERT", "\\vspace{#1}", 0 },
   { "zeta", "5", 0 },
   { "ZETA", "Z", 0 },
/* Words with 3 chars */
   {"'''","{0'''}", 0 },
   { "AND", "\\wedge", 0 },
   { "arc", "{\\rm arc}", 0 },
   { "BAR", "{5{0}}", 0 },
   { "CAP", "5", 0 },
   { "chi", "5", 0 },
   { "CHI", "X", 0 },
   { "cos", "5", 0 },
   { "cot", "5", 0 },
   { "CUP", "5", 0 },
   { "DEG", "\\degrees", 0 },
   { "det", "5", 0 },
   { "DIV", "5", 0 },
   { "DOT", "5{0}", 0 },
   { "eta", "5", 0 },
   { "ETA", "H", 0 },
   { "exp", "5", 0 },
   { "gcd", "5", 0 },
   { "GGG", "\\gg", 0 },
   { "HAT", "5{0}", 0 },
   { "INF", "\\infty", 0 },
   { "INT", "5", 0 },
   { "ISO", "\\Bumpeq", AMSSYMB },		/* translation only with amssymb.sty */
   { "lim", "5", 0 },
   { "LLL", "\\ll", 0 },
   { "log", "{\\rm log}", 0 },
   { "max", "5", 0 },
   { "MHO", "5", LATEXSYM },
   { "min", "5", 0 },
   { "mod", "{\\rm mod}", 0 },
   { "NOT", "\\neg", 0 },
   { "phi", "5", 0 },
   { "PHI", "\\Phi", 0 },
   { "psi", "5", 0 },
   { "PSI", "\\Psi", 0 },
   { "rho", "5", 0 },
   { "RHO", "P", 0 },
   { "sec", "5", 0 },
   { "SIM", "5", 0 },
   { "sin", "5", 0 },
   { "SUB", "_{1}", 0 },
   { "SUM", "5", 0 },
   { "SUP", "^{1}", 0 },
   { "tan", "5", 0 },
   { "tau", "5", 0 },
   { "TAU", "\\Tau", 0 },
   { "TOP", "5", 0 },
   { "VEC", "5{0}", 0 },
   { "XOR", "\\underline\\vee", 0 },
/* words with 2 chars */
   { "!=", "\\neq", 0 },
   { "==", "\\equiv", 0 },
   { "+-", "\\pm", 0 },
   { "-+", "\\mp", 0 },
   { "<=", "\\le", 0 },
   { "<<", "\\ll", 0 },
   { ">=", "\\ge", 0 },
   { ">>", "\\gg", 0 },
   { "IN", "5", 0 },
   { "ln", "5", 0 },
   { "mu", "\\mu", 0 },
   { "MU", "\\Mu", 0 },
   { "nu", "\\nu", 0 },
   { "NU", "N", 0 },
   { "OR", "\\vee", 0 },
   { "pi", "5", 0 },
   { "PI", "\\Pi", 0 },
   { "TO", "^{9}", 0 },
   { "xi", "5", 0 },
   { "XI", "\\Xi", 0 },
   { "\\'", "'", 0 },			/* Bad conversion - symbol should be like ^{|} */
/* words with 1 char */
   { "^", "^{1}", 0 },
   { "_", "_{1}", 0 },
   {"#"," \\\\ ", 0 },
   {"`","\\,", 0 },
   { "'", "{0'}", 0 },
   { "%", "\\%", }

/*  (szWP:''';szTeX:'{0'}'),
    {"\\'","\\dq"},
    {"&","&"}   Ampersand is solved another way*/
    };
int nFmlTransWords = sizeof(FmlTransTable)/sizeof(FmlTransTable[0]);


static boolean CheckEquation(char *EquStr,int & Braces,int & BegBrace)
{
int len;

BegBrace=0;
Braces=0;
if(EquStr==NULL) return(false);
len=strlen(EquStr);

while(len>0)
	{
        switch(*EquStr)
           {
           case '\\':len--;EquStr++;
           	     break;
           case  '{':Braces++;
           	     break;
           case  '}':Braces--;
		     if(BegBrace>Braces) BegBrace=Braces;
                     break;
           }
        len--;
        EquStr++;
        }

return(BegBrace!=0 || Braces!=0);
}


static boolean RemoveSymbol(string & str1,string & str2,char * symbol, int *xpos)
{
  char *str,*UPStr;
  int  i,symlen;
  string t;

  if(symbol==NULL) return false;
  symlen=strlen(symbol);
  str=StrStr(str2(), symbol);

NewTest:
  if(str!=NULL)
    {
    if(str>str2())
	{			//fix symbol, which begins by \\anything
	if( (*(str-1)=='\\') ||
	    ( (isalpha(*(str-1))) && (isalpha(*symbol)) ) ||
	    ( (isdigit(*(str-1))) && (isdigit(*symbol)) ) )
		  {
		  str=strstr(++str, symbol);
		  goto NewTest;
		  }
	 }

     if( (isalpha(*(str+symlen))||isdigit(*(str+symlen))) &&
	    isalpha(*symbol) )
		  {
		  str=strstr(++str, symbol);
		  goto NewTest;
		  }
     }
  i=-1;
  if(str!=NULL) i=str-str2();
  if(i==0) goto SkipUpcaseTest;


  if(*symbol > 'z' || *symbol < 'a')	//small speedup trick
    {
    t=str2;
    for (i=0; i<length(t); i++)
	{
	t[i] = toupper(t[i]);
	}
    UPStr=StrStr(t(), symbol);		//find upper case match
    }
  else UPStr=NULL;


NewTest2:
  if(str!=NULL)
    {
    if(UPStr==NULL) goto SkipUpcaseTest;	//only str matches
    if(UPStr-t()>=i) goto SkipUpcaseTest;	//str match is better
    }
  else
    {
    if(UPStr == NULL) return false;  //Not found
    }


  if(UPStr>t())
	{			//fix symbol, which begins by \\anything
	if( (*(UPStr-1)=='\\') ||
	    ( (isalpha(*(UPStr-1))) && (isalpha(*symbol)) ) ||
	    ( (isdigit(*(UPStr-1))) && (isdigit(*symbol)) ) )
		  {
		  UPStr=strstr(++UPStr, symbol);
		  goto NewTest2;
		  }
	 }

  if(isalpha(*(UPStr+symlen))||isdigit(*(UPStr+symlen)))
	 {
	 UPStr=strstr(++UPStr, symbol);
	 goto NewTest2;
	 }


  i=UPStr-t();


SkipUpcaseTest:
  if (i >= 1 && (str2[i-1] == '\\' || isalpha(str2[i-1])) &&
      isalnum(symbol[0]) )
    return false;

  str1 += copy(str2,0,i);
  str2  = copy(str2,i+strlen(symbol),length(str2)-i-strlen(symbol));

  *xpos = i;
  return true;
}


static boolean Remove1stArg(string & str,string & arg)
{
  int i;
  char c;
  char LastC;
  string InternalArg;
  boolean FirstCipher=false;

  arg = "";

  while (str != "" && str[0] == ' ')
    str=copy(str,1,length(str)-1);

  LastC = '\0';
  i=0;
  while (str != "" && (str[0] != ' ' || i != 0))
     {
     c = str[0];

     if (i <= 0 && LastC!='\\' && length(arg)>0)
	{
	if (c == '^' || c == '_' || c == '{' || c == '\\' || c == '&' || c == '#' || c == ',' || c == ';' ||
	    c == '$' || c == '@' || c == '%' || c == '*'  || isspace(c) || c == '+' || c == '-' || c == '=' ||
	    c == ']' || c == '[' || c == '(' || c == ')' ||
	    (FirstCipher && !isdigit(c))  )
			break;
	}
     if(!LastC) FirstCipher=isdigit(c); //Test whether first character is digit 0..9

     str=copy(str,1,length(str)-1);

     if (LastC != '\\')
	 {
	 if (c == '{')
	      {
	      if(i==0 && LastC!=0)	//argument is already read
			 {
			 str=c+str;
			 break;
			 }
	      i++;
	      if (i == 1) continue;
	      }
	 if (c == '}')
	      {
	      i--;
	      if (i == 0) break;
	      }
	 if ((c=='\\' || c=='+')
		 && i == 0 && arg != "")
	     {
	     str=c+str;
	     break;
	     }
	 }

     arg+=c;

     if(c=='t' && (isspace(str[0])||str[0]==')'||str[0]=='('||str[0]==']'||str[0]=='['||str[0]=='|'))
	 {
	 if(LastC=='f' && arg[length(arg)-3]=='e' && arg[length(arg)-4]=='l')
		 {	//bracket    "\left ("   found
		 LastC = c;
		 if(length(arg)<5) c=' ';
			      else c=arg[length(arg)-5];
		 if(!isalnum(c)) i++;
		 continue;
		 }
	 if(LastC=='h' && arg[length(arg)-3]=='g' && arg[length(arg)-4]=='i' && arg[length(arg)-5]=='r')
		 {	//bracket    "\right )"   found
		 LastC = c;
		 c=arg[length(arg)-6];
		 if(!isalnum(c))
			 {
			 do {
			    c = str[0];
			    arg+=c;
			    str=copy(str,1,length(str)-1);
			    } while(isspace(c));
			 i--;
			 if (i == 0) break;
			 }
		 continue;
		 }

	 }

     LastC = c;
     }

if(arg=="") return(0);

InternalArg=arg; //Check whether arg contains any non-terminal symbol that need to be expanded
InternalArg.ToUpper();
for (i = 0; i < nFmlTransWords; i++)
  {
  if(arg==FmlTransTable[i].szWP || InternalArg==FmlTransTable[i].szWP)
    {
    if(strstr(FmlTransTable[i].szTeX,"1")!=NULL)
	{
	Remove1stArg(str,InternalArg);
	arg+=" "+InternalArg;
	if(strstr(FmlTransTable[i].szTeX,"2")!=NULL)
	   {
	   Remove1stArg(str,InternalArg);
	   arg+=" "+InternalArg;
	   }
	}
    break;
    }
  }


return(0);
}


static boolean RemoveAdvanced1stArg(string & str,string & arg)
{
  int i,ZeroPass;
  char c;
  char LastC, LastSupSub;
  string InternalArg;

  arg = "";

  while (str != "" && str[0] == ' ')
    str=copy(str,1,length(str)-1);

  LastSupSub = LastC = '\0';
  ZeroPass = i = 0;
  while (str != "" )
     {
     c = str[0];
     if (i <= 0  && !isspace(c) && !LastSupSub)
       if(arg!="")		//argument is not empty
	{
	if((isspace(LastC) || LastC=='}') &&
	   (!isspace(c) && c!='_' && c!='^') ) break;
	if(c=='\\' || c==',') break;
        }

    str=copy(str,1,length(str)-1);

    if (LastC != '\\')
        {
        if (c == '{')
	     {
	     i++;
	     }
        if (c == '}')
	     {
	     i--;
             if(i==0) ZeroPass++;
	     }
        }

    arg+=c;
    LastC = c;
    if(!isspace(c)) LastSupSub=0;
    if(c == '_' || c == '^') LastSupSub=1;
    }

while (arg != "" && arg[length(arg)-1] == ' ')
	arg=copy(arg,0,length(arg)-1);
if(ZeroPass==1 && arg[0]=='{' && arg[length(arg)-1]=='}')
	arg = copy(arg,1,length(arg)-2);


if(arg=="") return(0);

InternalArg=arg; //Check whether arg contains any non-terminal symbol that need to be expanded
InternalArg.ToUpper();
for (i = 0; i < nFmlTransWords; i++)
  {
  if(arg==FmlTransTable[i].szWP || InternalArg==FmlTransTable[i].szWP)
    {
    if(strstr(FmlTransTable[i].szTeX,"1")!=NULL)
	{
	Remove1stArg(str,InternalArg);
	arg+=" "+InternalArg;
	if(strstr(FmlTransTable[i].szTeX,"2")!=NULL)
	   {
	   Remove1stArg(str,InternalArg);
	   arg+=" "+InternalArg;
	   }
	}
    break;
    }
  }

return(0);
}


static boolean RemoveLastArg(string & str,string & arg)
{
  int i,j;
  char c,LastC;

  LastC=0;
  arg = "";
  while (str != "" && str[length(str) - 1] == ' ')
      {
      str=copy(str,0,length(str)-1);
      }

  i=0;			//Tady je chyba !!!!!!!!!!!!!!!!!!!!!
  while (str != "") // && (str[length(str) - 1] != ' ' || i != 0))
      {
      c = str[length(str) - 1];
      if (i <= 0)
	 {
	 if(length(arg)>0)
		{
		if (c == '^' || c == '_' || c == '{' || c == '&' || c == '#' || c == ',' || c == ';' ||
		    c == '$' || c == '@' || c == '%' || c == '*'  || isspace(c) || c == '+' || c == '-' || c == '=' ||
		    c == ']' || c == '[' || c == '(' || c == ')' )  break;
		if ( (isalpha(LastC) || LastC==0) && !isalpha(LastC)) break;
		if ( c == '\\')
			{
			str=copy(str,0,length(str)-1);
			arg=c+arg;
			break;
			}
		}
	 }

      str=copy(str,0,length(str)-1);

      if (c == '}')
	  {
	  i++;
	  if (i == 1)
		{
		if(arg=="") continue;
		str+='}';	// fix the situation    "{...}arg"
		i--;
		break;
		}
	  }
      if (c == '{')
	  {
	  i--;
	  if (i == 0) break;
	  }

      if(c==')' || c==']' || str[0]=='|')
	  {
	  j=length(str) - 2;
	  if(isspace(str[j])) j--;
	  if( str[j]=='t' && str[j-1]=='h' && str[j-2]=='g' && str[j-3]=='i' && str[j-4]=='r')
		{  // 'right )' detected
		arg="right "+(c+arg);
		LastC='r';
		str=copy(str,0,j-4);
		i++;
		continue;
		}
	  }
      if( (c=='(' || c=='[' || str[0]=='|') && i>0)
	  {
	  j=length(str) - 2;
	  if(isspace(str[j])) j--;
	  if( str[j]=='t' && str[j-1]=='f' && str[j-2]=='e' && str[j-3]=='l')
		{  // 'left (' detected
		arg="left "+(c+arg);
		LastC='r';
		str=copy(str,0,j-3);
		i--;
		continue;
		}
	  }

      arg=c+arg;
      LastC=c;
      }

return(0);
}


static void Interpret(TconvertedPass1 *cq, string & strB, string & strE,
                const string & seq,const string & symbol,const string & arg0,const string & arg1,const string & arg2)
{
#ifdef DEBUG
  fprintf(cq->log,"\n#Interpret(%s,%s) ",strB(),strE());fflush(cq->log);
#endif
  int i, j, a, a2;
  int p;
  string Numeral;

  if (strE[0] != '_' && strE[0] != '^') strE=' '+strE;

  p=0;
  for (i = 0; i < length(seq); i++)
    {
    switch (seq[i])
        {
        case '0': strB+=arg0;
                  break;

        case '1': if (p == 0) p = length(strB);
                  strB+=arg1;
                  break;

        case '2': if (p == 0)  p = length(strB);
                  strB+=arg2;
                  break;

	case '5': strB+="\\"+ToLower(symbol);
                  break;

        case '6': strB+="\\"+symbol;
                  break;

        case '7': a = 0;
                  a2 = 0;
                  for (j = 0; j < length(arg1); j++)
                         {
             	         if (arg1[j] == '&') a2++;
                         if (arg1[j] == '#')
                             {
             	             if (a2 > a) a = a2;
                             a2 = 0;
                             }
                         }
                   if (a2 > a) 	a = a2;

                   for (j = 0; j <= a; j++)  strB+='c';
                   break;

        case '9': if (p == 0) p = length(strB);
                  strB+=arg1;
                  break;

        case '#' : Numeral="";
	           switch (seq[++i])
		        {
		        case '0':Numeral=arg0; break;
                        case '1':Numeral=arg1; break;
                        case '2':Numeral=arg2; break;
                        default: i--;
			};
                   if(Numeral=="")
		   	{
                        if (cq->err != NULL)
			    fprintf(cq->err, _("\nWarning: #%d Missing numeric argument or error in formula grammar!"),FormulaNo);
                        strB+="0pt ";
			break;
                        }
                   WPU2cm(Numeral);
                   strB+= Numeral;
                   break;

        default: strB+=seq[i];
                 break;
        }
  }

  if (p != 0) {
             strE=copy(strB,p,length(strB)-p)+strE;
             strB=copy(strB,0,p);
             }

}


void PutFormula(TconvertedPass1 *cq,char *Str,const TBox & Box, const char WP)
{
char *pom,*pom2;
boolean MathArray=false;

if(StrStr(Str," \\nonumber ")!=NULL) MathArray=true;

if(Box.AnchorType==2)
  {
  pom=pom2="";
  if(MathArray)
	{
	pom="\\begin{array}{c}";
	pom2="\\end{array}";
	}
  if (cq->attr.Math_Depth)
	       fprintf(cq->strip, "}{%s%s%s}{\\rm ",pom,Str,pom2);
	  else fprintf(cq->strip, " $%s%s%s$ ",pom,Str,pom2);
  return;
  }


cq->envir = ' ';
if(cq->line_term=='h') cq->char_on_line=true; //after HRt leave one row empty
cq->line_term = 's';			/* Soft return */
if(cq->char_on_line==true)
		   {
                   NewLine(cq);
		   }

if(Box.Type==1 && Box.AnchorType!=2)
	{
	fprintf(cq->strip,"\\begin{table}[htbp]");
	NewLine(cq);
	fprintf(cq->strip,"\\begin{displaymath}");
	}
   else {
	if(Box.CaptionSize>0)
		{
		if(MathArray) fprintf(cq->strip,"\\begin{eqnarray}");
			 else fprintf(cq->strip,"\\begin{equation}");
		}
	else {
	     if(MathArray) fprintf(cq->strip,"\\begin{eqnarray*}");
		      else fprintf(cq->strip,"\\begin{displaymath}");
	     }
	}
NewLine(cq);


while(Str!=NULL)
   {
   pom=StrStr(Str,"\\\\");
   if(pom!=NULL) *pom=0;

   fputs(Str,cq->strip);

   if(pom!=NULL)
   	{
        fputs("\\\\",cq->strip);
	Str=pom+2;
	while(*Str==' ') Str++;
	if(*Str==0) break;
	NewLine(cq);
	}
   else Str=NULL;
   }

NewLine(cq);
if(WP==5) ExtractCaptionLabel5(cq,Box);
if(WP==6) ExtractCaptionLabel6(cq,Box);
if(Box.Type==1 && Box.AnchorType!=2)
	{
	fprintf(cq->strip,"\\end{displaymath}");
	NewLine(cq);
	if(Box.CaptionSize>0)
		      {
		      fseek(cq->wpd, Box.CaptionPos, SEEK_SET);
		      if(WP==5) DoCaption5(cq, Box.CaptionSize);
		      }
	fprintf(cq->strip,"\\end{table}");
	}
   else {
	if(Box.CaptionSize>0)
	    {
	    if(MathArray) fprintf(cq->strip,"\\end{eqnarray}");
		     else fprintf(cq->strip,"\\end{equation}");
	    }
	else {
	     if(MathArray) fprintf(cq->strip,"\\end{eqnarray*}");
		      else fprintf(cq->strip,"\\end{displaymath}");
	     }
	}
NewLine(cq);
}

//---------------
void OptimizeFormulaStr(TconvertedPass1 *cq, string & StrBeg)
{
  int j, xpos, i;
  string StrEnd, arg0, arg1, arg2;
  boolean error=false;
  BYTE styles;


  StrBeg=cutspaces(StrBeg);
  if(((length(StrBeg)>2) && (StrBeg[length(StrBeg)-2]!='\\'))   //fix one error when equation end with dollar $
	 && (StrBeg[length(StrBeg)-1]=='&'))
		StrBeg=copy(StrBeg,0,length(StrBeg)-1);
//We have all equation contents in the string StrBeg at this point

  if (StrBeg == "") return;  /*equation is empty-nothing to do*/


  if(CheckEquation(StrBeg(),i,j))
  	{		//An attempt to fix invalid equation is made
	if (cq->err != NULL)
	    {
	    cq->perc.Hide();
	    fprintf(cq->err, _("\nWarning: Formula #%d is syntaktically wrong, No. of '{'-'}' is %d, %d!"),FormulaNo,i,j);
	    }
        while(j<0 || i<0)
		{
                StrBeg='{'+StrBeg;
                j++;
                i++;
                }
        while(i>0)
		{
                StrBeg+='}';
		i--;
		}
	error = true;
	}

//Here starts an equation conversion phase

//  Fix multiline equations here
  j=0;
  for(i=0;i<length(StrBeg);i++)
	{
	if(StrBeg[i]=='\\')
		{
		i++;
		continue;
		}
	if(StrBeg[i]=='{') j++;
	if(StrBeg[i]=='}') j--;
	if(StrBeg[i]=='#' && j==0)
		{
		StrBeg=copy(StrBeg,0,i)+" \\nonumber # "+copy(StrBeg,i+1,length(StrBeg)-i-1);
		i+=11;   //  > strlen(" \\nonumber # ")-1
		}
	}
// if(cq->log!=NULL) fprintf(cq->log,"\n%s\n",StrBeg.ch);

  StrEnd = "";
  for (i = 0; i < nFmlTransWords; i++)
      {
      xpos = 0;

      StrEnd = StrBeg + StrEnd;
      StrBeg = "";

      while (RemoveSymbol(StrBeg, StrEnd, FmlTransTable[i].szWP, &xpos))
	   {
	   if(FmlTransTable[i].styles != 0) /* check aditional styles */
		{
		styles=FmlTransTable[i].styles;
		if( (styles & WP2LATEX) != 0) styles=0;
		if( (styles & AMSSYMB) != 0)
			{
			if(Amssymb>=false) {Amssymb=true; styles=0;}
			}
		if( (styles & LATEXSYM) != 0)
			{
			if ((LaTeX_Version & 0xFF00) >= 0x300)
			   {
			   if(LaTeXsym>=false) {LaTeXsym=true;styles=0;}
			   }
			else styles=0;
			}

		if(styles!=0)  /* The proper style cannot be used */
			{
			StrBeg += FmlTransTable[i].szWP;
			break;
			}
		}

           arg0 = arg1 = arg2 = "";

           if (strstr(FmlTransTable[i].szTeX, "0") != NULL)
	       {
	       RemoveLastArg(StrBeg, arg0);
               }
           if (strstr(FmlTransTable[i].szTeX, "9") != NULL)
	       {
	       RemoveAdvanced1stArg(StrEnd, arg1);
               } else
           if (strstr(FmlTransTable[i].szTeX, "1") != NULL)
	       {
	       Remove1stArg(StrEnd, arg1);
               }
           if (strstr(FmlTransTable[i].szTeX, "2") != NULL)
	       {
	       Remove1stArg(StrEnd, arg2);
	       }


	   if(FmlTransTable[i].szTeX[0]=='\\' || FmlTransTable[i].szTeX[0]=='{')
			   StrBeg = cutspaces(StrBeg);
	   StrEnd = cutspaces(StrEnd);

	   Interpret( cq, StrBeg, StrEnd, FmlTransTable[i].szTeX,
		     FmlTransTable[i].szWP, arg0, arg1, arg2);
	   /*               writeln;
			    writeln(strbeg+strend);*/
	   }
      }

  StrBeg += StrEnd; /* Prepare equation for writing */
  if(error) StrBeg += "  % Error fixed by WP2LaTeX!";
  StrEnd = "";

}


boolean BoxTexHeader(TconvertedPass1 *cq, const TBox & Box)
{

  switch(Box.Type)
    {
    case 0:switch(Box.AnchorType)
	     {
	     case 0:fprintf(cq->strip, "\\begin{figure}[htbp]");
		    return(true);
	     case 1:fprintf(cq->strip, "\\begin{figure}[p]");
		    return(true);
	     case 2:if(Box.Contents==3)
			{                       // Graphics in the minipage
			fprintf(cq->strip,"%%");
			return(false);
			}
		    fprintf(cq->strip, "\\makebox[%2.2fcm]{",float(Box.Width)/470);
		    break;
	     default:fprintf(cq->strip, "\\begin{figure}[htbp]");
		     return(true);
	     }
	   break;
    case 1:switch(Box.AnchorType)
	     {
	     case 0:fprintf(cq->strip, "\\begin{table}[htbp]");
		    return(true);
	     case 1:fprintf(cq->strip, "\\begin{table}[p]");
		    return(true);
	     case 2:if(Box.Contents==3)
			{                       // Graphics in the minipage
			fprintf(cq->strip,"%%");
			return(false);
			}
		    fprintf(cq->strip, "\\makebox[%2.2fcm]{",float(Box.Width)/470);
		    break;
	     default:fprintf(cq->strip, "\\begin{table}[htbp]");
		     return(true);
	     }
	    break;
    default:if(Box.Contents==3)
			{                       // Graphics in the minipage
			fprintf(cq->strip,"%%");
			return(false);
			}
	    fprintf(cq->strip,"\\begin{minipage}");
	    if(Box.HorizontalPos==3) fprintf(cq->strip,"{\\textwidth}");
				else fprintf(cq->strip,"{%2.2fcm}",float(Box.Width)/470);
    }

return(false);
}

void BoxTexFoot(TconvertedPass1 *cq, const TBox & Box)
{
  switch(Box.Type)
    {
    case 0:switch(Box.AnchorType)
	     {
	     case 0:
	     case 1:fprintf(cq->strip, "\\end{figure}"); break;
	     case 2:if(Box.Contents==3)
			{                       // Graphics contents
			fprintf(cq->strip,"%%");
			break;
			}
		    fprintf(cq->strip, "}");		 break;
	     default:fprintf(cq->strip, "\\end{figure}");break;
	     }
	    break;
    case 1:switch(Box.AnchorType)
	     {
	     case 0:
	     case 1:fprintf(cq->strip, "\\end{table}");	break;
	     case 2:if(Box.Contents==3)
			{                       // Graphics contents
			fprintf(cq->strip,"%%");
			break;
			}
		    fprintf(cq->strip, "}");		break;
	     default:fprintf(cq->strip, "\\end{table}");break;
	     }
	    break;
    default:if(Box.Contents==3) fprintf(cq->strip,"%%");
			   else fprintf(cq->strip,"\\end{minipage}");
    }
}


/* End. */
