/*
 * Created on 24-Mar-2004
 */
package jmemorize.strategy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.prefs.Preferences;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.CategoryObserver;
import jmemorize.gui.swing.MainFrame;
import jmemorize.gui.swing.LearnPanel;
import jmemorize.util.PreferencesTool;

/**
 * @author djemili
 */
public class Strategy implements CategoryObserver //TODO separate strategy settings and and learn session
{
    // schedules
    public static final int      SCHEDULE_LEVELS  = 10;
    public static final String[] SCHEDULE_PRESETS = new String[] {
        "Constant", "Linear (Default)", "Quadratic", "Exponential", "Custom"
    };
    
    // side mode enums
    public static final int      SIDES_NORMAL              = 0;
    public static final int      SIDES_FLIPPED             = 1;
    public static final int      SIDES_RANDOM              = 2;

    // category order when grouping
    public static final int      CATEGORY_ORDER_FIXED      = 0;
    public static final int      CATEGORY_ORDER_RANDOM     = 1;

    // pref key strings
    private static String        PREFS_LIMIT_CARDS_ENABLED = "limit.cards.enabled";
    private static String        PREFS_LIMIT_TIME_ENABLED  = "limit.time.enabled";
    private static String        PREFS_LIMIT_CARDS         = "limit.cards";
    private static String        PREFS_LIMIT_TIME          = "limit.time";
    private static String        PREFS_RETEST_FAILED_CARDS = "retest-failed-cards";
    private static String        PREFS_SCHEDULE_PRESET     = "schedule.preset";
    private static String        PREFS_SCHEDULE            = "schedule.values";
    private static String        PREFS_SIDES               = "sides";
    private static String        PREFS_GROUP_BY_CATEGORY   = "card-order.group-by-category";
    private static String        PREFS_CATEGORY_ORDER      = "card-order.group-by-category.order";
	private static String        PREFS_SHUFFLE_CARDS       = "card-order.shuffle";
    
    // current learn session state
    private boolean    m_quit;
    private int        m_cardsChecked;       // number of cards that have been checked
    private int        m_cardsLearned;       // number of cards that have been known
    private Date       m_testDate;           // moment in which the learn session was started
    private List       m_cards;
    private Card       m_currentCard;
    private List       m_cardsCheckedList;   // all cards that have been checked
    private Map        m_categoryGroupOrder; // order by which categories appear when grouping enabled

    // swing related
    private MainFrame  m_frame;
    private LearnPanel m_repeatCardPanel;

    // learn strategy settings
    private Category   m_category;
    private boolean    m_learnUnlearnedCards;
    private boolean    m_learnExpiredCards;
    
    private int        m_schedulePreset;    // currently selected preset, -1 if custom
    private int[]      m_schedule;          // delay after level i in minutes
    private boolean    m_limitCardsEnabled;
    private boolean    m_limitTimeEnabled;
    private int        m_limitCards;
    private int        m_limitTime;         // time is given in minutes
    private boolean    m_retestFailedCards; // failed cards will be retested in this session
    private int        m_sides;
    private boolean    m_groupByCategory;
    private int        m_categoryOrder;     // order of categories when grouping enabled
	private boolean    m_shuffleCards;      // shuffle cards for every deck and category
    
    // etc
    private Preferences m_prefs;
    private Random      m_rand = new Random();
    
    
    private class CardComparator implements Comparator
    {
        /*
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2)
        {
            Card card0 = (Card)o1;
            Card card1 = (Card)o2;
            
            if (card0.getLevel() < card1.getLevel() )
            {
                return -1;
            }
            else if (card0.getLevel() > card1.getLevel() )
            {
                return 1;
            }
            // else card0.getLevel() == card1.getLevel()
            
            if (m_groupByCategory)
            {
                int cat0 = ((Integer)m_categoryGroupOrder.get(card0.getCategory())).intValue();
                int cat1 = ((Integer)m_categoryGroupOrder.get(card1.getCategory())).intValue();
                
                if (cat0 <  cat1)
                {
                    return -1;
                }
                else if (cat0 > cat1)
                {
                    return 1;
                }
            }
            // else no grouping or same category
            
            Date date0 = card0.getDateTouched();
            Date date1 = card1.getDateTouched();
            
            return (date0.before(date1) ? -1 : 1);
        }
    }
    
    public Strategy(Preferences prefs, MainFrame frame)
    {
        m_prefs = prefs;
        loadFromPreferences();
        
        m_frame = frame;
        // this is okay because frame wont change
        m_repeatCardPanel = frame.getLearnPanel();
    }
    
    public void setCards(Category category, boolean learnUnlearned, boolean learnExpired)
    {
        m_category = category;
        m_learnUnlearnedCards = learnUnlearned;
        m_learnExpiredCards = learnExpired;
    }
    
    /**
     * @return The card that is currently checked/shown.
     */
    public Card getCard() //TODO rename to getCurrentCard
    {
        return m_currentCard;
    }
    
    /**
     * @return All cards that are left to be learned in this session.
     */
    public List getCards()
    {
        return m_cards;
    }
    
    /** 
     * @return The category (subset of cards) that is currently being learned.
     */
    public Category getCategory()
    {
        return m_category;
    }
    
    public void setCardLimit(boolean enabled, int limit) 
    {
        m_limitCardsEnabled = enabled;
        m_limitCards = limit;
    }
    
    public boolean isCardLimitEnabled()
    {
        return m_limitCardsEnabled;
    }
    
    public int getCardLimit()
    {
        return m_limitCards;
    }
    
    /**
     * @param enabled True if the time limit should be obeyed.
     * @param limit The time limit in minutes.
     */
    public void setTimeLimit(boolean enabled, int limit) 
    {
        m_limitTimeEnabled = enabled;
        m_limitTime = limit;
    }
    
    public boolean isTimeLimitEnabled()
    {
        return m_limitTimeEnabled;
    }
    
    /**
     * @return The time limit in minutes.
     */
    public int getTimeLimit()
    {
        return m_limitTime;
    }
    
    /**
     * @param retest True if cards that have been failed while learning should 
     *          be put back into the list of cards to learn. False if all cards 
     *          should never be tested more then once in a session.
     */
    public void setRetestFailedCards(boolean retest)
    {
        m_retestFailedCards = retest;
    }
    
    
    /**
     * @see Strategy#setRetestFailedCards(boolean)
     * @return True if failed cards can appear more then once in a session ().
     */
    public boolean isRetestFailedCards()
    {
        return m_retestFailedCards;
    }
    
    public void setSides(int mode)
    {
        m_sides = mode;
    }
    
    /**
     * @return The current sides mode as given by enum SIDES_NORMAL, 
     * SIDES_FLIPPED and SIDES_RANDOM.
     */
    public int getSides()
    {
        return m_sides;
    }
    
    /**
     * The schedule tells how much time should pass before a cards that has
     * moved into a higher deck level needs be rechecked.
     * 
     * @param schedule A int array that holds the time span values that need 
     * to pass and where the index is the deck level before the card moved - 
     * e.g. <code>schedule[0] = 60</code> says that a card that has moved from 
     * deck 0 to deck 1 should be rechecked in one hour. The time spans are
     * given in minutes.
     */
    public void setCustomSchedule(int[] schedule)
    {
        m_schedule = schedule;
        m_schedulePreset = -1;
    }
    
    /**
     * @see Strategy#setSchedule(int[])
     * @return The current schedule.
     */
    public int[] getSchedule()
    {
        return m_schedule;
    }
    
    /**
     * @return The list of all cards that were checked in this learn session
     * until now. The list only contains unique cards, that is if a card was
     * checked more then once, it appears at the position of its last check. A
     * card is considered checked, when it is shown.
     */
    public List getCheckedCardsList()
    {
        return m_cardsCheckedList;
    }
    
    public void setSchedulePreset(int idx)
    {
        m_schedule = getPresetSchedule(idx);
        m_schedulePreset = idx;
    }
    
    /**
     * @return The index of the currently selected preset (see 
     * SCHEDULE_PRESETS) or -1 if the schedule is a custom one. 
     */
    public int getSchedulePreset()
    {
        return m_schedulePreset;
    }
    
    public void setGroupByCategory(boolean enable)
    {
        m_groupByCategory = enable;
    }
    
    public boolean isGroupByCategory()
    {
        return m_groupByCategory;
    }
    
    /**
     * This method sets the order by which categories will be shown when
     * grouping by categories is enabled.
     * 
     * @param order Is either CATEGORY_ORDER_CARDS or CATEGORY_ORDER_FIXED.  
     */
    public void setCategoryOrder(int order)
    {
        m_categoryOrder = order;
    }
    
    public int getCategoryOrder()
    {
        return m_categoryOrder;
    }

    public void setShuffleCards(boolean shuffle)
    {
        m_shuffleCards = shuffle;
    }

    public boolean isShuffleCards()
    {
        return m_shuffleCards;
    }
    
    public static int[] getPresetSchedule(int idx)
    {
        int schedule[] = new int[SCHEDULE_LEVELS];
        
        switch (idx)
        {
            case 0 : //constant
                for (int i = 0; i < SCHEDULE_LEVELS; i++)
                {
                    schedule[i] = 60 * 24;
                }
                return schedule; 
                
            case 1 : //linear
                for (int i = 0; i < SCHEDULE_LEVELS; i++)
                {
                    schedule[i] = (i+1) * 60 * 24;
                }
                return schedule;

            case 2 : //quadratic
                for (int i = 0; i < SCHEDULE_LEVELS; i++)
                {
                    schedule[i] = (int)Math.pow(i+1, 2) * 60 * 24;
                }
                return schedule;

            case 3 : //exponential
                for (int i = 0; i < SCHEDULE_LEVELS; i++)
                {
                    schedule[i] = (int)Math.pow(2, i) * 60 * 24;
                }
                return schedule;
        }
        
        throw new IllegalArgumentException("Preset schedule with this index not found.");
    }
    
    public void startLearning()
    {
        List cards = new ArrayList();
        if (m_learnUnlearnedCards)
        {
            cards.addAll(m_category.getUnlearnedCards());
        }
        if (m_learnExpiredCards)
        {
            cards.addAll(m_category.getExpiredCards());
        }
        
        m_category.addObserver(this);
        
        m_categoryGroupOrder = m_groupByCategory ? createCategoryGroupOrder() : null;
        
        m_quit = false;
        m_cardsChecked = 0;
        m_cardsLearned = 0;
        m_cardsCheckedList = new LinkedList();
        m_testDate = new Date();
        
        m_cards = cards;
        Collections.sort(m_cards, new CardComparator());
        
        m_repeatCardPanel.onStartLearning(this);
        m_frame.gotoLearnMode();
        
        checkNextCard();
    }
    
    public void endLearning()
    {
        m_category.removeObserver(this);
        m_repeatCardPanel.onEndLearning();
        m_frame.gotoBrowseMode();
    }

    public void cardChecked(boolean succesfull)
    {
        Card card = m_currentCard;
        m_cardsChecked++;
        
        if (succesfull)
        {
            m_cardsLearned++;
            
            long millis = m_testDate.getTime() + 
                60l * 1000l * m_schedule[Math.min(card.getLevel(), 9)];
            
            m_category.raiseCardLevel(card, m_testDate, new Date(millis));
        }
        else
        {
            if (m_retestFailedCards)
            {
                m_cards.add(card); // add again so that it remains part of this learn session
            }
            
            m_category.resetCardLevel(card, m_testDate);
        }
        
        // note that raising/reseting card level will be noticed by onCardEvent.
        // program flow continues there.
    }
    
    public void cardSkipped()
    {
        m_cards.add(m_currentCard);
        m_category.reappendCard(m_currentCard);
        
        // program flow continues in onCardEvent.
    }
    
    public void storeToPreferences()
    {
        m_prefs.putBoolean(PREFS_LIMIT_CARDS_ENABLED, m_limitCardsEnabled);
        m_prefs.putBoolean(PREFS_LIMIT_TIME_ENABLED, m_limitTimeEnabled);
        m_prefs.putInt(PREFS_LIMIT_CARDS, m_limitCards);
        m_prefs.putInt(PREFS_LIMIT_TIME, m_limitTime);
        m_prefs.putBoolean(PREFS_RETEST_FAILED_CARDS, m_retestFailedCards);
        m_prefs.putBoolean(PREFS_SHUFFLE_CARDS, m_shuffleCards);
        
        m_prefs.putInt(PREFS_SIDES, m_sides);
        
        m_prefs.putInt(PREFS_SCHEDULE_PRESET, m_schedulePreset);
        PreferencesTool.putIntArray(m_prefs, PREFS_SCHEDULE, m_schedule);
        
        m_prefs.putBoolean(PREFS_GROUP_BY_CATEGORY, m_groupByCategory);
        m_prefs.putInt(PREFS_CATEGORY_ORDER, m_categoryOrder);
    }
    
    public void onTimer()
    {
        m_quit = true;
    }
    
    /*
     * @see jmemorize.core.CategoryObserver#onCardEvent(int, jmemorize.core.Card, int)
     */
    public void onCardEvent(int type, Card card, int deck)
    {
        switch (type)
        {
            case ADDED_EVENT:
                m_cards.add(card);
                break;
                
            case REMOVED_EVENT:
                if (card == m_currentCard)
                {
                    checkNextCard();
                }
                else
                {
                    m_cards.remove(card);
                }
                
                m_cardsCheckedList.remove(card);
                break;
                
            case DECK_EVENT:
                if (card == m_currentCard)
                {
                    checkNextCard();
                }
                
                // not sure what to do with other cards that have been reset 
                // while learning. i'll just ignore this fact for now
                break;
        }
    }
    
    /*
     * @see jmemorize.core.CategoryObserver#onCategoryEvent(int, jmemorize.core.Category)
     */
    public void onCategoryEvent(int type, Category category)
    {
        // no category events should occure while learning.
        assert false;
    }
    
    private void checkNextCard()
    {
        // check for end condition
        if (m_quit || m_cards.size() == 0 || 
           (m_limitCardsEnabled && m_cardsLearned >= m_limitCards))
        {
            endLearning();
        }
        else
        {
            if(m_shuffleCards)
            {
                /*
                 * m_cards holds all cards that are to be learned ordered 
                 * by their deck (foremost), touch date (this is the last time 
                 * they appeared in a session) and category (the former 
                 * only if group_by_category is enabled.
                 * 
                 * Shuffling only works on the cards in the same deck and
                 * category (if group_by_category is enabled). Therefore we need
                 * to count all cards that are in consideration to be picked
                 * next and then randomly take one of them.
                 */
                
                Card firstCard = (Card)m_cards.get(0);
                int deck = firstCard.getLevel();
                Category category = firstCard.getCategory();
                
                int countNextCards = 0; // number of cards that are in the same deck
                for (Iterator it = m_cards.iterator(); it.hasNext();)
                {
                    Card card = (Card)it.next();
                    if (card.getLevel() == deck && 
                        (!m_groupByCategory || card.getCategory() == category))
                    {
                        countNextCards++;
                    }
                    else
                    {
                        break;
                    }
                } 
                
                m_currentCard = (Card)m_cards.remove(m_rand.nextInt(countNextCards));
			}
			else
			{
                m_currentCard = (Card)m_cards.remove(0);
            }
            
            boolean flippedMode;
            if (m_sides == SIDES_RANDOM)
            {
                flippedMode = m_rand.nextInt(2) == 1; // 50% chance
            }
            else
            {
                flippedMode = (m_sides == SIDES_FLIPPED);
            }
            
            // append card to checked cards list
            m_cardsCheckedList.remove(m_currentCard);
            m_cardsCheckedList.add(m_currentCard);
            
            m_repeatCardPanel.showCard(m_currentCard, flippedMode);
        }
    }
    
    /**
     * @return Cards that are being learned can be grouped by categories. In
     * this case the map holds for every category the position when it should
     * appear.
     */
    private Map createCategoryGroupOrder()
    {
        List categories = m_category.getSubtreeList();
        
        if (m_categoryOrder == CATEGORY_ORDER_RANDOM)
        {
            Collections.shuffle(categories);
        }
        
        HashMap map = new HashMap();
        int i = 0;
        for (Iterator it = categories.iterator(); it.hasNext();)
        {
            Category category = (Category)it.next();
            map.put(category, new Integer(i++));
        }
        
        return map;
    }
    
    private void loadFromPreferences()
    {
        // TODO load default values from properties
        m_limitCardsEnabled = m_prefs.getBoolean(PREFS_LIMIT_CARDS_ENABLED, false);
        m_limitTimeEnabled  = m_prefs.getBoolean(PREFS_LIMIT_TIME_ENABLED,  true);
        m_limitCards        = m_prefs.getInt(PREFS_LIMIT_CARDS, 20);
        m_limitTime         = m_prefs.getInt(PREFS_LIMIT_TIME, 20);
        m_retestFailedCards = m_prefs.getBoolean(PREFS_RETEST_FAILED_CARDS, true);
        
        m_sides             = m_prefs.getInt(PREFS_SIDES, SIDES_NORMAL);
        
        int preset          = m_prefs.getInt(PREFS_SCHEDULE_PRESET, 1); //linear as default
        if (preset > -1) // if preconfigured schedule
        {
            setSchedulePreset(preset);
        }
        else // if custom
        {
            int[] schedule = PreferencesTool.getIntArray(m_prefs, PREFS_SCHEDULE);
            if (schedule != null)
            {
                setCustomSchedule(schedule);
            }
            else
            {
                setSchedulePreset(1);
            }
        }
        
        m_groupByCategory   = m_prefs.getBoolean(PREFS_GROUP_BY_CATEGORY, true);
        m_categoryOrder     = m_prefs.getInt(PREFS_CATEGORY_ORDER, CATEGORY_ORDER_FIXED);
		m_shuffleCards      = m_prefs.getBoolean(PREFS_SHUFFLE_CARDS, true);
    }
}
