/*
 * Created on 23. Mrz 2004, 19:15
 */
package jmemorize.gui.swing;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.InputEvent;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.border.EmptyBorder;
import javax.swing.border.EtchedBorder;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.Main;
import jmemorize.util.EscapableFrame;

import com.jgoodies.forms.builder.ButtonBarBuilder;
import com.jgoodies.forms.builder.DefaultFormBuilder;
import com.jgoodies.forms.factories.Borders;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * @author  djemili
 */
public class NewCardManager
{
    private Dimension m_lastSize = new Dimension(400, 250);
    
    private List m_newCardFrames = new LinkedList();
    private List m_createdCards  = new ArrayList();
    
    public void addNewCardWindow(Category category)
    {
        NewCardFrame frame;
        if (m_newCardFrames.isEmpty())
        {
            frame = new NewCardFrame(this, null, m_lastSize, category);
            m_newCardFrames.add(0, frame); // insert at head
        }
        else
        {
            addNewCardWindow((NewCardFrame)m_newCardFrames.get(0), category);
        }
    }
    
    public void addNewCardWindow(NewCardFrame fatherFrame, Category category)
    {
        Point newLocation = new Point(fatherFrame.getX() + 25, fatherFrame.getY() + 25);
        NewCardFrame frame = new NewCardFrame(this, newLocation, 
            fatherFrame.getCardPanelSize(), category);
        
        // insert at head
        m_newCardFrames.add(0, frame);
    }
    
    public void editRecentlyCreatedCards()
    {
        if (!m_createdCards.isEmpty())
        {
            EditCardFrame.getInstance().showCard((Card)m_createdCards.get(m_createdCards.size() - 1), m_createdCards,
                Main.getInstance().getLesson().getRootCategory());
        }
    }
    
    public boolean closeAllFrames()
    {
        // first close all open newCardFrames
        List frames = new ArrayList(m_newCardFrames);
        for (Iterator it = frames.iterator(); it.hasNext();)
        {
            NewCardFrame frame = (NewCardFrame)it.next();
            if (!frame.close())
            {
                return false; // user canceled closing of this new card frame
            }
        }
        
        return true;
    }
    
    public Dimension getLastSize()
    {
        return m_lastSize;
    }

    public void onNewCardCreated(Card card)
    {
        m_createdCards.add(card);
    }
    
    public void onNewCardFrameClosed(NewCardFrame frame)
    {
        m_newCardFrames.remove(frame);
        
        if (m_newCardFrames.size() == 0)
        {
            m_lastSize = frame.getCardPanelSize();
        }
    }
}

class NewCardFrame extends EscapableFrame 
{
    private class NewWindowAction extends AbstractAction
    {
        public NewWindowAction()
        {
            putValue(NAME, "New window");
            putValue(MNEMONIC_KEY, new Integer(7));
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/card_add.gif")));
            putValue(SHORT_DESCRIPTION, "Open another new add card window");
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_N, InputEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e) 
        {
            m_manager.addNewCardWindow(NewCardFrame.this, m_categoryBox.getSelectedCategory());
        }
    }
    
    private class EditRecentCardAction extends AbstractAction
    {
        public EditRecentCardAction()
        {
            putValue(NAME, "Edit recently");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(SHORT_DESCRIPTION, "Edit recently created card");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/card_edit.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_ENTER, 0));
        }

        public void actionPerformed(java.awt.event.ActionEvent e) 
        {
            m_manager.editRecentlyCreatedCards();
        }
    }
    
    // swing elements
    private NewCardManager   m_manager;
    private CardSidesPanel   m_cardPanel         = new CardSidesPanel();
    private JButton          m_createMoreButton;
    private CategoryComboBox m_categoryBox       = new CategoryComboBox();
    private Action           m_editRecentAction  = new EditRecentCardAction();
    
    /**
     * Creates new form NewCardFrame 
     */
    public NewCardFrame(NewCardManager manager, Point location, Dimension cardPanelSize, Category currentCategory) 
    {
        m_manager = manager;
        
        initComponents();
        
        m_categoryBox.setRootCategory(Main.getInstance().getLesson().getRootCategory());
        m_categoryBox.setSelectedCategory(currentCategory);

        m_cardPanel.reset();
        m_cardPanel.setPreferredSize(cardPanelSize);
        pack();
        
        if (location == null)
        {
            setLocationRelativeTo(null);
        }
        else
        {
            setLocation(location);
        }
        
        setVisible(true);
    }
    
    /**
     * @return True if window was closed. False if it was left open by user decision.
     */
    public boolean close()
    {
        if (m_cardPanel.getFrontside().trim().length() > 0 || 
            m_cardPanel.getBackside().trim().length() > 0)
        {
            int n = JOptionPane.showConfirmDialog(this, "Card data is about to be dismissed.\n" +
                    "Do you want to create the card before the window is closed?",
                "Warning", JOptionPane.YES_NO_CANCEL_OPTION,  JOptionPane.WARNING_MESSAGE);

            if (n == JOptionPane.CANCEL_OPTION)
            {
                return false; //window isnt closed
            }
            
            if (n == JOptionPane.YES_OPTION)
            {
                return create(false);
            }
        }
        
        // if card was empty or NO was chosen
        closeWindow();
        return true;
    }
    
    /**
     * Needed to save size of this frame, because setPreferredSize(frame.getSize) doesnt work.
     */
    Dimension getCardPanelSize()
    {
        return m_cardPanel.getSize();
    }
    
    private void initComponents() 
    {        
        getContentPane().add(buildToolBar(),   java.awt.BorderLayout.NORTH);
        getContentPane().add(buildMainPanel(), java.awt.BorderLayout.CENTER);
        getContentPane().add(buildButtonBar(), java.awt.BorderLayout.SOUTH);
        
        getRootPane().setDefaultButton(m_createMoreButton);
        pack();
        
        setTitle("Create Card");
    }
    
    private JToolBar buildToolBar()
    {
        JToolBar toolBar = new JToolBar();
        toolBar.setFloatable(false);
        toolBar.add(new JButton(new NewWindowAction()));
        toolBar.add(new JButton(m_editRecentAction));
        
        return toolBar;
    }
    
    private JPanel buildMainPanel()
    {
        m_cardPanel.setBorder(Borders.DIALOG_BORDER);
        
        // build it using formlayout
        FormLayout layout = new FormLayout(
            "12px, 28dlu, 3dlu, default:grow, 13px", // columns
            "10px, p, fill:d:grow"); // rows
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        builder.setBorder(new EtchedBorder());        
        
        builder.addLabel("Category",       cc.xy ( 2, 2   ));
        builder.add(m_categoryBox,         cc.xy ( 4, 2   ));
        builder.add(m_cardPanel,           cc.xyw( 1, 3, 5));
        
        return builder.getPanel();
    }
    
    private JPanel buildButtonBar()
    {
        // buttons
        m_createMoreButton = new JButton("Add and create next");
        m_createMoreButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                create(true);
            }
        });

        JButton createButton = new JButton("Add");
        createButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                create(false);
            }
        });

        JButton cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                // directly close without confirmation
                closeWindow();
            }
        });
        
        // button bar
        CellConstraints cc = new CellConstraints();
        FormLayout layout = new FormLayout(
            "right:p:grow, p, 3dlu, 40dlu, 3dlu, 40dlu", // columns
            "p"); // rows

        ButtonBarBuilder builder = new ButtonBarBuilder();
        builder.addGlue();
        builder.addFixedNarrow(m_createMoreButton);
        builder.addRelatedGap();
        builder.addGridded(createButton);
        builder.addRelatedGap();
        builder.addGridded(cancelButton);
        builder.setBorder(new EmptyBorder(5, 5, 5, 5));
        
        return builder.getPanel();
    }
    
    /**
     * @return True if card was created.
     */
    private boolean create(boolean keepOpen) 
    {
        try
        {
            Card card = new Card(m_cardPanel.getFrontside(), m_cardPanel.getBackside());
            m_categoryBox.getSelectedCategory().addCard(card);
            m_manager.onNewCardCreated(card);
            m_cardPanel.reset();
            
            if (!keepOpen)
            {
                closeWindow();
            }

            return !keepOpen;
        }
        catch (IllegalArgumentException e)
        {
            JOptionPane.showMessageDialog(this, "You need to specify both card sides.", 
                "Bad card sides", JOptionPane.ERROR_MESSAGE);
            return false;
        }
    }
    
    private void closeWindow()
    {
        m_manager.onNewCardFrameClosed(this);
        dispose();
    }
}
