/*
 * Created on 23. Mrz 2004, 19:08
 */
package jmemorize.gui.swing;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.TransferHandler;
import javax.swing.UIManager;
import javax.swing.filechooser.FileFilter;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.CategoryObserver;
import jmemorize.core.Lesson;
import jmemorize.core.Main;
import jmemorize.strategy.Strategy;
import jmemorize.util.RecentItems;
import jmemorize.util.RecentItems.RecentItemsObserver;

/**
 * @author djemili
 */
public class MainFrame extends javax.swing.JFrame implements CategoryObserver, PropertyChangeListener,
    RecentItemsObserver
{
    // action classes
    private class NewLessonAction extends AbstractAction
    {
        public NewLessonAction()
        {
            putValue(NAME, "New");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/file_new.gif")));
            putValue(SHORT_DESCRIPTION, "Create empty lesson");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_N, InputEvent.CTRL_MASK
                + InputEvent.SHIFT_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            if (confirmCloseLesson())
            {
                m_main.createNewLesson();
            }
        }
    }

    private class OpenLessonAction extends AbstractAction
    {
        public OpenLessonAction()
        {
            putValue(NAME, "Open");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/file_open.gif")));
            putValue(SHORT_DESCRIPTION, "Open lesson file");
            putValue(MNEMONIC_KEY, new Integer('1'));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_O,
                java.awt.event.InputEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            if (confirmCloseLesson())
            {
                JFileChooser chooser = new JFileChooser();
                chooser.setCurrentDirectory(new File("."));
                chooser.setFileFilter(new LessonFileFilter());

                int returnVal = chooser.showOpenDialog(MainFrame.this);

                if (returnVal == JFileChooser.APPROVE_OPTION)
                {
                    File file = chooser.getSelectedFile();
                    loadLesson(file);
	
		    File ftemp = new File(m_main.getRecentFiles().get(0));
		    String rct = ftemp.getParent();
		    m_learnPanel.recent_lesson = rct + File.separator + "WAV" + File.separator;
                }
            }
        }
    }

    private class OpenRecentLessonAction extends AbstractAction
    {
        private int m_id;

        public OpenRecentLessonAction(int id)
        {
            m_id = id;
            File file = new File(m_main.getRecentFiles().get(id));

            putValue(NAME, (id + 1) + ". " + file.getName());
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/blank.gif")));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            if (confirmCloseLesson())
            {
                File file = new File(m_main.getRecentFiles().get(m_id));
                loadLesson(file);
            }
        }
    }

    private class SaveLessonAction extends AbstractAction
    {
        public SaveLessonAction()
        {
            putValue(NAME, "Save");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/file_save.gif")));
            putValue(SHORT_DESCRIPTION, "Save lesson");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_MASK));
        }

        public void save()
        {
            File file = m_main.getLesson().getFile();
            if (file != null)
            {
                saveLesson(file);
            }
            else
            {
                new SaveLessonAsAction().saveAs();
            }

            updateButtons();
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            save();
        }
    }

    private class SaveLessonAsAction extends AbstractAction
    {
        public SaveLessonAsAction()
        {
            putValue(NAME, "Save as");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/file_saveas.gif")));
            putValue(SHORT_DESCRIPTION, "Save lesson as..");
            putValue(MNEMONIC_KEY, new Integer(2));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_MASK
                + InputEvent.SHIFT_MASK));
        }

        public void saveAs()
        {
            JFileChooser chooser = new JFileChooser();
            chooser.setCurrentDirectory(new File("."));
            chooser.setFileFilter(new LessonFileFilter());

            int returnVal = chooser.showSaveDialog(MainFrame.this);
            if (returnVal == JFileChooser.APPROVE_OPTION)
            {
                File file = chooser.getSelectedFile();

                if (!file.getName().endsWith(".jml"))
                {
                    file = new File(file.getAbsolutePath() + ".jml");
                }

                saveLesson(file);

                updateFrameTitle();
                updateButtons();
            }
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            saveAs();
        }
    }

    private class AddCardAction extends AbstractAction
    {
        public AddCardAction()
        {
            putValue(NAME, "Add card");
            putValue(MNEMONIC_KEY, new Integer(7));
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/card_add.gif")));
            putValue(SHORT_DESCRIPTION, "Add new card");
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_N, InputEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            m_newCardManager.addNewCardWindow(m_currentCategory);
        }
    }

    private class EditCardAction extends AbstractAction
    {
        private List m_editCardFrames = new LinkedList();

        public EditCardAction()
        {
            putValue(NAME, "Edit card");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(SHORT_DESCRIPTION, "Edit and view card details");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/card_edit.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_ENTER, 0));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            if (m_mode == LEARNING_MODE)
            {
                Strategy strategy = m_main.getStrategy();

                EditCardFrame.getInstance().showCard(strategy.getCard(), strategy.getCheckedCardsList(),
                    strategy.getCategory());
            }
            else
            {
                m_deckTablePanel.editCards();
            }
        }
    }

    private class CopyAction extends AbstractAction
    {
        public CopyAction()
        {
            putValue(NAME, "Copy");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/edit_copy.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_C, KeyEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            CardCategoryTransferHandler.getCopyAction().actionPerformed(
                new ActionEvent(getEditFocusOwner(), ActionEvent.ACTION_PERFORMED, "copy"));
        }
    }

    private class CutAction extends AbstractAction
    {
        public CutAction()
        {
            putValue(NAME, "Cut");
            putValue(MNEMONIC_KEY, new Integer(2));
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/edit_cut.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_X, KeyEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            CardCategoryTransferHandler.getCutAction().actionPerformed(
                new ActionEvent(getEditFocusOwner(), ActionEvent.ACTION_PERFORMED, "cut"));
        }
    }

    private class PasteAction extends AbstractAction
    {
        public PasteAction()
        {
            putValue(NAME, "Paste");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/edit_paste.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_V, KeyEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            CardCategoryTransferHandler.getPasteAction().actionPerformed(
                new ActionEvent(getEditFocusOwner(), ActionEvent.ACTION_PERFORMED, "paste"));
        }
    }

    private class ResetCardAction extends AbstractAction
    {
        public ResetCardAction()
        {
            putValue(NAME, "Reset card");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(SHORT_DESCRIPTION, "Reset card statistics");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/card_reset.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_R, InputEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            int n = JOptionPane.showConfirmDialog(MainFrame.this, "You are about to reset " + m_focusedCards.size()
                + " card(s). Do you want to continue?", "Card Reset", JOptionPane.YES_NO_OPTION,
                JOptionPane.QUESTION_MESSAGE);

            if (n == JOptionPane.OK_OPTION)
            {
                for (Iterator it = m_focusedCards.iterator(); it.hasNext();)
                {
                    Card card = (Card)it.next();
                    m_currentCategory.resetCard(card);
                }
            }
        }
    }

    private class RemoveAction extends AbstractAction
    {
        public RemoveAction()
        {
            putValue(NAME, "Remove");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(SHORT_DESCRIPTION, "Remove currently selected item(s)");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/remove.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            if (m_focusedCards != null)
            {
                int n = JOptionPane.showConfirmDialog(MainFrame.this, "You are about to delete "
                    + m_focusedCards.size() + " card(s). Do you want to continue?", "Category Deletion",
                    JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

                if (n != JOptionPane.OK_OPTION)
                {
                    return;
                }

                for (Iterator it = m_focusedCards.iterator(); it.hasNext();)
                {
                    Card card = (Card)it.next();
                    m_currentCategory.removeCard(card);
                }
            }
            else
            // m_focusedCategories != null
            {
                // HACK only one category allowed

                for (Iterator it = m_focusedCategories.iterator(); it.hasNext();)
                {
                    Category category = (Category)it.next();
                    int nrCards = category.getCards().size();

                    if (nrCards > 0)
                    {
                        int n = JOptionPane.showConfirmDialog(MainFrame.this, "Deleting category " + category.getName()
                            + " will delete " + nrCards + " cards. Do you want to continue?", "Category Deletion",
                            JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

                        if (n != JOptionPane.OK_OPTION)
                        {
                            continue;
                        }
                    }

                    category.remove();
                }
            }
        }
    }

    private class ShowCategoryTreeAction extends AbstractAction
    {
        public ShowCategoryTreeAction()
        {
            putValue(NAME, "Category tree");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/tree.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_T, 0));
            putValue(SHORT_DESCRIPTION, "Show/Hide category tree");
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            showCategoryTree(!MainFrame.this.m_showTreeButton.isSelected());
        }
    }

    private class LearnAction extends AbstractAction
    {
        public LearnAction()
        {
            putValue(NAME, "Learn");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/learn.gif")));
            putValue(SHORT_DESCRIPTION, "Start lerning session with all unlearned and expired cards");
            putValue(MNEMONIC_KEY, new Integer(1));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_L, InputEvent.CTRL_MASK));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            m_learnSettingsFrame.show(m_main.getStrategy(), m_main.getLesson().getRootCategory(), m_currentCategory);
            MainFrame.this.setEnabled(false);
        }
    }

    private class AddCategoryAction extends AbstractAction
    {
        public AddCategoryAction()
        {
            putValue(NAME, "Add category");
            putValue(SHORT_DESCRIPTION, "Add new category");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/category_add.gif")));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            String name = JOptionPane.showInputDialog("Input the name of the new category that you want to create.");

            if (name != null && name.trim().length() > 0)
            {
                m_currentCategory.addCategoryChild(new Category(name.trim()));
            }
        }
    }

    private class FindAction extends AbstractAction
    {
        public FindAction()
        {
            putValue(NAME, "Find");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/find.gif")));
            putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_F,
                java.awt.event.InputEvent.CTRL_MASK));
            putValue(SHORT_DESCRIPTION, "Find cards");
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            m_findFrame.show(m_main.getLesson().getRootCategory(), m_currentCategory);
        }
    }

    private class ExitAction extends AbstractAction
    {
        public ExitAction()
        {
            putValue(NAME, "Exit");
            putValue(SHORT_DESCRIPTION, "Exit application");
            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/blank.gif")));
            putValue(MNEMONIC_KEY, new Integer(2));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            exit();
        }

        public void exit()
        {
            if (confirmCloseLesson())
            {
                // HACK
                Main.USER_PREFS.putInt(PREFS_TREE_WIDTH, m_showTreeButton.isSelected() ? 
                    m_horizontalSplitPane.getDividerLocation() : m_categoryTreeWidth);
                m_main.exit();
            }
        }
    }

    private class AboutAction extends AbstractAction
    {
        public AboutAction()
        {
            putValue(NAME, "About");
            putValue(SHORT_DESCRIPTION, "About this application");
            putValue(MNEMONIC_KEY, new Integer(1));
        }

        public void actionPerformed(java.awt.event.ActionEvent e)
        {
            String name = Main.PROPERTIES.getProperty("project.name");
            String version = Main.PROPERTIES.getProperty("project.version");
            String buildId = Main.PROPERTIES.getProperty("buildId");

            JOptionPane.showMessageDialog(MainFrame.this, "<html>" + "<b>" + name + " " + version + "</b> (" + buildId
                + ")<br><br>" + "Homepage:<br>http://www.riad.de/jmemorize<br><br>" + "2004-2005 Riad Djemili"
                + "</html>", "About", JOptionPane.INFORMATION_MESSAGE);
        }
    }

//    private class PreferencesAction extends AbstractAction
//    {
//        public PreferencesAction()
//        {
//            putValue(NAME, "Preferences..");
//            putValue(SMALL_ICON, new ImageIcon(getClass().getResource("/resource/icons/blank.gif")));
//            putValue(SHORT_DESCRIPTION, "Set your preferences.");
//        }
//
//        public void actionPerformed(java.awt.event.ActionEvent e)
//        {
//            // new PreferencesFrame(m_main, MainFrame.this); //Hack
//            // MainFrame.this.setEnabled(false);
//        }
//    }

    
    private class LessonFileFilter extends FileFilter
    {
        /*
         * @see javax.swing.filechooser.FileFilter#accept(java.io.File)
         */
        public boolean accept(File f)
        {
            return f.isDirectory() || f.getName().endsWith(".jml");
        }

        /*
         * @see javax.swing.filechooser.FileFilter#getDescription()
         */
        public String getDescription()
        {
            return "jMemorize Lessons";
        }

    }

    // TODO get date locale from settings
    public static final DateFormat      SHORT_DATE_FORMATER       = DateFormat.getDateTimeInstance(
        DateFormat.SHORT, DateFormat.SHORT);
    public static final DateFormat      LONG_DATE_FORMATER        = DateFormat.getDateTimeInstance(
        DateFormat.FULL,  DateFormat.MEDIUM);   // TODO.

    public static final TransferHandler TRANSFER_HANDLER          = new CardCategoryTransferHandler();

    private static final int            BROWSING_MODE             = 0;
    private static final int            LEARNING_MODE             = 1;

    // actions vars
    private AddCardAction               m_addCardAction           = new AddCardAction();
    private EditCardAction              m_editCardAction          = new EditCardAction();
    private ResetCardAction             m_resetCardAction         = new ResetCardAction();
    private ExitAction                  m_exitAction              = new ExitAction();
    private LearnAction                 m_learnAction             = new LearnAction();
    private NewLessonAction             m_newLessonAction         = new NewLessonAction();
    private OpenLessonAction            m_openLessonAction        = new OpenLessonAction();
    private RemoveAction                m_removeAction            = new RemoveAction();
    private SaveLessonAction            m_saveLessonAction        = new SaveLessonAction();
    private SaveLessonAsAction          m_saveLessonAsAction      = new SaveLessonAsAction();
    private ShowCategoryTreeAction      m_showCategoryTreeAction  = new ShowCategoryTreeAction();
    private AddCategoryAction           m_addCategoryAction       = new AddCategoryAction();
//    private PreferencesAction           m_preferencesAction       = new PreferencesAction();
    private CopyAction                  m_copyAction              = new CopyAction();
    private CutAction                   m_cutAction               = new CutAction();
    private PasteAction                 m_pasteAction             = new PasteAction();
    private FindAction                  m_findAction              = new FindAction();
    private AboutAction                 m_aboutAction             = new AboutAction();
    private List                        m_openRecentLessonActions = new ArrayList();

    // jmemorize swing elements
    private CategoryComboBox            m_categoryBox;
    private CategoryTree                m_categoryTree;
    private DeckTablePanel              m_deckTablePanel;
    private DeckChartPanel              m_lessonInfoPanel;
    private LearnPanel                  m_learnPanel;
    private StatusBar                   m_statusBar               = new StatusBar();
    private LearnSettingsFrame          m_learnSettingsFrame;
    private NewCardManager              m_newCardManager          = new NewCardManager();

    // native swing elements
    private JPanel                      m_bottomPanel;
    private JButton                     m_showTreeButton;
    private JSplitPane                  m_horizontalSplitPane;
    private JScrollPane                 m_treeScrollPane;
    private JMenu                       m_fileMenu;

    // frames
    private FindFrame                   m_findFrame               = new FindFrame();

    private Main                        m_main;
    private Category                    m_currentCategory;
    private int                         m_mode;
    
    private boolean                     m_showCategoryTree;
    private boolean                     m_showCategoryTreeOld;
    private int                         m_categoryTreeWidth       = Main.USER_PREFS.getInt(PREFS_TREE_WIDTH, 150);

    // either cards or categories can be focused, not both at the same time
    private List                        m_focusedCards; // = new LinkedList(); 
    private List                        m_focusedCategories;

    private final static String         PREFS_TREE_WIDTH          = "category-tree.width";

    // set look and feel before we load any frames
    static
    {
        try
        {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Creates new form MainFrame
     */
    public MainFrame()
    {
        m_main = Main.getInstance();
        m_main.getRecentFiles().addObserver(this);

        initComponents();
        setLocationRelativeTo(null);
        showCategoryTree(false);
        m_lessonInfoPanel.setFrame(this); // HACK
        m_lessonInfoPanel.setMinimumSize(new Dimension(100, 150));
        m_deckTablePanel.getCardTable().setStatusBar(m_statusBar);
        m_learnPanel.setStatusBar(m_statusBar);

        setLesson(m_main.getLesson()); // GUI is first loaded with empty lesson
        
        gotoBrowseMode();
    }

    public void setLesson(Lesson lesson)
    {
        Category rootCategory = lesson.getRootCategory();

        m_categoryBox.setRootCategory(rootCategory);
        m_categoryTree.setRootCategory(rootCategory);
        setCurrentCategory(rootCategory);

        EditCardFrame.getInstance().setVisible(false);

        updateFrameTitle();
    }

    public void setCurrentCategory(Category category)
    {
        if (category == null) // HACK
        {
            return;
        }

        if (m_currentCategory != null)
        {
            m_currentCategory.removeObserver(this);
        }
        m_currentCategory = category;
        m_currentCategory.addObserver(this);

        m_lessonInfoPanel.setCategory(category);
        m_deckTablePanel.setCategory(category); // TODO refactor. give only list of cards
        m_categoryBox.setSelectedCategory(category);
        m_categoryTree.setSelectedCategory(category);

        // in learn mode the focused item should always our currently learned card
        if (m_mode != LEARNING_MODE) // HACK
        {
            m_focusedCards = null;
            m_focusedCategories = new ArrayList(1); // HACK
            m_focusedCategories.add(category);
        }

        updateButtons();
    }

    public void showCategoryTree(boolean show)
    {
        if (!show)
        {
            if (m_showCategoryTree)
            {
                m_categoryTreeWidth = m_horizontalSplitPane.getDividerLocation();
            }
            
            m_horizontalSplitPane.setDividerSize(0);
            m_showTreeButton.setSelected(false);
            m_treeScrollPane.setVisible(false);
        }
        else
        {
            if (!m_showCategoryTree)
            {
                m_horizontalSplitPane.setDividerLocation(m_categoryTreeWidth);
            }
            
            m_showTreeButton.setSelected(true);
            m_treeScrollPane.setVisible(true);
            m_horizontalSplitPane.setDividerSize(5);
        }
             
        m_showCategoryTree = show;
    }

    public void startLearning()
    {
        m_showCategoryTreeOld = m_showCategoryTree;
        showCategoryTree(false);
        m_main.getStrategy().startLearning();
    }

    public DeckTablePanel getDeckPanel()
    {
        return m_deckTablePanel;
    }

    public LearnPanel getLearnPanel()
    {
        return m_learnPanel;
    }

    public void gotoBrowseMode()
    {
        ((CardLayout)m_bottomPanel.getLayout()).show(m_bottomPanel, "deckCard");

        m_focusedCategories = null;
        m_focusedCards = m_deckTablePanel.getCardTable().getSelectedCards();

        m_deckTablePanel.getCardTable().addPropertyChangeListener("selectedCards", this);
        m_learnPanel.removePropertyChangeListener("selectedCards", this);

        m_mode = BROWSING_MODE;
        updateButtons();
        
        showCategoryTree(m_showCategoryTreeOld);
    }

    public void gotoLearnMode()
    {
        ((CardLayout)m_bottomPanel.getLayout()).show(m_bottomPanel, "repeatCard");

        m_focusedCategories = null;
        m_focusedCards = m_learnPanel.getSelectedCards();

        m_learnPanel.addPropertyChangeListener("selectedCards", this);
        m_deckTablePanel.getCardTable().removePropertyChangeListener("selectedCards", this);
        
        m_deckTablePanel.showDeck(-1);//needed to get right values in status bar while learning 

        m_mode = LEARNING_MODE;
        updateButtons();
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCategoryEvent(int, jmemorize.core.Category)
     */
    public void onCategoryEvent(int type, Category category)
    {
        if (type == REMOVED_EVENT)
        {
            setCurrentCategory(m_main.getLesson().getRootCategory()); // HACK
        }

        updateButtons();
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCardEvent(int, jmemorize.core.Card, int)
     */
    public void onCardEvent(int type, Card card, int deck)
    {
        updateButtons();
    }

    /*
     * @see jmemorize.util.RecentItems.RecentItemsObserver#onRecentItemChange(jmemorize.util.RecentItems)
     */
    public void onRecentItemChange(RecentItems src)
    {
        buildFileMenu();
    }

    public void onPreferencesFrameClosed()
    {
        setEnabled(true);
    }

    /**
     * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
     */
    public void propertyChange(PropertyChangeEvent evt)
    {
        if (evt.getPropertyName().equals("selectedCards"))
        {
            m_focusedCards = (List)evt.getNewValue();
            m_focusedCategories = null;
            updateButtons();
        }
    }

    private void updateButtons()
    {
        assert !(m_focusedCards != null && m_focusedCategories != null) : "cards and categories cant be focused at the same time";

        m_newLessonAction.setEnabled(m_mode != LEARNING_MODE);
        m_openLessonAction.setEnabled(m_mode != LEARNING_MODE);

        for (Iterator it = m_openRecentLessonActions.iterator(); it.hasNext();)
        {
            Action openRecentLessonAction = (Action)it.next();
            openRecentLessonAction.setEnabled(m_mode != LEARNING_MODE);
        }

        m_saveLessonAction.setEnabled(m_mode != LEARNING_MODE && m_main.getLesson().isModified());
        m_saveLessonAsAction.setEnabled(m_mode != LEARNING_MODE);

        m_editCardAction.setEnabled(m_focusedCards != null && m_focusedCards.size() > 0);
        m_copyAction.setEnabled(m_mode != LEARNING_MODE && (m_focusedCards != null || m_focusedCategories != null));
        m_cutAction.setEnabled(m_mode != LEARNING_MODE && (m_focusedCards != null || m_focusedCategories != null));
        m_pasteAction.setEnabled(m_mode != LEARNING_MODE
            && (this.getToolkit().getSystemClipboard().getContents(null).isDataFlavorSupported(
                CardCategoryTransferHandler.CARDS_FLAVOR) || this.getToolkit().getSystemClipboard().getContents(null)
                .isDataFlavorSupported(CardCategoryTransferHandler.CATEGORY_FLAVOR)));

        m_removeAction.setEnabled((m_focusedCards != null && m_focusedCards.size() > 0)
            || (m_focusedCategories != null && m_main.getLesson() != null && !m_focusedCategories.contains(m_main
                .getLesson().getRootCategory()))); // HACK
        m_resetCardAction.setEnabled(m_focusedCards != null && m_focusedCards.size() > 0);

        m_addCategoryAction.setEnabled(m_mode != LEARNING_MODE);

        m_learnAction.setEnabled(m_mode != LEARNING_MODE && !m_currentCategory.getLearnableCards().isEmpty());

//        m_preferencesAction.setEnabled(m_mode != LEARNING_MODE);
        m_exitAction.setEnabled(m_mode != LEARNING_MODE);

        m_showCategoryTreeAction.setEnabled(m_mode != LEARNING_MODE);
    }

    public void loadLesson(File file)
    {
        try
        {
            m_main.loadLesson(file);
        }
        catch (IOException ex)
        {
            JOptionPane.showMessageDialog(MainFrame.this, "An error occured while loading " + file.getName() + "\n"
                + "Failed to load lesson.", "Error", JOptionPane.ERROR_MESSAGE);
        }
    }

    private void saveLesson(File file)
    {
        try
        {
            m_main.saveLesson(file);
        }
        catch (IOException e)
        {
            JOptionPane.showMessageDialog(MainFrame.this, "An error occured while saving " + file.getName() + "\n"
                + "Lesson has not been saved.", "Error", JOptionPane.ERROR_MESSAGE);

            e.printStackTrace();
        }
    }

    private void updateFrameTitle()
    {
        String name = Main.PROPERTIES.getProperty("project.name");
        String version = Main.PROPERTIES.getProperty("project.version");
        File file = m_main.getLesson().getFile();

        if (file != null && !getTitle().equals(file.getName()))
        {
            setTitle(file.getName() + " - " + name + " " + version);
        }
        else if (file == null)
        {
            setTitle("New lesson - " + name + " " + version);
        }
    }

    private JComponent getEditFocusOwner()
    {
        if (m_focusedCards != null)
        {
            return m_deckTablePanel.getCardTable();
        }
        else if (m_focusedCategories != null)
        {
            return m_categoryTree;
        }
        else
        {
            return null;
        }
    }

    /**
     * If lesson was modified this shows a dialog that asks if the user wants to
     * save the lesson before closing it.
     * 
     * @return true if user chose not to cancel the lesson close operation.
     */
    private boolean confirmCloseLesson()
    {
        // first check the editCardFrame for unsaved changes
        EditCardFrame editFrame = EditCardFrame.getInstance();
        if (editFrame.isVisible() && !editFrame.close())
        {
            return false; // user canceled closing of edit card frame
        }

        m_findFrame.setVisible(false);

        if (!m_newCardManager.closeAllFrames()) //close all addCard frames
        {
            return false;
        }

        // then see if lesson should to be saved
        if (m_main.getLesson().isModified())
        {
            int n = JOptionPane.showConfirmDialog(MainFrame.this,
                "Lesson has been modified. Do you want to save before exiting?", "Warning",
                JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);

            if (n == JOptionPane.OK_OPTION)
            {
                m_saveLessonAction.save(); // CHECK move method to mainframe!?
            }

            // if OK or NO chosen continue, otherwise CANCEL was chosen
            return (n == JOptionPane.OK_OPTION || n == JOptionPane.NO_OPTION);
        }

        return true;
    }

    private void initComponents()
    {
        JPanel mainPanel = new JPanel(new BorderLayout());

        m_lessonInfoPanel = new DeckChartPanel();
        m_learnPanel = new LearnPanel();

        m_deckTablePanel = new DeckTablePanel();
        m_learnSettingsFrame = new LearnSettingsFrame(this);

        // north panel
        JPanel northPanel = new JPanel();
        northPanel.setLayout(new javax.swing.BoxLayout(northPanel, javax.swing.BoxLayout.Y_AXIS));
        northPanel.add(buildOperationsBar());
        northPanel.add(buildCategoryBar());

        m_categoryTree = new CategoryTree();
        m_categoryTree.addTreeSelectionListener(new javax.swing.event.TreeSelectionListener() {
            public void valueChanged(javax.swing.event.TreeSelectionEvent evt)
            {
                selectionChanged(evt);
            }
        });
        m_treeScrollPane = new JScrollPane(m_categoryTree);

        // bottom panel
        m_bottomPanel = new JPanel(new java.awt.CardLayout());
        m_bottomPanel.add(m_deckTablePanel, "deckCard");
        m_bottomPanel.add(m_learnPanel, "repeatCard");

        // vertical split pane
        JSplitPane verticalSplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        verticalSplitPane.setDividerLocation(200);
        verticalSplitPane.setOneTouchExpandable(true);
        verticalSplitPane.setPreferredSize(new java.awt.Dimension(16, 500));

        verticalSplitPane.setTopComponent(m_lessonInfoPanel);
        verticalSplitPane.setBottomComponent(m_bottomPanel);

        mainPanel.setPreferredSize(new java.awt.Dimension(800, 500));
        mainPanel.add(verticalSplitPane, BorderLayout.CENTER);

        // horizontal split pane
        m_horizontalSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        m_horizontalSplitPane.setDividerLocation(m_categoryTreeWidth);
        m_horizontalSplitPane.setDividerSize(4);

        m_horizontalSplitPane.setLeftComponent(m_treeScrollPane);
        m_horizontalSplitPane.setRightComponent(mainPanel);

        // frame content pane
        getContentPane().add(northPanel, BorderLayout.NORTH);
        getContentPane().add(m_horizontalSplitPane, BorderLayout.CENTER);
        getContentPane().add(m_statusBar, BorderLayout.SOUTH);
        setJMenuBar(buildMenu());

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                exitForm(evt);
            }
        });

        pack();
    }

    private JPanel buildCategoryBar()
    {
        JToolBar categoryToolbar = new JToolBar();
        categoryToolbar.setFloatable(false);
        categoryToolbar.setMargin(new java.awt.Insets(2, 2, 2, 2));

        m_showTreeButton = new JButton(m_showCategoryTreeAction);
        m_showTreeButton.setPreferredSize(new java.awt.Dimension(100, 21));
        categoryToolbar.add(m_showTreeButton);

        JSeparator separator = new JSeparator(javax.swing.SwingConstants.VERTICAL);
        separator.setMaximumSize(new java.awt.Dimension(20, 32767));
        separator.setPreferredSize(new java.awt.Dimension(6, 0));
        categoryToolbar.add(separator);

        JLabel categoryLabel = new JLabel("Category", javax.swing.SwingConstants.CENTER);
        categoryLabel.setPreferredSize(new java.awt.Dimension(60, 15));
        categoryToolbar.add(categoryLabel);

        m_categoryBox = new CategoryComboBox();
        m_categoryBox.setPreferredSize(new java.awt.Dimension(24, 24));
        m_categoryBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                categoryBoxActionPerformed();
            }
        });
        categoryToolbar.add(m_categoryBox);

        JPanel categoryPanel = new JPanel(new java.awt.BorderLayout());
        categoryPanel.setBorder(new javax.swing.border.EtchedBorder());
        categoryPanel.add(categoryToolbar, java.awt.BorderLayout.NORTH);

        return categoryPanel;
    }

    private JPanel buildOperationsBar()
    {
        JToolBar operationsToolbar = new JToolBar();
        operationsToolbar.setFloatable(false);

        operationsToolbar.add(new JButton(m_newLessonAction));
        operationsToolbar.add(new JButton(m_openLessonAction));
        operationsToolbar.add(new JButton(m_saveLessonAction));

        operationsToolbar.add(new JButton(m_addCardAction));
        operationsToolbar.add(new JButton(m_editCardAction));
        operationsToolbar.add(new JButton(m_resetCardAction));
        operationsToolbar.add(new JButton(m_removeAction));

        operationsToolbar.add(new JButton(m_addCategoryAction));
        operationsToolbar.add(new JButton(m_findAction));
        operationsToolbar.add(new JButton(m_learnAction));

        JPanel operationsPanel = new JPanel(new java.awt.FlowLayout(java.awt.FlowLayout.LEFT, 1, 1));
        operationsPanel.add(operationsToolbar);

        return operationsPanel;
    }

    private JMenuBar buildMenu()
    {
        JMenuBar menuBar = new JMenuBar();

        m_fileMenu = new JMenu("File");
        menuBar.add(m_fileMenu);
        buildFileMenu();

        JMenu editMenu = new JMenu("Edit");
        editMenu.add(new JMenuItem(m_editCardAction));
        editMenu.addSeparator();
        editMenu.add(new JMenuItem(m_cutAction));
        editMenu.add(new JMenuItem(m_copyAction));
        editMenu.add(new JMenuItem(m_pasteAction));
        editMenu.addSeparator();
        editMenu.add(new JMenuItem(m_removeAction));
        editMenu.add(new JMenuItem(m_resetCardAction));
        editMenu.addSeparator();
        editMenu.add(new JMenuItem(m_findAction));
        menuBar.add(editMenu);

        JMenu learnMenu = new JMenu("Lesson");
        learnMenu.add(new JMenuItem(m_addCardAction));
        learnMenu.add(new JMenuItem(m_addCategoryAction));
        learnMenu.addSeparator();
        learnMenu.add(new JMenuItem(m_learnAction));
        menuBar.add(learnMenu);

        JMenu helpMenu = new JMenu("Help");
        helpMenu.add(new JMenuItem(m_aboutAction));
        menuBar.add(helpMenu);

        return menuBar;
    }

    private void buildFileMenu()
    {
        m_fileMenu.removeAll();
        m_fileMenu.add(new JMenuItem(m_newLessonAction));
        m_fileMenu.add(new JMenuItem(m_openLessonAction));
        m_fileMenu.add(new JMenuItem(m_saveLessonAction));
        m_fileMenu.add(new JMenuItem(m_saveLessonAsAction));

        // add recent files menu items
        int recentFiles = m_main.getRecentFiles().size();

        if (recentFiles > 0)
        {
            m_fileMenu.addSeparator();
        }

        for (int i = 0; i < recentFiles; i++)
        {
            Action action = new OpenRecentLessonAction(i);
            m_openRecentLessonActions.add(action);
            JMenuItem menuItem = new JMenuItem(action);

            m_fileMenu.add(menuItem);
        }

//        m_fileMenu.addSeparator();
//        m_fileMenu.add(new JMenuItem(m_preferencesAction));
        m_fileMenu.addSeparator();
        m_fileMenu.add(new JMenuItem(m_exitAction));
    }

    private void categoryBoxActionPerformed()
    {
        setCurrentCategory(m_categoryBox.getSelectedCategory());
    }

    private void selectionChanged(javax.swing.event.TreeSelectionEvent evt)
    {
        setCurrentCategory(m_categoryTree.getSelectedCategory());
    }

    /** Exit the Application */
    private void exitForm(java.awt.event.WindowEvent evt)
    {
        m_exitAction.exit();
    }
}
