/*
 * Created on 29.11.2004
 */
package jmemorize.gui.swing;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jmemorize.core.Category;
import jmemorize.strategy.Strategy;

import com.jgoodies.forms.builder.DefaultFormBuilder;
import com.jgoodies.forms.builder.PanelBuilder;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * @author djemili
 */
public class LearnSettingPanels extends JFrame
{
    // general panel
    private JSpinner         m_cardLimitSpinner          = new JSpinner(new SpinnerNumberModel(20, 1, 300, 1));
    private JSpinner         m_timeLimitSpinner          = new JSpinner(new SpinnerNumberModel(20, 1, 300, 1));
    private boolean          m_updatingSpinners          = false;    

    // card list panel
    private CategoryComboBox m_categoryComboBox          = new CategoryComboBox();
    private JRadioButton     m_allCardsButton            = new JRadioButton(
        "Learn all unlearned and expired cards.", true);
    private JRadioButton     m_unlearnedCardsButton      = new JRadioButton(
        "Learn only unlearned cards.");
    private JRadioButton     m_expiredCardsButton        = new JRadioButton(
        "Learn only expired cards.");
    
    // limiter panel
    private JCheckBox        m_timeLimitCheckBox         = new JCheckBox(
        "End session when a certain time limit has been reached.");
    private JCheckBox        m_cardLimitCheckBox         = new JCheckBox(
        "End session when a certain card limit has been reached.");
    private JCheckBox        m_dontRetestCheckBox        = new JCheckBox(
        "Dont retest cards that have been failed in this session.");
    
    // side mods panel
    private JRadioButton     m_sidesNormalButton         = new JRadioButton(
        "Learn in normal mode.");
    private JRadioButton     m_sidesFlippedButton        = new JRadioButton(
        "Learn with flipped card sides.");
    private JRadioButton     m_sidesRandomButton         = new JRadioButton(
        "Learn in random mode with card sides being randomly flipped.");
    
    // card order panel
    private JCheckBox        m_categoryGroupsCheckBox    = new JCheckBox(
        "Group cards by categories while learning.");
    private JRadioButton     m_categoryOrderFixedButton  = new JRadioButton(
        "Show categories in fixed natural order.");
    private JRadioButton     m_categoryOrderRandomButton = new JRadioButton(
        "Show categories in random order.");
	private JCheckBox        m_shuffleCardsCheckBox       = new JCheckBox(
        "Shuffle cards of all decks and categories.");
        
    // schedule panel
    private JSpinner[]       m_scheduleDays              = new JSpinner[Strategy.SCHEDULE_LEVELS];
    private JSpinner[]       m_scheduleHours             = new JSpinner[Strategy.SCHEDULE_LEVELS];
    private JSpinner[]       m_scheduleMinutes           = new JSpinner[Strategy.SCHEDULE_LEVELS];
    private JComboBox        m_schedulePresetsComboBox   = new JComboBox(Strategy.SCHEDULE_PRESETS);
    
    // other
    private Strategy         m_strategy;

    public LearnSettingPanels()
    {
        for (int level = 0; level<Strategy.SCHEDULE_LEVELS; level++)
        {
            m_scheduleDays[level]    = new JSpinner(new SpinnerNumberModel(1, 0, 999, 1));
            m_scheduleHours[level]   = new JSpinner(new SpinnerNumberModel(1, 0, 23 , 1));
            m_scheduleMinutes[level] = new JSpinner(new SpinnerNumberModel(1, 0, 59 , 1));
        }
        
        m_categoryComboBox.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e)
            {
                categoryBoxActionPerformed();
            }
        });
    }
    
    /**
     * @param strategy The strategy that is shown and editted by the panels.
     */
    public void setStrategy(Strategy strategy)
    {
        m_strategy = strategy;
        
        resetSettings();
    }
    
    public void setCategory(Category rootCategory, Category category)
    {
        m_categoryComboBox.setRootCategory(rootCategory);
        m_categoryComboBox.setSelectedCategory(category);
        
        updateCardButtons();
    }
    
    /**
     * Sets all learn settings widgets according to the settings of the current
     * strategy.
     */
    public void resetSettings()
    {
        // get general settings
        m_cardLimitCheckBox.setSelected(m_strategy.isCardLimitEnabled());
        m_timeLimitCheckBox.setSelected(m_strategy.isTimeLimitEnabled());
        updateLimiterCheckboxes();
        
        m_dontRetestCheckBox.setSelected(!m_strategy.isRetestFailedCards());
        
        m_cardLimitSpinner.setValue(new Integer(m_strategy.getCardLimit()));
        m_timeLimitSpinner.setValue(new Integer(m_strategy.getTimeLimit()));
        
        switch (m_strategy.getSides())
        {
            case Strategy.SIDES_NORMAL:
                m_sidesNormalButton.setSelected(true);
                break;
            case Strategy.SIDES_FLIPPED:
                m_sidesFlippedButton.setSelected(true);
                break;
            default: // SIDES.RANDOM
                m_sidesRandomButton.setSelected(true);
                break;
        }
        
        // get schedule
        int preset = m_strategy.getSchedulePreset();
        m_schedulePresetsComboBox.setSelectedIndex(preset > -1 ? preset : 4); // custom (4) if no preset
        updateScheduleSpinners(m_strategy.getSchedule());
        
        // get category settings
        m_categoryGroupsCheckBox.setSelected(m_strategy.isGroupByCategory());
        
        if (m_strategy.getCategoryOrder() == Strategy.CATEGORY_ORDER_FIXED)
        {
            m_categoryOrderFixedButton.setSelected(true);
        }
        else
        {
            m_categoryOrderRandomButton.setSelected(true);
        }
        updateCategoryOrderButtons();
		
		m_shuffleCardsCheckBox.setSelected(m_strategy.isShuffleCards());
    }
    
    /**
     * Applies all settings currently entered into the learn settings panels to 
     * the current strategy.
     */
    public void applySettings()
    {
        // apply general settings
        m_strategy.setCards(getSelectedCategory(),
            m_allCardsButton.isSelected() || m_unlearnedCardsButton.isSelected(),
            m_allCardsButton.isSelected() || m_expiredCardsButton.isSelected()
        );
        m_strategy.setCardLimit(m_cardLimitCheckBox.isSelected(), intValue(m_cardLimitSpinner));
        m_strategy.setTimeLimit(m_timeLimitCheckBox.isSelected(), intValue(m_timeLimitSpinner));
        m_strategy.setRetestFailedCards(!m_dontRetestCheckBox.isSelected());
        
        m_strategy.setSides(m_sidesNormalButton.isSelected() ? Strategy.SIDES_NORMAL : 
            m_sidesFlippedButton.isSelected() ? Strategy.SIDES_FLIPPED : Strategy.SIDES_RANDOM
        );
        
        // apply schedule
        int preset = m_schedulePresetsComboBox.getSelectedIndex();
        if (preset < 4)
        {
            m_strategy.setSchedulePreset(preset);
        }
        else
        {
            // schedule holds the time spans in minutes
            int[] schedule = new int[Strategy.SCHEDULE_LEVELS];
            for (int i = 0; i < Strategy.SCHEDULE_LEVELS; i++)
            {
                schedule[i] = (24 * 60 * intValue(m_scheduleDays[i]))
                    + (60 * intValue(m_scheduleHours[i])) + intValue(m_scheduleMinutes[i]);
            }
            m_strategy.setCustomSchedule(schedule);
        }
        
        // apply category settings
        m_strategy.setGroupByCategory(m_categoryGroupsCheckBox.isSelected());
        m_strategy.setCategoryOrder(m_categoryOrderRandomButton.isSelected() ?
            Strategy.CATEGORY_ORDER_RANDOM : Strategy.CATEGORY_ORDER_FIXED);
        m_strategy.setShuffleCards(m_shuffleCardsCheckBox.isSelected());
    }
    
    public Category getSelectedCategory()
    {
        return m_categoryComboBox.getSelectedCategory();
    }
    
    public JPanel buildCardsPanel()
    {
        // prepare widgets
        ButtonGroup group = new ButtonGroup();
        group.add(m_allCardsButton);
        group.add(m_unlearnedCardsButton);
        group.add(m_expiredCardsButton);
        
        // build panel
        FormLayout layout = new FormLayout(
            "p:grow",       // columns
            "p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p");  // stop condition rows
        
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        builder.addSeparator("Cards to learn",      cc.xy( 1,  1));
        builder.add(m_categoryComboBox,             cc.xy( 1,  3));
        builder.add(m_allCardsButton,               cc.xy( 1,  5));
        builder.add(m_unlearnedCardsButton,         cc.xy( 1,  7));
        builder.add(m_expiredCardsButton,           cc.xy( 1,  9));
        
        return builder.getPanel();
    }
    
    public JPanel buildCardOrderPanel()
    {
        m_categoryGroupsCheckBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0)
            {
                updateCategoryOrderButtons();
            }
        });
        
        ButtonGroup categoriesGroup = new ButtonGroup();
        categoriesGroup.add(m_categoryOrderFixedButton);
        categoriesGroup.add(m_categoryOrderRandomButton);
        
        // build panel
        FormLayout layout = new FormLayout(
            "18dlu, p:grow",                                 // columns
            "p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p");        // grouping rows
        
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        
        builder.addSeparator("Cards order",          cc.xyw( 1,  1, 2));
        builder.add(m_categoryGroupsCheckBox,        cc.xyw( 1,  3, 2));
        builder.add(m_categoryOrderFixedButton,      cc.xy ( 2,  5  ));
        builder.add(m_categoryOrderRandomButton,     cc.xy ( 2,  7  ));
		builder.add(m_shuffleCardsCheckBox,          cc.xyw( 1,  9, 2));
        
        return builder.getPanel();
    }
    
    public JPanel buildSidesModePanel()
    {
        // radio button groups
        ButtonGroup sidesModeGroup = new ButtonGroup();
        sidesModeGroup.add(m_sidesNormalButton);
        sidesModeGroup.add(m_sidesFlippedButton);
        sidesModeGroup.add(m_sidesRandomButton);
        m_sidesNormalButton.setSelected(true);
        
        // build panel
        FormLayout layout = new FormLayout(
            "p:grow",                                       // columns
            "p, 3dlu, p, 3dlu, p, 3dlu, p");          // side mode rows
        
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        
        builder.addSeparator("Side mode",            cc.xy( 1,  1));
        builder.add(m_sidesNormalButton,             cc.xy( 1,  3));
        builder.add(m_sidesFlippedButton,            cc.xy( 1,  5));
        builder.add(m_sidesRandomButton,             cc.xy( 1,  7));
        
        return builder.getPanel();
    }
    
    public JPanel buildLimiterPanel()
    {
        // add action listeners
        m_cardLimitCheckBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e)
            {
                updateLimiterCheckboxes();
            }
        });
        m_timeLimitCheckBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e)
            {
                updateLimiterCheckboxes();
            }
        });
        updateLimiterCheckboxes();
        
        // build panel
        FormLayout layout = new FormLayout(
            "18dlu, 35dlu, left:d:grow",                        // columns
            "p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p");  // stop condition rows
        
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        
        builder.addSeparator("Delimiters",           cc.xyw( 1,  1, 3));
        builder.add(m_timeLimitCheckBox,             cc.xyw( 1,  3, 3));
        builder.addLabel("Minute limit",             cc.xy ( 2,  5   ));
        builder.add(m_timeLimitSpinner,              cc.xy ( 3,  5   ));
        
        builder.add(m_cardLimitCheckBox,             cc.xyw( 1,  7, 3));
        builder.addLabel("Card limit",               cc.xy ( 2,  9   ));
        builder.add(m_cardLimitSpinner,              cc.xy ( 3,  9   ));
        
        builder.add(m_dontRetestCheckBox,            cc.xyw( 1, 11, 3));
        
        return builder.getPanel();
    }
    
    public JPanel buildSchedulePanel()
    {
        // prepare widgets
        m_schedulePresetsComboBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e)
            {
                onPresetScheduleSelected();
            }
        });
        
        // build panel
        FormLayout layout = new FormLayout(
            "p, 20dlu:grow, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, " +// columns
            "p,  3dlu,  p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p", 
            "p, 15dlu,  p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, " + // rows
            "p,  3dlu,  p, 3dlu, p, 3dlu, p, 3dlu"); 
        
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        builder.setDefaultDialogBorder();
        
        builder.addLabel("Schedule Presets",   cc.xy(1, 1));
        builder.add(m_schedulePresetsComboBox, cc.xyw(3, 1, 11));
        
        for (int i = 0; i < Strategy.SCHEDULE_LEVELS; i++)
        {
            addScheduleRow(builder, cc, i);
        }
        
        return builder.getPanel();
    }
    
    private void categoryBoxActionPerformed()
    {
        updateCardButtons();
    }
    
    private void addScheduleRow(PanelBuilder builder, CellConstraints cc, int level)
    {
        m_scheduleDays[level].addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e)
            {
                spinnerValueChanged();
            }
        });
        m_scheduleHours[level].addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e)
            {
                spinnerValueChanged();
            }
        });
        m_scheduleMinutes[level].addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e)
            {
                spinnerValueChanged();
            }
        });
        
        builder.addLabel("Delay after level "+level, cc.xy ( 1, 3+2*level));
        builder.add(m_scheduleDays[level],           cc.xy ( 3, 3+2*level));
        builder.addLabel("Days",                     cc.xy ( 5, 3+2*level));
        builder.add(m_scheduleHours[level],          cc.xy ( 7, 3+2*level));
        builder.addLabel("Hours",                    cc.xy ( 9, 3+2*level));
        builder.add(m_scheduleMinutes[level],        cc.xy (11, 3+2*level));
        builder.addLabel("Minutes",                  cc.xy (13, 3+2*level));
    }
    
    private void updateLimiterCheckboxes()
    {
        m_timeLimitSpinner.setEnabled(m_timeLimitCheckBox.isSelected());
        m_cardLimitSpinner.setEnabled(m_cardLimitCheckBox.isSelected());
    }
    
    private void updateCategoryOrderButtons()
    {
        m_categoryOrderRandomButton.setEnabled(m_categoryGroupsCheckBox.isSelected());
        m_categoryOrderFixedButton.setEnabled(m_categoryGroupsCheckBox.isSelected());
    }
    
    private void updateCardButtons()
    {
        boolean enableUnlearned = !getSelectedCategory().getUnlearnedCards().isEmpty();
        boolean enableExpired = !getSelectedCategory().getExpiredCards().isEmpty();
        
        m_unlearnedCardsButton.setEnabled(enableUnlearned);
        m_expiredCardsButton.setEnabled(enableExpired);
        
        if ((m_unlearnedCardsButton.isSelected() && !enableUnlearned) ||
            (m_expiredCardsButton.isSelected()   && !enableExpired))
        {
            m_allCardsButton.setSelected(true);
        }
    }
    
    private void onPresetScheduleSelected()
    {
        int idx = m_schedulePresetsComboBox.getSelectedIndex();
        
        if (idx < 4) // if not custom
        {
            int[] schedule = Strategy.getPresetSchedule(idx);
            updateScheduleSpinners(schedule);
        }
    }
    
    private void updateScheduleSpinners(int[] schedule)
    {
        m_updatingSpinners = true;
        
        for (int i = 0; i < Strategy.SCHEDULE_LEVELS; i++)
        {
            m_scheduleDays[i].setValue(new Integer(schedule[i] / (60 * 24) ));
            m_scheduleHours[i].setValue(new Integer((schedule[i] % (60 * 24)) / 60 ));
            m_scheduleMinutes[i].setValue(new Integer(schedule[i] % 60));
        }
        
        m_updatingSpinners = false;
    }
    
    private void spinnerValueChanged()
    {
        // just to make sure that the value was changed by a user and not 
        // by choosing from the presets combobox.
        if (!m_updatingSpinners)
        {
            m_schedulePresetsComboBox.setSelectedIndex(4);
        }
    }
    
    private int intValue(JSpinner spinner)
    {
        return ((SpinnerNumberModel)spinner.getModel()).getNumber().intValue();
    }
}
