/*
 * Created on 29-Oct-2004
 */
package jmemorize.gui.swing;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.border.EmptyBorder;
import javax.swing.border.EtchedBorder;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.CategoryObserver;
import jmemorize.core.Main;
import jmemorize.core.SearchTool;
import jmemorize.core.Events;
import jmemorize.util.EscapableFrame;
import jmemorize.util.RecentItems;

import com.jgoodies.forms.builder.DefaultFormBuilder;
import com.jgoodies.forms.factories.ButtonBarFactory;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * @author djemili
 */
public class FindFrame extends EscapableFrame implements CategoryObserver
{
    private CardTable        m_cardTable         = new CardTable(Main.USER_PREFS.node("find.table"),new int[]{
        CardTable.COLUMN_FRONTSIDE, 
        CardTable.COLUMN_BACKSIDE, 
        CardTable.COLUMN_CATEGORY});
    
    // swing widgets
    private JComboBox        m_searchTextBox     = new JComboBox();
    private RecentItems      m_recentSearchTexts = new RecentItems(
        10, Main.USER_PREFS.node("recent.search.texts"));

    private JCheckBox        m_matchCaseBox      = new JCheckBox("Match case");
    private JRadioButton     m_radioBothSides    = new JRadioButton("Both sides", true);
    private JRadioButton     m_radioFrontSide    = new JRadioButton("Frontside");
    private JRadioButton     m_radioBackSide     = new JRadioButton("Backside");
    private CategoryComboBox m_categoryBox       = new CategoryComboBox();
    
    private StatusBar        m_statusBar         = new StatusBar();
    
    // these vars are stored when search button is clicked
    private String           m_searchText;
    private int              m_searchSides;
    private boolean          m_matchCase;
    private Category         m_searchCategory;
    
    private class FindAction extends AbstractAction
    {
        public FindAction()
        {
            putValue(NAME, "Find");
        }

        public void actionPerformed(java.awt.event.ActionEvent e) 
        {
            String searchText = (String)m_searchTextBox.getSelectedItem();
            
            if (!searchText.equals(""))
            {
                m_searchText       = searchText;
                m_recentSearchTexts.push(m_searchText);

                if (m_radioBothSides.isSelected())
                {
                    m_searchSides = SearchTool.BOTH_SIDES; 
                }
                else
                {
                    m_searchSides = m_radioFrontSide.isSelected() ? 
                        SearchTool.FRONT_SIDE : SearchTool.FLIP_SIDE; 
                }
                    
                if (m_searchCategory != null)
                {
                    m_searchCategory.removeObserver(FindFrame.this);
                }
                m_searchCategory = m_categoryBox.getSelectedCategory();
                m_searchCategory.addObserver(FindFrame.this);
                
                m_matchCase = m_matchCaseBox.isSelected();
                
                m_cardTable.getView().setCards(SearchTool.search(
                    m_searchText, m_searchSides, m_matchCase, m_searchCategory.getCards()), m_searchCategory);
            }
        }
    }
    
    private class CloseAction extends AbstractAction
    {
        public CloseAction()
        {
            putValue(NAME, "Close");
        }

        public void actionPerformed(java.awt.event.ActionEvent e) 
        {
            close();
        }
    }
    
    public FindFrame()
    {
        initComponents();
        
        setLocationRelativeTo(null);
    }
    
    public void show(Category rootCategory, Category selectedCategory)
    {
        m_categoryBox.setRootCategory(rootCategory);
        m_categoryBox.setSelectedCategory(selectedCategory);
        m_searchTextBox.requestFocus();
        setVisible(true);
    }
    
    public boolean close()
    {
        setVisible(false);
        return true;
    }
    
    /*
     * @see jmemorize.core.CategoryObserver#onCardEvent(int, jmemorize.core.Card, int)
     */
    public void onCardEvent(int type, Card card, int deck) //CHECK move into cardtable!?
    {
        List cards = m_cardTable.getView().getCards();
        
        // for now we only remove cards but dont add new cards
        if (type == Events.REMOVED_EVENT)
        {
            cards.remove(card);
        }
        
        m_cardTable.getView().setCards(cards, m_searchCategory);
        updateStatusBar();
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCategoryEvent(int, jmemorize.core.Category)
     */
    public void onCategoryEvent(int type, Category category)
    {
        // category combo box handles this event by itself
    }
    
    private void updateStatusBar()
    {
        m_statusBar.setCards(m_cardTable.getView().getCards());
    }
    
    private JPanel buildSearchPanel()
    {
        m_searchTextBox.setEditable(true);
        m_searchTextBox.setMaximumRowCount(10);
        
        m_searchTextBox.addPopupMenuListener(new PopupMenuListener(){
            public void popupMenuCanceled(PopupMenuEvent arg0)
            {
                // ignore
            }

            public void popupMenuWillBecomeInvisible(PopupMenuEvent arg0)
            {
                // ignore
            }

            public void popupMenuWillBecomeVisible(PopupMenuEvent arg0)
            {
                m_searchTextBox.setModel(new DefaultComboBoxModel(
                    m_recentSearchTexts.getItems().toArray()));
            }
        });
        m_searchTextBox.addActionListener(new FindAction());
        
        JScrollPane scrollPane = new JScrollPane(m_cardTable);
        scrollPane.getViewport().setBackground(javax.swing.UIManager.getColor("Table.background"));
        scrollPane.setPreferredSize(new Dimension(500, 200));
        
        FormLayout layout = new FormLayout(
            "right:pref, 3dlu, pref:grow, 3dlu, pref:grow, 3dlu, pref:grow", // columns
            "p, 3dlu, p, 3dlu, p, 3dlu, p, 9dlu, p, 9dlu, fill:d:grow");     // rows
        
        CellConstraints cc = new CellConstraints();
        
        DefaultFormBuilder builder = new DefaultFormBuilder(layout);
        builder.setDefaultDialogBorder();
        
        builder.addLabel("Search text",         cc.xy (1, 1));
        builder.add(m_searchTextBox,            cc.xyw(3, 1, 5));
        
        builder.addLabel("Category",            cc.xy (1, 3));
        builder.add(m_categoryBox,              cc.xyw(3, 3, 5));
        
        builder.addLabel("Settings",            cc.xy (1, 5));
        builder.add(m_radioBothSides,           cc.xy (3, 5));
        builder.add(m_radioFrontSide,           cc.xy (5, 5));
        builder.add(m_radioBackSide,            cc.xy (7, 5));
        
        builder.add(m_matchCaseBox,             cc.xy (3, 7));
        
        builder.addSeparator("Results",         cc.xyw(1, 9, 7));
        
        builder.add(scrollPane,                 cc.xyw(1,11, 7));
        
        return builder.getPanel();
    }
    
    private JPanel buildSearchBar()
    {
        JButton closeButton = new JButton(new CloseAction());
        JButton searchButton = new JButton(new FindAction());
        
        JPanel buttonPanel = ButtonBarFactory.buildRightAlignedBar(searchButton, closeButton);
        buttonPanel.setBorder(new EmptyBorder(0, 5, 5, 10));
        
        getRootPane().setDefaultButton(searchButton);
        
        return buttonPanel;
    }
    
    private void initComponents()
    {
        setTitle("Find");
        
        // build main panel
        ButtonGroup group = new ButtonGroup();
        group.add(m_radioBothSides);
        group.add(m_radioFrontSide);
        group.add(m_radioBackSide);
        
        JPanel mainPanel = new JPanel(new BorderLayout());
        mainPanel.add(buildSearchPanel(), BorderLayout.CENTER);
        mainPanel.add(buildSearchBar(),   BorderLayout.SOUTH);
        mainPanel.setBorder(new EtchedBorder());
        
        // set status bar
        m_statusBar = new StatusBar();
        m_cardTable.setStatusBar(m_statusBar);
        
        getContentPane().add(mainPanel, BorderLayout.CENTER);
        getContentPane().add(m_statusBar, BorderLayout.SOUTH);
        
        
        m_cardTable.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });
        
        // overwrite moving to next row when pressing ENTER
        m_cardTable.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), "Edit");
        m_cardTable.getActionMap().put("Edit", new AbstractAction(){
            public void actionPerformed(ActionEvent e)
            {
                editCards();
            }
        });
        
        pack();
    }
    
    private void tableMouseClicked(MouseEvent evt)
    {
        if (evt.getClickCount() == 2)
        {
            editCards();
        }
    }

    private void editCards()
    {
        Card card  = m_cardTable.getSelectedCard();
        List cards = m_cardTable.getView().getCards();
        Category category = m_cardTable.getView().getCategory();
        
        EditCardFrame.getInstance().showCard(card, cards, category,
            m_searchText, m_searchSides, m_matchCase);
    }
}
