package  jmemorize.gui.swing;

import java.awt.Color;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.CategoryObserver;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.LegendRenderingOrder;
import org.jfree.chart.StandardLegend;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.entity.CategoryItemEntity;
import org.jfree.chart.entity.ChartEntity;
import org.jfree.chart.labels.ItemLabelAnchor;
import org.jfree.chart.labels.ItemLabelPosition;
import org.jfree.chart.labels.StandardCategoryLabelGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.ui.TextAnchor;

/**
 * @author  djemili
 */
public class DeckChartPanel extends javax.swing.JPanel implements CategoryObserver
{    
    private class MouseClicked implements ChartMouseListener
    {
        /**
         * @see org.jfree.chart.ChartMouseListener#chartMouseClicked(org.jfree.chart.ChartMouseEvent)
         */
        public void chartMouseClicked(ChartMouseEvent evt)
        {
            ChartEntity entity = evt.getEntity();
            if (entity instanceof CategoryItemEntity)
            {
                int cat = ((CategoryItemEntity)entity).getCategoryIndex();
                m_frame.getDeckPanel().showDeck(cat - 1);
                
//                CategoryItemRenderer rend = m_chartPanel.getChart().getCategoryPlot().getRenderer();
//                rend.setBaseOutlinePaint(Color.BLUE);
            }
        }

        /**
         * @see org.jfree.chart.ChartMouseListener#chartMouseMoved(org.jfree.chart.ChartMouseEvent)
         */
        public void chartMouseMoved(ChartMouseEvent arg0)
        {
            // do nothing
        }
    }
    
    /** The minimal amount of deck bars show at all time exclusive the summary bar. */
    private final static int        MINIMAL_DECK_BARS = 4;
    
    private final static String     DECK0_NAME            = "Start Deck";
    private final static String     SUMMARY_BAR_NAME      = "Summary";
    
    /* the bar colors */
    private final static Color      UNLEARNED_CARDS_COLOR = Color.GRAY;
    private final static Color      EXPIRED_CARDS_COLOR   = Color.RED;
    private final static Color      LEARNED_CARDS_COLOR   = Color.GREEN;
    
    private Category                m_category;
    private MainFrame               m_frame;
    
    private DefaultCategoryDataset  m_dataset;
    private ChartPanel              m_chartPanel;
    
    public DeckChartPanel() 
    {
        initComponents();
        
        // add the chart to a panel...
        JFreeChart chart = createChart();
        m_chartPanel = new ChartPanel(chart);
        m_chartPanel.addChartMouseListener(new MouseClicked());
        
        m_chartPanel.setMinimumDrawHeight(100);
        m_chartPanel.setMinimumDrawWidth(400);
        
        add(m_chartPanel);
    }
    
    public void setFrame(MainFrame frame)
    {
        m_frame = frame;
    }
    
    public void setCategory(Category category)
    {
        if (m_category != null)
        {
            m_category.removeObserver(this);
        }
        
        m_category = category;        
        category.addObserver(this);
        
        createDataset();
    }
    
    private void createDataset()
    {
        m_dataset = createDefaultDataSet();
        updateBars();
        ((CategoryPlot)m_chartPanel.getChart().getPlot()).setDataset(m_dataset);
    }
    
    private JFreeChart createChart() 
    {
        m_dataset = createDefaultDataSet();
        
        JFreeChart chart = ChartFactory.createStackedBarChart3D(
            "",                       // chart title
            "Decks",                  // domain axis label
            "Cards",                  // range axis label
            m_dataset,                // data
            PlotOrientation.VERTICAL, // the plot orientation
            true,                     // include legend
            true,                     // tooltips
            false                     // urls
        );
        
        StandardLegend legend = (StandardLegend)chart.getLegend();
        legend.setRenderingOrder(LegendRenderingOrder.REVERSE);
        
        CategoryPlot plot = (CategoryPlot)chart.getPlot();
        plot.getRangeAxis().setStandardTickUnits(NumberAxis.createIntegerTickUnits()); //CHECK use locale
        
        CategoryItemRenderer renderer = plot.getRenderer();
        renderer.setLabelGenerator(new StandardCategoryLabelGenerator());
        renderer.setItemLabelsVisible(true);
        renderer.setPositiveItemLabelPosition(
            new ItemLabelPosition(ItemLabelAnchor.CENTER, TextAnchor.CENTER)
        );
        renderer.setSeriesPaint(0, LEARNED_CARDS_COLOR); //HACK
        renderer.setSeriesPaint(1, EXPIRED_CARDS_COLOR);
        renderer.setSeriesPaint(2, UNLEARNED_CARDS_COLOR);
        
        return chart;        
    }
    
    private void initComponents() 
    {
        setLayout(new java.awt.BorderLayout());
        setBorder(new javax.swing.border.EtchedBorder());
    }
    
    private DefaultCategoryDataset createDefaultDataSet()
    {
        DefaultCategoryDataset dataset = new DefaultCategoryDataset();

        dataset.addValue(0, "Learned Cards", SUMMARY_BAR_NAME);
        for (int i = 0; i < MINIMAL_DECK_BARS; i++)
        {
            dataset.addValue(0, "Learned Cards", i > 0 ? "Deck "+i : DECK0_NAME);
        }
        
        return dataset;
    }
    
    private void updateBars() //CHECK put Dataset as argument!?
    {
        updateSummaryBar();
        
        for (int i=0; i < Math.max(m_category.getNumberOfDecks(), MINIMAL_DECK_BARS); i++)
        {
            updateBar(i);
        }
    }
    
    private void updateSummaryBar()
    {
        m_dataset.setValue(m_category.getLearnedCards().size(),   "Learned Cards", SUMMARY_BAR_NAME);
        m_dataset.setValue(m_category.getExpiredCards().size(),   "Expired Cards", SUMMARY_BAR_NAME);
        m_dataset.setValue(m_category.getUnlearnedCards().size(), "Unlearned Cards", SUMMARY_BAR_NAME);
    }
    
    private void updateBar(int level)
    {
        if (level == 0)
        {
            m_dataset.setValue(m_category.getCards(level).size(), "Unlearned Cards", DECK0_NAME);
        }
        else if (level >= m_category.getNumberOfDecks())
        {
            m_dataset.setValue(0, "Unlearned Cards", "Deck " + level);
            m_dataset.setValue(0, "Learned Cards", "Deck " + level);
            m_dataset.setValue(0, "Expired Cards", "Deck " + level);
        }
        else
        {
            m_dataset.setValue(m_category.getLearnedCards(level).size(), "Learned Cards", "Deck " + level);
            m_dataset.setValue(m_category.getExpiredCards(level).size(), "Expired Cards", "Deck " + level);
        }
    }
    
    /*
     * @see jmemorize.core.CategoryObserver#onCategoryEvent(int, jmemorize.core.Category)
     */
    public void onCategoryEvent(int type, Category category)
    {
        // ignore. mainframe already looks for important category changes
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCardEvent(int, jmemorize.core.Card, int)
     */
    public void onCardEvent(int type, Card card, int level) //HACK
    {   
        updateBars(); 
        
        // if we are showing more decks then really in lesson and more then minimum bars visible
        while (m_dataset.getColumnCount() > Math.max(m_category.getNumberOfDecks() + 1, MINIMAL_DECK_BARS + 1))
        {
            m_dataset.removeColumn(m_dataset.getColumnCount() - 1);
        }
    }
}
