/*
 * Created on 14-May-2004
 */
package jmemorize.gui.swing;

import java.awt.Component;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.EventObject;
import java.util.Iterator;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellEditor;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.CategoryObserver;


/**
 * @author djemili
 */
public class CategoryTree extends JTree implements CategoryObserver
{
    private class CellRenderer extends DefaultTreeCellRenderer
    {
        public CellRenderer()
        {
            setLeafIcon(FOLDER_ICON);
            setOpenIcon(FOLDER_ICON);
            setClosedIcon(FOLDER_ICON);
        }
        
        /**
         * @see javax.swing.tree.DefaultTreeCellRenderer#getTreeCellRendererComponent(javax.swing.JTree, java.lang.Object, boolean, boolean, boolean, int, boolean)
         */
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel,
            boolean expanded, boolean leaf, int row, boolean hasFocus)
        {
            JLabel label = (JLabel)super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            Object nodeValue = ((DefaultMutableTreeNode)value).getUserObject();
            
            if (nodeValue instanceof Category)
            {
                Category category = (Category)nodeValue;
                label.setText(category.getName());
            }
            
            if (hasFocus)
            {
//                label.set
//                this.set
//                label.setBackground(Color.LIGHT_GRAY);
//                System.out.println("focus"+nodeValue);
            }
            
            return label;
        }
    }
    
    private class CategoryTreeModel extends DefaultTreeModel
    {
        /**
         * @param root
         */
        public CategoryTreeModel(TreeNode root)
        {
            super(root);
        }
        
        /**
         * Instead of overwriting userObject like the super method does, the input is
         * set as the new name of the category.
         * 
         * @see javax.swing.tree.DefaultTreeModel#valueForPathChanged(javax.swing.tree.TreePath, java.lang.Object)
         */
        public void valueForPathChanged(TreePath path, Object newValue)
        {
            DefaultMutableTreeNode aNode = (DefaultMutableTreeNode)path.getLastPathComponent();
            Category category = (Category)aNode.getUserObject();
            category.setName((String)newValue);
            
            nodeChanged(aNode);
        }
        
    }
    
    private class CellEditor extends DefaultTreeCellEditor
    {
        public CellEditor(JTree tree, DefaultTreeCellRenderer renderer)
        {
            super(tree, renderer);
        }
        
        public DefaultMutableTreeNode getEditedNode()
        {
            return m_editedNode; 
        }
        
        /**
         * @return Returns the nodeCategory.
         */
        public Category getNodeCategory()
        {
            return m_editedCategory;
        }
        
        /**
         * @see javax.swing.tree.DefaultTreeCellEditor#isCellEditable(java.util.EventObject)
         */
        public boolean isCellEditable(EventObject event)
        {
            // event is null if edit is started by click-pause-click
            if (event != null)
            {
                MouseEvent mEvent = (MouseEvent)event;
                TreePath path     = getPathForLocation(mEvent.getX(), mEvent.getY());
                m_editedNode      = (DefaultMutableTreeNode)path.getLastPathComponent();
                m_editedCategory  = (Category)m_editedNode.getUserObject();
            }
            
            // make root not editable
            return super.isCellEditable(event) && m_editedCategory != m_rootCategory;
        }
        
        /**
         * @see javax.swing.tree.DefaultTreeCellEditor#getTreeCellEditorComponent(javax.swing.JTree, java.lang.Object, boolean, boolean, boolean, int)
         */
        public Component getTreeCellEditorComponent(JTree tree, Object value, boolean isSelected,
            boolean expanded, boolean leaf, int row)
        {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
            Category category = (Category)node.getUserObject();
            return super.getTreeCellEditorComponent(tree, category.getName(), isSelected, expanded, leaf, row);
        }
        
    }
    
    private final ImageIcon FOLDER_ICON = new ImageIcon(getClass().getResource("/resource/icons/folder.gif"));
    
    private DefaultMutableTreeNode  m_editedNode;     //HACK
    private Category                m_editedCategory;

    private Category                m_rootCategory;
    
    public CategoryTree()
    {
        CellRenderer renderer = new CellRenderer();
        setCellRenderer(renderer);
        
        setCellEditor(new CellEditor(this, renderer));
        setTransferHandler(MainFrame.TRANSFER_HANDLER);
        
        setDragEnabled(true);
        setEditable(true);
    }
    
    public void setRootCategory(Category category)
    {
        if (m_rootCategory != null)
        {
            m_rootCategory.removeObserver(this);
        }
        
        m_rootCategory = category;
        m_rootCategory.addObserver(this);
        
        MutableTreeNode root = createCategory(category);
        setModel(new CategoryTreeModel(root));
        this.repaint();
    }
    
    public void setSelectedCategory(Category category)
    {
        if (category == null || m_rootCategory == null) //HACK
        {
            return;
        }
        
        DefaultMutableTreeNode root = (DefaultMutableTreeNode)getModel().getRoot();
        Enumeration enumer = root.depthFirstEnumeration();
        
        while (enumer.hasMoreElements())
        {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)enumer.nextElement();
            Object userValue = node.getUserObject();
            
            if (userValue == category)
            {
                setSelectionPath(new TreePath(node.getPath()));
            }
        }
    }
    
    public Category getSelectedCategory()
    {
        return (Category)getSelectedValue();
    }
    
    /*
     * @see jmemorize.core.CategoryObserver#onCategoryEvent(int, jmemorize.core.Category)
     */
    public void onCategoryEvent(int type, Category category)
    {
        switch (type)
        {
            case ADDED_EVENT: 
                MutableTreeNode father    = getNode(category.getFather());
                MutableTreeNode newChild  = new DefaultMutableTreeNode(category);
                father.insert(newChild, father.getChildCount()); //CHECK adding at end
                
                DefaultTreeModel model = (DefaultTreeModel)getModel();
                model.reload(father);
                break;
            case REMOVED_EVENT:
                father = getNode(category.getFather());
                MutableTreeNode child  = getNode(category);
                father.remove(child);
                            
                model = (DefaultTreeModel)getModel();
                model.reload(father);
                break;
            case EDITED_EVENT:
                MutableTreeNode node = getNode(category);
                
                model = (DefaultTreeModel)getModel();
                model.reload(node);
                break;
        }
        
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCardEvent(int, jmemorize.core.Card, int)
     */
    public void onCardEvent(int type, Card card, int deck)
    {
        // ignore
    }
    
    private DefaultMutableTreeNode getNode(Object userValue)
    {
        DefaultTreeModel model = (DefaultTreeModel)getModel();
        DefaultMutableTreeNode root = (DefaultMutableTreeNode)model.getRoot();
        
        for (Enumeration enumer = root.depthFirstEnumeration(); enumer.hasMoreElements();)
        {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)enumer.nextElement();
            
            if (node.getUserObject() == userValue)
            {
                return node;
            }
        }
        
        return null;
    }
    
    private Object getSelectedValue()
    {
        TreePath path = getSelectionPath();
        
        if (path != null)
        {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)path.getLastPathComponent();
            return node.getUserObject();
        }

        return null;
    }
    
    private List getSelectedValues()
    {
        TreePath[] path = getSelectionPaths();
        List userObjects = new ArrayList(path.length);
        
        for (int i = 0; i < path.length; i++)
        {
            DefaultMutableTreeNode leaf = (DefaultMutableTreeNode)path[i].getLastPathComponent();
            userObjects.add(leaf.getUserObject());
        }

        return userObjects;
    }
    
    private MutableTreeNode createCategory(Category root)
    {
        DefaultMutableTreeNode node = new DefaultMutableTreeNode(root);
        
        // for all child categories
        for (Iterator it = root.getChildCategories().iterator(); it.hasNext();)
        {
            Category cat = (Category)it.next();
            
            node.add(createCategory(cat));
        }
        
        return node;
    }
}
