/*
 * Created on 31-Mar-2004
 */
package jmemorize.gui.swing;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.prefs.Preferences;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import jmemorize.core.Card;
import jmemorize.core.Category;
import jmemorize.core.ExitObserver;
import jmemorize.core.Main;
import jmemorize.util.Arrow;
import jmemorize.util.PreferencesTool;
import jmemorize.util.ReverseOrder;
import jmemorize.util.TimeSpan;

/**
 * This component adds a clientProperty named "selectedCards" that always holds 
 * all currently selected cards. 
 *
 * @author djemili
 */
public class CardTable extends JTable implements ExitObserver
{
    private class DateRenderer extends DefaultTableCellRenderer 
    {   
        private String m_nullString;
        
        public DateRenderer(String nullString)
        {
            m_nullString = nullString;
        }
        
        protected void setValue(Object value) 
        {
            setText(value == null ? m_nullString : MainFrame.SHORT_DATE_FORMATER.format(value));            
        }
    }
    
    private class DateExpiredRenderer extends DefaultTableCellRenderer 
    {    
        private final ImageIcon OK_ICON      = new ImageIcon(getClass().getResource("/resource/icons/state_ok.gif"));
        private final ImageIcon TODAY_ICON   = new ImageIcon(getClass().getResource("/resource/icons/state_soon.gif"));
        private final ImageIcon NO_ICON      = new ImageIcon(getClass().getResource("/resource/icons/state_no.gif"));
        private final ImageIcon EXPIRED_ICON = new ImageIcon(getClass().getResource("/resource/icons/state_forgotten.gif"));

        public void setValue(Object value) 
        {
            // if not learned
            if (value == null)
            {
                setText("not learned");
                setIcon(NO_ICON);
            }
            else
            {
                Date expiration = (Date)value;
                
                // if tomorrow still valid
                if (expiration.after(Main.getTomorrow()))
                {
                    setIcon(OK_ICON);
                }
                // if only valid day left is today
                else if (expiration.after(Main.getNow()))
                {
                    setIcon(TODAY_ICON);
                }
                // if expired
                else
                {
                    setIcon(EXPIRED_ICON);
                }
                
                setText(MainFrame.SHORT_DATE_FORMATER.format(value));
            }
        }
    }
    
    private class LazyRowComparator implements Comparator //CHECK rename
    {
        private int m_modelIndex;
        
        public LazyRowComparator(int modelIndex)
        {
            m_modelIndex = modelIndex;
        }

        /**
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object arg0, Object arg1)
        {
            Comparable col0 = (Comparable)getValue((Card)arg0, m_modelIndex);
            Comparable col1 = (Comparable)getValue((Card)arg1, m_modelIndex);
            
            if (col0 == null)
            {
                return col1 == null ? 0 : -1;
            } 
            else
            {
                return col1 == null ? 1 : col0.compareTo(col1);
            } 
        }
    }
    
    /**
     * Copied from Java Sample.
     */
    private class SortableHeaderRenderer implements TableCellRenderer
    {
        private TableCellRenderer tableCellRenderer;

        public SortableHeaderRenderer(TableCellRenderer tableCellRenderer)
        {
            this.tableCellRenderer = tableCellRenderer;
        }

        public Component getTableCellRendererComponent(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column)
        {
            Component c = tableCellRenderer.getTableCellRendererComponent(table, value, isSelected,
                hasFocus, row, column);
            
            if (c instanceof JLabel)
            {
                JLabel l = (JLabel)c;
                l.setHorizontalTextPosition(SwingConstants.LEFT);
                int modelColumn = table.convertColumnIndexToModel(column);
                
                if (modelColumn == m_tableModel.getSortingColumn())
                {
                    l.setIcon(m_tableModel.getSortingDir() == ViewModel.ASCENDING ? 
                        m_ascendingArrow : m_descendingArrow);
                }
                else
                {
                    l.setIcon(null);
                }
            }
            
            return c;
        }
    }
    
    public class ViewModel extends AbstractTableModel
    {
        public static final int ASCENDING  = 0;
        public static final int DESCENDING = 1;

        private List            m_cards    = new ArrayList();
        private Category        m_category;    //HACK currently this is only needed for transferhandlers

        private int             m_orderModelIndex;
        private int             m_orderDir;
        
        public Category getCategory()
        {
            return m_category;
        }
        
        public void setCards(List cards, Category category)
        {
            m_cards = cards;
            m_category = category;
            resort();
            
            updateCardCountStatusBar();
        }
        
        /**
         * @param modelIndex The model index by which the sorting should happen.
         * @param direction  Can be in ASCENDING or DESCENDING order.
         */
        public void setSorting(int modelIndex, int direction)
        {
            m_orderModelIndex = modelIndex;
            m_orderDir = direction;
            
            resort();
        }
        
        public List getCards()
        {
            return m_cards;
        }

        /**
         * @see javax.swing.table.TableModel#getRowCount()
         */
        public int getRowCount()
        {
            return m_cards != null ? m_cards.size() : 0;
        }

        /**
         * @see javax.swing.table.TableModel#getColumnCount()
         */
        public int getColumnCount()
        {
            return COLUMN_NAMES.length;
        }
        
        /**
         * @see javax.swing.table.AbstractTableModel#getColumnName(int)
         */
        public String getColumnName(int column)
        {
            return COLUMN_NAMES[column];
        }
        
        /**
         * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
         */
        public boolean isCellEditable(int rowIndex, int columnIndex)
        {
            return false;
        }

        /**
         * @see javax.swing.table.TableModel#getValueAt(int, int)
         */
        public Object getValueAt(int rowIndex, int columnIndex)
        {
            Card card = (Card)m_cards.get(rowIndex);
            return getValue(card, columnIndex);
        }

        /**
         * @return Returns the orderColumn.
         */
        public int getSortingColumn()
        {
            return m_orderModelIndex;
        }
        
        /**
         * @return Returns the orderDir.
         */
        public int getSortingDir()
        {
            return m_orderDir;
        }
        
        private void resort()
        {
            if (m_cards != null) //CHECK
            {
                Comparator comparator = new LazyRowComparator(m_orderModelIndex);
                Collections.sort(m_cards, m_orderDir == ASCENDING ? 
                    comparator : new ReverseOrder(comparator));
                
                fireTableDataChanged();
            }
        }
    }
    
    // column enum
    public static final int       COLUMN_FRONTSIDE  = 0;
    public static final int       COLUMN_BACKSIDE   = 1;
    public static final int       COLUMN_DECK       = 2;
    public static final int       COLUMN_CATEGORY   = 3;
    public static final int       COLUMN_PATH       = 4;
    public static final int       COLUMN_CREATED    = 5;
    public static final int       COLUMN_TESTED     = 6;
    public static final int       COLUMN_EXPIRES    = 7;
    
    private static final String[] COLUMN_NAMES    = 
        {"Frontside", "Flipside", "Deck", "Category", "Path", "Created", "Last Test", "Expires"};

    // widgets
    private JCheckBoxMenuItem[]   m_checkBoxItems   = new JCheckBoxMenuItem[COLUMN_NAMES.length];
    private JPopupMenu            m_menu            = new JPopupMenu("Columns");
    private StatusBar             m_statusBar;

    // icons
    private Icon                  m_ascendingArrow  = new Arrow(false, 15);
    private Icon                  m_descendingArrow = new Arrow(true, 15);
    
    private ViewModel             m_tableModel      = new ViewModel();
    private int[]                 m_columns;
    private int[]                 m_defaultColumns;
    private int[]                 m_columnWidths    = new int[COLUMN_NAMES.length];
    private Preferences           m_prefs;
    
    
    /**
     * Creates new form BatchListPanel 
     */
    public CardTable(Preferences prefs, int[] defaultColumns) 
    {
        Main.getInstance().addExitObserver(this);
        
        m_prefs = prefs;
        m_defaultColumns = defaultColumns;
        
        for (int i = 0; i < COLUMN_NAMES.length; i++)
        {
            m_checkBoxItems[i] = new JCheckBoxMenuItem(COLUMN_NAMES[i]);
            m_checkBoxItems[i].addActionListener(new ActionListener(){
                public void actionPerformed(ActionEvent e)
                {
                    columnSelectionChanged();
                }
            });
            m_menu.add(m_checkBoxItems[i]);
        }
        
        setModel(m_tableModel);
        
        // table is sorted with expiration date by default
        m_tableModel.setSorting(COLUMN_EXPIRES, ViewModel.ASCENDING); //TODO save in prefs
        
        getTableHeader().addMouseListener(new MouseAdapter(){
            public void mouseClicked(MouseEvent e)
            {
                headerClicked(e);
            }
        });
        
        loadFromPreferences();
        
        setTransferHandler(MainFrame.TRANSFER_HANDLER);
        setDragEnabled(true);
        setShowGrid(false);
        //setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
    }
    
    public void setColumns(int[] columns)
    {
        m_columns = columns;
        
        // create and set tablecolumn model
        TableColumnModel columnModel = new DefaultTableColumnModel();
        for (int i = 0; i < columns.length; i++)
        {
            TableColumn column = new TableColumn(columns[i]);
            column.setHeaderValue(COLUMN_NAMES[columns[i]]);
            column.setHeaderRenderer(new SortableHeaderRenderer(getTableHeader().getDefaultRenderer()));
            
            if (m_columnWidths[columns[i]] > 0)
            {
                column.setPreferredWidth(m_columnWidths[columns[i]]);
            }
            
            switch (columns[i])
            {
                case COLUMN_CREATED:
                    column.setCellRenderer(new DateRenderer("-"));
                    break;
                    
                case COLUMN_EXPIRES:
                    column.setCellRenderer(new DateExpiredRenderer());
                    break;
                    
                case COLUMN_TESTED:
                    column.setCellRenderer(new DateRenderer("not learned"));
                    break;
                    
                default:
                    column.setCellRenderer(new DefaultTableCellRenderer());
            }
            
            columnModel.addColumn(column);
        }
        
        setColumnModel(columnModel);
    }
    
    public void setStatusBar(StatusBar statusBar)
    {
        m_statusBar = statusBar;
        
        if (m_statusBar != null)
        {
            updateCardCountStatusBar();
            updateSelectedCardCountStatusBar();
        }
    }
    
    public ViewModel getView()
    {
        return m_tableModel;
    }
    
    public Card getSelectedCard()
    {
        int row = getSelectedRow();
        return (Card)m_tableModel.getCards().get(row);
    }
    
    public List getSelectedCards()
    {
        if (m_tableModel == null)
        {
            return new ArrayList();
        }
        
        List cards     = m_tableModel.getCards();
        List selection = new ArrayList(cards.size());
        
        int rows[] = getSelectedRows();
        for (int i = 0; i < rows.length; i++)
        {
            selection.add(cards.get(rows[i]));
        }
        
        return selection;
    }
    
    /**
     * Is called when a listSelection occured. Besides calling the supermethod it
     * updates the clientProperty "selectedCards" with the currently selected cards.
     *
     * @see javax.swing.JTable#valueChanged(javax.swing.event.ListSelectionEvent)
     */
    public void valueChanged(ListSelectionEvent e)
    {
        super.valueChanged(e);
        
        updateSelectedCardCountStatusBar();
        
        if (!e.getValueIsAdjusting())
        {
            putClientProperty("selectedCards", getSelectedCards());
        }
    }
    
    /**
     * @see javax.swing.JTable#getToolTipText(java.awt.event.MouseEvent)
     */
    public String getToolTipText(MouseEvent event)
    {
        int row = rowAtPoint(event.getPoint());
        int col = columnAtPoint(event.getPoint());
        
        Card card = (Card)m_tableModel.getCards().get(row);
        int modelIndex = m_columns[col];
        
        switch (modelIndex)
        {
            case 3:
                return card.getCategory().getPath();
            case 5:
                Date date = card.getDateCreated();
                return ((date != null) ? TimeSpan.format(new Date(), date) : null);
            case 6:
                date = card.getDateTested();
                return ((date != null) ? TimeSpan.format(new Date(), date) : "This card has never been tested.");
            case 7:
                date = card.getDateExpired();
                return ((date != null) ? TimeSpan.format(new Date(), date) : "This card has never been tested.");
            default:
                return null;
        }
    }
    
    /*
     * @see javax.swing.JTable#columnMarginChanged(javax.swing.event.ChangeEvent)
     */
    public void columnMarginChanged(ChangeEvent e) 
    {
        // CHECK replace by application exit behaviour
        super.columnMarginChanged(e);
        
        for (int i = 0; i < m_columns.length; i++)
        {
            TableColumn column = getColumnModel().getColumn(i);
            m_columnWidths[column.getModelIndex()] = column.getWidth();
        }
    }
    
    /*
     * @see javax.swing.JTable#columnMoved(javax.swing.event.TableColumnModelEvent)
     */
    public void columnMoved(TableColumnModelEvent evt)
    {
        super.columnMoved(evt);
        
        int[] columns = new int[m_columns.length];
        for (int i = 0; i < columns.length; i++)
        {
            TableColumn column = getColumnModel().getColumn(i);
            columns[i] = column.getModelIndex();
        }
        
        m_columns = columns;
    }
    
    /*
     * @see jmemorize.core.ExitObserver#onExit()
     */
    public void onExit()
    {
        storeToPreferences();
    }
    
    private static Object getValue(Card card, int modelIndex)
    {
        switch (modelIndex)
        {
            case 0: 
                return card.getFrontSide().replace('\n', ' ');
            case 1:
                return card.getBackSide().replace('\n', ' ');
            case 2:
                return new Integer(card.getLevel());
            case 3: 
                return card.getCategory().getName();
            case 4: 
                return card.getCategory().getPath();    
            case 5: 
                return card.getDateCreated();
            case 6:    
                return card.getDateTested();
            case 7:
                return card.getDateExpired();
            default:
                return "-"; // this should never be reached
        }
    }
    
    private void columnSelectionChanged()
    {
        ArrayList columns = new ArrayList();
        for (int i = 0; i < m_columns.length; i++)
        {
            columns.add(new Integer(m_columns[i]));
        }
        
        for (int i = 0; i < COLUMN_NAMES.length; i++)
        {
            if (m_checkBoxItems[i].isSelected() && !columns.contains(new Integer(i)))
            {
                columns.add(Math.min(i, columns.size()), new Integer(i));
            }
            else if (!m_checkBoxItems[i].isSelected() && columns.contains(new Integer(i)))
            {
                columns.remove(new Integer(i));
            }
        }
        
        int[] columnArray = new int[columns.size()];
        int idx = 0;
        for (Iterator it = columns.iterator(); it.hasNext();)
        {
            Integer integer = (Integer)it.next();
            columnArray[idx++] = integer.intValue();
        }
        
        setColumns(columnArray);
    }
    
    private void headerClicked(MouseEvent e)
    {
        if (e.getButton() == MouseEvent.BUTTON1) // left button
        {
            JTableHeader h = (JTableHeader)e.getSource();            
            TableColumnModel columnModel = h.getColumnModel();            
            int viewColumn = columnModel.getColumnIndexAtX(e.getX());            
            int column = columnModel.getColumn(viewColumn).getModelIndex();
            
            if (column != -1)
            {
                // if column already sorting invert sort direction
                if (column == m_tableModel.getSortingColumn())
                {
                    int dir = m_tableModel.getSortingDir();
                    m_tableModel.setSorting(column, dir == ViewModel.ASCENDING ? 
                        ViewModel.DESCENDING : ViewModel.ASCENDING);
                }
                // if new sorting column selected
                else
                {
                    m_tableModel.setSorting(column, ViewModel.ASCENDING);
                }
                
                h.repaint();
            }
        }
        else if (e.getButton() == MouseEvent.BUTTON3) // right button
        {
            // first set all checkbox menu items to not selected..
            for (int i = 0; i < m_checkBoxItems.length; i++)
            {
                m_checkBoxItems[i].setSelected(false);
            }
            // ..now we only select those given as new columns
            for (int i = 0; i < m_columns.length; i++)
            {
                m_checkBoxItems[m_columns[i]].setSelected(true);
            }
            
            m_menu.show(e.getComponent(), e.getX(), e.getY());
        }
    }
    
    private void loadFromPreferences()
    {
        // we need to check the result because there might be stored values
        // from older project versions that might have become invalid.
        
        m_columnWidths = PreferencesTool.getIntArray(m_prefs, "widths");
        if (m_columnWidths == null || m_columnWidths.length != COLUMN_NAMES.length)
        {
            m_columnWidths = new int[COLUMN_NAMES.length];
        }
        
        int[] columns = PreferencesTool.getIntArray(m_prefs, "columns");
        setColumns(columns != null ? columns : m_defaultColumns);
    }
    
    private void updateSelectedCardCountStatusBar()
    {
        if (m_statusBar != null)
        {
            if (getSelectedRowCount() >= 2)
            {
                m_statusBar.setLeftText("Selected cards: "+ getSelectedRowCount());
            }
            else
            {
                m_statusBar.setLeftText("");
            }
        }
    }
    
    private void updateCardCountStatusBar()
    {
        if (m_statusBar != null)
        {
            m_statusBar.setCards(m_tableModel.getCards());
        }
    }
    
    private void storeToPreferences()
    {
        PreferencesTool.putIntArray(m_prefs, "widths",  m_columnWidths);
        PreferencesTool.putIntArray(m_prefs, "columns", m_columns);
    }
}
