/*
 * Created on 07-Oct-2004
 */
package jmemorize.gui.swing;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.util.Iterator;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.TransferHandler;

import jmemorize.core.Card;
import jmemorize.core.Category;

/**
 * Organizes datatransfers between the card table and the category tree.
 * 
 * @author djemili
 */
public class CardCategoryTransferHandler extends TransferHandler
{
    public class CardsTransferable implements Transferable
    {
        private List m_cards;

        public CardsTransferable(List cards)
        {
            m_cards = cards;
        }

        public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException
        {
            if (!isDataFlavorSupported(flavor))
            {
                throw new UnsupportedFlavorException(flavor);
            }
            
            return m_cards;
        }

        public DataFlavor[] getTransferDataFlavors()
        {
            return new DataFlavor[] {CARDS_FLAVOR};
        }

        public boolean isDataFlavorSupported(DataFlavor flavor)
        {
            return CARDS_FLAVOR.equals(flavor);
        }
    }
    
    public class CategoryTransferable implements Transferable
    {
        private Category m_category;

        public CategoryTransferable(Category category)
        {
            m_category = category;
        }

        public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException
        {
            if (!isDataFlavorSupported(flavor))
            {
                throw new UnsupportedFlavorException(flavor);
            }
            
            return m_category;
        }

        public DataFlavor[] getTransferDataFlavors()
        {
            return new DataFlavor[] {CATEGORY_FLAVOR};
        }

        public boolean isDataFlavorSupported(DataFlavor flavor)
        {
            return CATEGORY_FLAVOR.equals(flavor);
        }
    }
    
    public final static DataFlavor CARDS_FLAVOR    = new DataFlavor(Card.class, "Card");
    public final static DataFlavor CATEGORY_FLAVOR = new DataFlavor(Category.class, "Category");

    /*
     * @see javax.swing.TransferHandler#getSourceActions(javax.swing.JComponent)
     */
    public int getSourceActions(JComponent c)
    {
        return COPY_OR_MOVE;
    }
    
    /*
     * @see javax.swing.TransferHandler#canImport(javax.swing.JComponent, java.awt.datatransfer.DataFlavor[])
     */
    public boolean canImport(JComponent comp, DataFlavor[] transferFlavors)
    {
        if (comp instanceof CategoryTree)
        {
            for (int i = 0; i < transferFlavors.length; i++)
            {
                if (transferFlavors[i] == CARDS_FLAVOR || transferFlavors[i] == CATEGORY_FLAVOR)
                {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /*
     * @see javax.swing.TransferHandler#createTransferable(javax.swing.JComponent)
     */
    protected Transferable createTransferable(JComponent c)
    {
        if (c instanceof CardTable)
        {
            CardTable table = (CardTable)c;
            return new CardsTransferable(table.getSelectedCards());
        }
        else if (c instanceof CategoryTree)
        {
            CategoryTree tree = (CategoryTree)c;
            Category category = tree.getSelectedCategory();
            
            // dont allow operations with root category
            return category.getFather() != null ? new CategoryTransferable(category) : null;
        }
            
        return null;
    }
    
    /*
     * @see javax.swing.TransferHandler#importData(javax.swing.JComponent, java.awt.datatransfer.Transferable)
     */
    public boolean importData(JComponent comp, Transferable t)
    {
        Category targetCategory;
        if (comp instanceof CategoryTree)
        {
            CategoryTree tree = (CategoryTree)comp;
            targetCategory = tree.getSelectedCategory();
        }
        else if (comp instanceof CardTable)
        {
            CardTable table = (CardTable)comp;
            targetCategory = table.getView().getCategory();
        }
        else
        {
            return false;
        }
        
        try
        {
            if (t.isDataFlavorSupported(CARDS_FLAVOR))
            {
                List cards = (List) t.getTransferData(CARDS_FLAVOR);
                for (Iterator it = cards.iterator(); it.hasNext();)
                {
                    Card card = (Card) it.next();
                    targetCategory.addCard((Card)card.clone(), card.getLevel());
                }

                return true;
            }
            else if (t.isDataFlavorSupported(CATEGORY_FLAVOR))
            {
                Category category = (Category)t.getTransferData(CATEGORY_FLAVOR);
                if (!category.contains(targetCategory))
                {
                    copyCategories(targetCategory, category);
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        
        return false;
    }
    
    private void copyCategories(Category newFather, Category original) throws CloneNotSupportedException
    {
        Category copy = new Category(original.getName());
        newFather.addCategoryChild(copy);
        
        // first copy categories..
        for (Iterator it = original.getChildCategories().iterator(); it.hasNext();)
        {
            Category category = (Category)it.next();
            copyCategories(copy, category);
        }
        
        // ..then copy cards
        for (int i = 0; i < original.getNumberOfDecks(); i++)
        {
            for (Iterator it = original.getLocalCards(i).iterator(); it.hasNext();)
            {
                Card card = (Card)it.next();
                copy.addCard((Card)card.clone(), i);
            }
        }
    }
    
    /*
     * @see javax.swing.TransferHandler#exportDone(javax.swing.JComponent, java.awt.datatransfer.Transferable, int)
     */
    protected void exportDone(JComponent source, Transferable data, int action)
    {
        if (action != MOVE)
        {
            return;
        }
        
        try
        {
            if (data.isDataFlavorSupported(CARDS_FLAVOR))
            {
                CardTable table = (CardTable)source;
                Category category = table.getView().getCategory();

                List cards = (List) data.getTransferData(CARDS_FLAVOR);
                for (Iterator it = cards.iterator(); it.hasNext();)
                {
                    Card card = (Card)it.next();
                    category.removeCard(card);
                }
            }
            else if (data.isDataFlavorSupported(CATEGORY_FLAVOR))
            {
                Category category = (Category)data.getTransferData(CATEGORY_FLAVOR);
                category.remove();
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
}
