/*
 * Created on 23-Mar-2004
 */
package jmemorize.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.DateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * @author djemili
 */
public class Lesson implements CategoryObserver 
{
    /** True if this lesson has been modified since last save or load process */
    private boolean           m_modified = false; //TODO move to jmemorize
    private File              m_file;
    
    // we need a fixed formatter in file (not locale depent)
    private static DateFormat m_dateFormat = 
        DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM, Locale.UK);
        
    private Category          m_rootCategory;
    
    public Lesson()
    {
        m_rootCategory =  new Category("All");
        m_rootCategory.addObserver(this);
    }
    
    /**
     * @return Returns the file.
     */
    public File getFile()
    {
        return m_file;
    }
    
    /**
     * @param file The file to set.
     */
    public void setFile(File file)
    {
        m_file = file;
    }
    
    public boolean isModified()
    {
        return m_modified;
    }
    
    /**
     * @return Returns the rootCategory.
     */
    public Category getRootCategory()
    {
        return m_rootCategory;
    }
    
    public Card getNextExpirationCard(Date after)
    {
        List cards = m_rootCategory.getCards();
        
        Card minCard = null;
        
        for (Iterator it = cards.iterator(); it.hasNext();)
        {
            Card newCard = (Card)it.next();
            
            Date expiration = newCard.getDateExpired();
            
            Date oldVal = minCard != null ? minCard.getDateExpired() : null;
            Date newVal = newCard.getDateExpired() != null && newCard.getDateExpired().compareTo(after) > 0
                ? newCard.getDateExpired() : null;
            
            if (oldVal == null && newVal != null)
            {
                minCard = newCard;
            }
            else if (oldVal != null && newVal != null && oldVal.compareTo(newVal) > 0) 
            {
                minCard = newCard;
            }
        }
        
        return minCard;
    }
    
    /**
     * Returns the XML-String representation of this lesson.
     * 
     * XML-Schema:
     * 
     * <lesson>
     * 		<deck>
     * 			<card frontside="bla" backside="bla"/> 				
     * 			..
     * 		</deck>
     * 	    .. 
     * </lesson>
     * 
     * @throws TransformerException
     * @throws ParserConfigurationException
     */
    void toXML(File file) throws IOException, TransformerException, ParserConfigurationException
    {   
        GZIPOutputStream zipOut  = new GZIPOutputStream(new FileOutputStream(file));
        
        try
        {
            Document document = DocumentBuilderFactory.newInstance()
                .newDocumentBuilder().newDocument();

            // add lesson tag as root
            Element lessonTag = document.createElement("Lesson");
            document.appendChild(lessonTag);

            // add category tags
            writeCategory(document, lessonTag, getRootCategory());

            // transform document for file output
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
            transformer.setOutputProperty(OutputKeys.INDENT, "yes"); //TODO make identing work

            transformer.transform(new DOMSource(document), new StreamResult(zipOut));

            setModified(false);
        }
        finally
        {
            if (zipOut != null)
            {
                zipOut.close();
            }
        }
    }
    
    private static void writeCategory(Document document, Element father, Category category)
    {
        Element categoryTag = document.createElement("Category");
        categoryTag.setAttribute("name", category.getName());
        father.appendChild(categoryTag);
        
        // for all decks add a deck tag
        for (int i = 0; i < category.getNumberOfDecks(); i++)
        {
            Element deckTag = document.createElement("Deck");
            categoryTag.appendChild(deckTag);
            
            // for all cards add a card tag
            for (Iterator it = category.getLocalCards(i).iterator(); it.hasNext();)
            {
                Card card = (Card)it.next();
                
                Element cardTag = document.createElement("Card");
                
                // save card sides
                cardTag.setAttribute("Frontside", card.getFrontSide());
                cardTag.setAttribute("Backside", card.getBackSide());
                
                // save dates
                cardTag.setAttribute("DateCreated", m_dateFormat.format(card.getDateCreated()));
                cardTag.setAttribute("DateTouched", m_dateFormat.format(card.getDateTouched()));
                
                if (card.getDateTested() != null)
                {
                    cardTag.setAttribute("DateTested", m_dateFormat.format(card.getDateTested()));
                }
                if (card.getDateExpired() != null)
                {
                    cardTag.setAttribute("DateExpired", m_dateFormat.format(card.getDateExpired()));
                }
                
                // save stats
                cardTag.setAttribute("TestsTotal", Integer.toString(card.getTestsTotal()));
                cardTag.setAttribute("TestsHit", Integer.toString(card.getTestsHit()));
                
                
                deckTag.appendChild(cardTag);
            }
        }
        
        // now add child categories
        for (Iterator it = category.getChildCategories().iterator(); it.hasNext();)
        {
            Category child = (Category)it.next();
            writeCategory(document, categoryTag, child);
        }
    }
    
    private static void fillCategory(Category category, Category father, Element categoryTag, int depth) //CHECK refactor/rename vards
    {
        // for all child tags in category tag
        int deckLevel = 0;
        NodeList childs  = categoryTag.getChildNodes();
        for(int i = 0; i < childs.getLength(); i++)
        { 
            Node child = childs.item(i);
            
            // if deck tag
            if (child.getNodeName().equalsIgnoreCase("Deck"))
            {
                // for all card tags in deck tag
                NodeList childTags = child.getChildNodes();
                for (int j = 0; j < childTags.getLength(); j++)
                {
                    Node childTag = childTags.item(j);
                    
                    // if its a card child tag
                    if (childTag.getNodeName().equalsIgnoreCase("Card"))
                    {
                        NamedNodeMap attributes = childTag.getAttributes();
                        
                        // read front/backside
                        String frontSide = attributes.getNamedItem("Frontside").getNodeValue();
                        String backSide  = attributes.getNamedItem("Backside").getNodeValue();
                        
                        // read dates
                        Date dateCreated = readDate(attributes, "DateCreated");
                        Date dateTested  = readDate(attributes, "DateTested");
                        Date dateExpired = readDate(attributes, "DateExpired");
                        Date dateTouched = readDate(attributes, "DateTouched");
                        
                        // HACK remove this later
                        if (dateCreated == null)
                        {
                            dateCreated = dateTested != null ? dateTested : new Date();
                        }
                        if (dateTouched == null)
                        {
                            dateTouched = dateTested != null ? dateTested : dateCreated;
                        }
                        
                        // read stats
                        int testsTotal = readInt(attributes, "TestsTotal");
                        int testsHit   = readInt(attributes, "TestsHit");
                        
                        // create card
                        Card card = new Card(dateCreated, frontSide, backSide);
                        card.setDateTested(dateTested);
                        card.setDateExpired(dateExpired);
                        card.setDateTouched(dateTouched);
                        card.incStats(testsHit, testsTotal);
                        
                        category.addCard(card, deckLevel);
                    }
                }
                
                deckLevel++;
            }
            // if category tag
            else if (child.getNodeName().equalsIgnoreCase("Category"))
            {
                Element catTag = (Element)child;
                
                String name = catTag.getAttribute("name");
                Category childCategory = new Category(name);
                category.addCategoryChild(childCategory);
                
                fillCategory(childCategory, category, catTag, depth + 1);
            }
        }
    }
    
    private static int readInt(NamedNodeMap attributes, String attributeItem)
    {
        Node num = attributes.getNamedItem(attributeItem);
        
        if (num != null)
        {
            return Integer.parseInt(num.getNodeValue());
        }
        
        return 0;
    }
    
    private static Date readDate(NamedNodeMap attributes, String attributeItem)
    {
        Node date = attributes.getNamedItem(attributeItem);
        
        if (date != null)
        {
            try
            {
                return m_dateFormat.parse(date.getNodeValue());
            }
            catch (Exception e)
            {
                // TODO log
            }
        }
        
        return null;
    }
    
    /**
     * Construct a lesson from a XML document.
     * 
     * @param xmlString The xml document.
     */
    static Lesson fromXML(File xmlFile) throws SAXException, IOException, ParserConfigurationException
    {
        GZIPInputStream gzipIn = new GZIPInputStream(new FileInputStream(xmlFile));
        
        // get lesson tag
        try
        {
            Document doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(gzipIn);

            // there must be a root category
            Element categoryTag = (Element) doc.getElementsByTagName("Category").item(0);

            Category rootCategory = new Category(categoryTag.getAttribute("name"));
            fillCategory(rootCategory, null, categoryTag, 0);
            
            // create lesson
            Lesson lesson = new Lesson();
            lesson.setRootcategry(rootCategory);
            lesson.setFile(xmlFile);
            lesson.setModified(false);
            
            return lesson;
        }
        finally
        {
            if (gzipIn != null)
            {
                gzipIn.close();
            }
        }
    }
    
    private void setModified(boolean b) //CHECK
    {
        if (b != m_modified)
        {
            //TODO fire modified event
            m_modified = b;
        }
    }
    
    private void setRootcategry(Category category)
    {
        m_rootCategory.removeObserver(this);
        m_rootCategory = category;
        m_rootCategory.addObserver(this);
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCategoryEvent(int, jmemorize.core.Category)
     */
    public void onCategoryEvent(int type, Category category)
    {
        setModified(true);
    }

    /*
     * @see jmemorize.core.CategoryObserver#onCardEvent(int, jmemorize.core.Card, int)
     */
    public void onCardEvent(int type, Card card, int deck)
    {
        if (type != EXPIRED_EVENT)
        {
            setModified(true);
        }
    }
}
