/*
 * Created on 14-Apr-2004
 */
package jmemorize.core;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * @author djemili
 */
public class Category implements Events
{
    // use CopyOnWriteArrayList in Java1.5
    private List     m_observers = new ArrayList(); 
    
    private String   m_name;
    private int      m_depth = 0;               // is 0 for root category
    
    private List     m_decks = new ArrayList(); // list of card lists
    
    private Category m_father;
    private List     m_childCategories = new LinkedList();
    
    
    /**
     * Creates a new Category.
     * 
     * @param name The name of the new category.
     */
    public Category(String name)
    {
        m_name   = name;
    }
    
    /*
     * Card related methods.
     */
    
    public void addCard(Card card)
    {
        addCard(card, 0);
    }
    
    /**
     * Adds a card to a level and fires an event. 
     */
    public void addCard(Card card, int level)
    {
        addCardInternal(card, level);
        
        fireCardEvent(ADDED_EVENT, card, level);
    }
    
    /**
     * Remove this card from its associated deck and fires an event.
     */
    public void removeCard(Card card)
    {
        int level = card.getLevel();
        
        removeCardInternal(card);
        
        fireCardEvent(REMOVED_EVENT, card, level);
    }
    
    /**
     * Card is removed from its current deck and added to the next deck. The
     * card is also added with the new expiration date.
     */
    public void raiseCardLevel(Card card, Date testDate, Date newExpirationDate)
    {
        card.incStats(1, 1);
        changeCardLevel(card, card.getLevel() + 1, testDate, newExpirationDate);
    }
    
    /**
     * Card is appended to deck 0 (even if its already at level 0). TotalTests
     * is increased by one.
     */
    public void resetCardLevel(Card card, Date testDate)
    {
        card.incStats(0, 1);
        changeCardLevel(card, 0, testDate, null);
    }
    
    /**
     * Card is taken from its current deck and reappended at the end of the same
     * deck. This doesnt change any values besides DateTouched.
     */
    public void reappendCard(Card card)
    {
        card.setDateTouched(new Date());
        
        card.getCategory().fireCardEvent(DECK_EVENT, card, card.getLevel());
    }
    
    /**
     * Card is appended to deck 0 (even if its already at level 0). Stats are
     * set back to 0 too.
     */
    public void resetCard(Card card) //HACK
    {
        card.resetStats();
        changeCardLevel(card, 0, null, null);
    }
    
    /*
     * Card getter methods
     */
    
    public List getCards()
    {
        List cardList = new ArrayList();
        
        //get cards from all decks
        for (int i=0; i < m_decks.size(); i++)
        {
            cardList.addAll(getCards(i));
        }
        
        return cardList;
    }
    
    public List getCards(int level)
    {
        if (level >= getNumberOfDecks())
        {
            return new ArrayList(); //HACK
        }
        
        //get cards in this category
        List cardList = new ArrayList((List)m_decks.get(level));
        
        //get cards in child categories
        for (Iterator it = getChildCategories().iterator(); it.hasNext();)
        {
            Category child = (Category)it.next();
            
            if (child.getNumberOfDecks() > level)
            {
                cardList.addAll(child.getCards(level));
            }
        }
        
        return cardList;
    }
    
    public List getExpiredCards()
    {
        List expiredCards = getCards();
        
        for (Iterator it = expiredCards.iterator(); it.hasNext();)
        {
            Card card = (Card)it.next();
            if (!card.isExpired())
            {
                it.remove();
            }
        }
        
        return expiredCards;
    }
    
    public List getExpiredCards(int level)
    {
        List expiredCards = getCards(level);
        
        for (Iterator it = expiredCards.iterator(); it.hasNext();)
        {
            Card card = (Card)it.next();
            if (!card.isExpired())
            {
                it.remove();
            }
        }
        
        return expiredCards;
    }
    
    public List getLearnedCards()
    {
        List learnedCards = getCards();
        
        for (Iterator it = learnedCards.iterator(); it.hasNext();)
        {
            Card card = (Card)it.next();
            if (!card.isLearned())
            {
                it.remove();
            }
        }
        
        return learnedCards;
    }
    
    public List getLearnedCards(int level)
    {
        // level 0 decks have no learned cards
        if (level == 0)
        {
            return new ArrayList();
        }
        
        List learnedCards = getCards(level);        
        for (Iterator it = learnedCards.iterator(); it.hasNext();)
        {
            Card card = (Card)it.next();
            if (!card.isLearned())
            {
                it.remove();
            }
        }
        
        return learnedCards;        
    }
    
    /**
     * Unlearned cards (all cards in deck 0) and expired cards are learnable. 
     */
    public List getLearnableCards(int level)
    {
        return level == 0 ? getCards(0) : getExpiredCards(level);
    }
    
    public List getLearnableCards()
    {
        List learnableCards = new LinkedList();
        for (int i = 0; i < getNumberOfDecks(); i++)
        {
            learnableCards.addAll(getLearnableCards(i));
        }
        
        return learnableCards;
    }
    
    public List getUnlearnedCards()
    {
        return m_decks.size() > 0 ? getCards(0) : new ArrayList();
    }
    
    /**
     * @return The cards that are local for this category. That is all cards in
     * this deck that dont belong to child decks.
     */
    public List getLocalCards(int level)
    {
        return (List)m_decks.get(level);
    }
    
    /**
     * @return The number of decks.
     */
    public int getNumberOfDecks()
    {
        return m_decks.size();
    }    
    
    /*
     * Category related methods.
     */
    
    /**
     * @return Returns the childCategories.
     */
    public List getChildCategories()
    {
        return m_childCategories;
    }
    
    /**
     * Appends the category at end of category child list.
     */
    public Category addCategoryChild(Category category)
    {
        return addCategoryChild(m_childCategories.size(), category);
    }
    
    /**
     * Inserts the category at specified position and fires a event.
     */
    public Category addCategoryChild(int position, Category category)
    {
        category.m_father = this;
        category.m_depth  = m_depth + 1;
        
        m_childCategories.add(position, category);

        fireCategoryEvent(ADDED_EVENT, category);
        
        return category;
    }
    
    public void remove()
    {
        assert m_father != null : "Root category cant be deleted";

        m_father.m_childCategories.remove(this);
        fireCategoryEvent(REMOVED_EVENT, this);
    }
    
    /**
     * @return True if given category is a child of this category. False otherwise.
     */
    public boolean contains(Category category)
    {
        if (this == category)
        {
            return true;
        }
        
        for (Iterator it = m_childCategories.iterator(); it.hasNext();)
        {
            Category cat = (Category) it.next();
            if (cat.contains(category))
            {
                return true;
            }
        }
        
        return false;
    }
    
    public Category getFather()
    {
        return m_father;
    }
    
    public void setName(String newName)
    {
        m_name = newName;
        
        fireCategoryEvent(EDITED_EVENT, this);
    }
    
    /**
     * @return The name of this category.
     */
    public String getName()
    {
        return m_name;
    }
    
    public String getPath()
    {
        return m_father != null ? m_father.getPath() +  "/" + getName() : getName();
    }
    
    /**
     * @return Number of hops from this node to root.
     */
    public int getDepth()
    {
        return m_depth;
    }
    
    /**
     * @return A list of all child categories and their childs etc.
     */
    public List getSubtreeList()
    {
        List list = new ArrayList(m_childCategories.size() + 1);
        
        list.add(this);
        for (Iterator it = m_childCategories.iterator(); it.hasNext();)
        {
            Category category = (Category)it.next();
            list.addAll(category.getSubtreeList());
        }
        
        return list;
    }
    
    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return "Category("+m_name+")";
    }
    
    /*
     * Event related methods
     */
    
    public void addObserver(CategoryObserver observer)
    {
        m_observers.add(observer);
    }
    
    public void removeObserver(CategoryObserver observer)
    {
        m_observers.remove(observer);
    }
    
    void fireCardEvent(int type, Card card, int deck)
    {
        if (type != EDITED_EVENT)
        {
            adjustNumberOfDecks();
        }
        
        if (m_father != null)
        {
            m_father.fireCardEvent(type, card, deck);
        }
        
        List observersCopy = new ArrayList(m_observers);
        for (Iterator it = observersCopy.iterator(); it.hasNext();)
        {
            CategoryObserver observer = (CategoryObserver)it.next();
            observer.onCardEvent(type, card, deck);
        }
    }
    
    void fireCategoryEvent(int type, Category category)
    {
        adjustNumberOfDecks();
        
        if (m_father != null)
        {
            m_father.fireCategoryEvent(type, category);
        }
        
        List observersCopy = new ArrayList(m_observers);
        for (Iterator it = observersCopy.iterator(); it.hasNext();)
        {
            CategoryObserver observer = (CategoryObserver)it.next();
            observer.onCategoryEvent(type, category);
        }
    }
    
    /**
     * Adds a card to this category without emitting a ADDED_EVENT. 
     */
    private void addCardInternal(Card card, int level)
    {
        // check boundary
        while (m_decks.size() <= level)
        {
            m_decks.add(new ArrayList());
        }
        
        List cards = (List)m_decks.get(level);
        cards.add(card);
        
        card.setCategory(this);
        card.setLevel(level);
    }
    
    /**
     * Removes a card from this category without emitting a REMOVED_EVENT. 
     */
    private void removeCardInternal(Card card)
    {
        Category cat = card.getCategory();
        if (cat == this)
        {
            int level = card.getLevel();
            List cards = (List)m_decks.get(level);
            cards.remove(card);
            
            card.setCategory(null);
        }
        else
        {
            cat.removeCardInternal(card);
        }
    }
    
    /**
     * Changes the deck level of card and emitts a DECK_EVENT. 
     */
    private void changeCardLevel(Card card, int newLevel, Date newTest, Date newExpiration)
    {
        Category category = card.getCategory();
        int level = card.getLevel();
        
        category.removeCardInternal(card);
        
        card.setDateTested(newTest);
        card.setDateExpired(newExpiration);
        card.setDateTouched(new Date());
        
        // note also that new expiration date is set before adding again
        category.addCardInternal(card, newLevel);
        
        category.fireCardEvent(DECK_EVENT, card, level);
    }
    
    private void adjustNumberOfDecks()
    {
        // find child category with most decks
        int maxChildDecks = 0;
        for (Iterator it = m_childCategories.iterator(); it.hasNext();)
        {
            Category child = (Category)it.next();
            if (child.getNumberOfDecks() > maxChildDecks)
            {
                maxChildDecks = child.getNumberOfDecks();
            }
        }
        
        //grow decks
        while (maxChildDecks > getNumberOfDecks())
        {
            m_decks.add(new ArrayList());
        }
        
        //trim decks
        while (maxChildDecks < getNumberOfDecks() 
            && ((List)m_decks.get(getNumberOfDecks()-1)).isEmpty() )
        {
            m_decks.remove(getNumberOfDecks()-1);
        }
    }
}
