/*
 * Created on 23-Mar-2004
 */
package jmemorize.core;

import java.util.Date;

/**
 * @author djemili
 */
public class Card implements Events, Cloneable
{
    public static final long ONE_DAY = 1000*60*60*24;
    
    private Category m_category;
    private int      m_level;
    
    // text
    private String   m_frontSide;
    private String   m_backSide;
    
    // dates
    private Date     m_dateTested;
    private Date     m_dateExpired;
    private Date     m_dateCreated;
    private Date     m_dateTouched; //this date is used internaly to order cards
    
    // stats
    private int      m_testsTotal;
    private int      m_testsHit;    //succesfull learn repetitions
    
    public Card(String frontSide, String backSide) throws IllegalArgumentException
    {
        this(new Date(), frontSide, backSide);
        
        // do this here because we only want to check this condition while creating
        // new cards from GUI and not while loading cards from lesson file.
        if (frontSide.length() == 0 || backSide.length() == 0)
        {
            throw new IllegalArgumentException("String arguments can't have zero-length.");
        }
    }
    
    /**
     * @throws IllegalArgumentException
     */
    public Card(Date dateCreated, String frontSide, String backSide)
    {
        m_dateCreated = dateCreated;
        m_dateTouched = dateCreated;
        
        m_frontSide   = frontSide;
        m_backSide    = backSide;
    }
    
    /**
     * @throws IllegalArgumentException If frontSide or backSide has no text.
     */
    public void setSides(String frontSide, String backSide) throws IllegalArgumentException
    {
        if (frontSide.length() == 0 || backSide.length() == 0)
        {
            throw new IllegalArgumentException("String arguments can't have zero-length.");
        }
        
        if (m_frontSide.equals(frontSide) && m_backSide.equals(backSide))
        {
            return;
        }
        
        m_frontSide = frontSide;
        m_backSide  = backSide;
            
        m_category.fireCardEvent(EDITED_EVENT, this, m_level);
    }

    public String getFrontSide()
    {
        return m_frontSide;
    }
    
    public String getBackSide()
    {
        return m_backSide;
    }
    
    public Date getDateTested()
    {
        return m_dateTested;
    }
    
    public void setDateTested(Date date)
    {
        m_dateTested = date;
        m_dateTouched = date;
    }
    
    public Date getDateExpired()
    {
        return m_dateExpired;
    }
    
    public void setDateExpired(Date date) // CHECK should this throw a event?
    {
        m_dateExpired = date;
    }
    
    public Date getDateCreated()
    {
        return m_dateCreated;
    }
    
    public void setDateCreated(Date date)
    {
        m_dateCreated = date;
    }

    /**
     * @return DateTouched is the date that this card was learned, skipped,
     *         reset or created the last time. This value is used to sort cards
     *         by a global value that is unique for all categories and decks.
     */
    public Date getDateTouched()
    {
        return m_dateTouched;
    }
    
    public void setDateTouched(Date date)
    {
        m_dateTouched = date;
    }
    
    /**
     * @return Number of times this card has been tested.
     */
    public int getTestsTotal()
    {
        return m_testsTotal;
    }
    
    /**
     * @return Number of times this card has been tested succesfully.
     */
    public int getTestsHit()
    {
        return m_testsHit;
    }
    
    public void incStats(int hit, int total)
    {
        m_testsTotal += total;
        m_testsHit += hit;
    }
    
    public void resetStats()
    {
        m_testsTotal = 0;
        m_testsHit = 0;
    }
    
    public Category getCategory()
    {
        return m_category;
    }
    
    void setCategory(Category category)
    {
        m_category = category;
    }
    
    /**
     * A card is expired when it was learned/repeated succesfully, but its learn
     * time has expired (is in the past from current perspective).
     * 
     * @return True if the card has expired.
     */
    public boolean isExpired()
    {
        return m_dateExpired != null && m_dateExpired.before(Main.getNow());
    }
    
    /**
     * A card is learned when it was learned/repeated succesfully and its learn 
     * time hasnt expired.
     * 
     * @return True if the card is learned.
     */
    public boolean isLearned()
    {
        return m_dateExpired != null && m_dateExpired.after(Main.getNow());
    }
    
    /**
     * A card is unlearned when it wasnt succesfully repeated or never been l
     * earned at all.
     * 
     * @return True if the card is unlearned.
     */
    public boolean isUnlearned()
    {
        // learned and expiration date must both be set or unset.
        assert (m_dateTested == null) == (m_dateExpired == null);
        
        return m_dateExpired == null;
    }
    
    /**
     * @return Returns the level.
     */
    public int getLevel()
    {
        return m_level;
    }
    
    /**
     * @param level The level to set.
     */
    public void setLevel(int level)
    {
        m_level = level;
    }
    
    /*
     * @see java.lang.Object#clone()
     */
    public Object clone() throws CloneNotSupportedException
    {
        return super.clone();
    }
    
    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        return "Card("+m_frontSide+")";
    }
}
