#include "petsc.h"
#include "parpre_pc.h"
#include "src/sles/pc/pcimpl.h"
#include "ml_impl.h"

/****************************************************************
 * User Interface                                               *
 ****************************************************************/

#undef __FUNC__
#define __FUNC__ "AMLSetFillMethod"
/*@
  AMLSetFillMethod - specify the method of handling fill-in when
  eliminating to the next level.

  Parameters:
+ pc - the preconditioner
- fill_method - choice of FillNone, FillDiag, FillStrong, FillFull

@*/
int AMLSetFillMethod(PC pc,AMLFillMethod fill_method)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Fill method can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->fill_method = fill_method;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetSolutionScheme"
/*@
  AMLSetSolutionScheme - set the solution scheme to be used
  in making the two-level transfer

  Parameters:
+ pc - the preconditioner
- solve_scheme - choice of AMLSolveMG, AMLSolveILU
@*/
int AMLSetSolutionScheme(PC pc,AMLSolveScheme solve_scheme)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Solve scheme can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->solve_scheme = solve_scheme;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetCoarseGridDependent"
/*@
  AMLSetCoarseGridDependent - declare that the coarse is to be
  takes as the complement of an independent set

  Parameters:
. pc - the preconditioner
@*/
int AMLSetCoarseGridDependent(PC pc)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Grid choice can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->grid_choice = AMLCoarseGridDependent;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetCoarseGridIndependent"
/*@
  AMLSetCoarseGridDependent - declare that the coarse is to be
  takes as an independent set

  Parameters:
. pc - the preconditioner
@*/
int AMLSetCoarseGridIndependent(PC pc)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Grid choice can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->grid_choice = AMLCoarseGridIndependent;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetSchurElimination"
/*@
  AMLSetSchurElimination - declare that the coarse grid operator
  is to be formed by ILU-type elimination

  Parameters:
. pc - the preconditioner
@*/
int AMLSetSchurElimination(PC pc)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Schur choice can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->schur_choice = AMLSchurElimination;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetSchurVariational"
/*@
  AMLSetSchurVariational - declare that the coarse grid operator
  is to be formed by multigrid-type variational elimination

  Parameters:
. pc - the preconditioner
@*/
int AMLSetSchurVariational(PC pc)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Schur choice can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->schur_choice = AMLSchurVariational;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetCutoffSize"
/*@
  AMLSetCutoffSize - set the cutoff size below which no further levels
  will be generated

  Parameters:
+ pc - the preconditioner
- siz - the minimum size for a subdomain; when one subdomain falls below
  this threshold, the whole method switches to a single solver application.
  @*/
int AMLSetCutoffSize(PC pc,int siz)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Cutoff sizes can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->cutoff = siz;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetStrongRatio"
/*@
  AMLSetStrongRatio - set the threshold for off-diagonal connections
  to be considered strong connections.

  Parameters:
+ pc - the preconditioner
- weight - a fraction
@*/
int AMLSetStrongRatio(PC pc,double weight)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Strong ratio can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->weight = weight;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetCycleDegree"
/*@
  AMLSetCycleDegree - set the degree of the generalised W-cycle

  Parameters:
+ pc - the preconditioner
- deg - a positive integer
@*/
int AMLSetCycleDegree(PC pc,int deg)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Cutoff sizes can only be set for AML");
  if (deg<1) SETERRQ(1,0,"Degree has to be positive");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->degree = deg;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetModification"
/*@
  AMLSetModification - specify the use of modified factorisation

  Parameters:
+ pc - the preconditioner
- mod - 0 for no modification, 1 for modification
@*/
int AMLSetModification(PC pc,int mod)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Modifiation only be set for AML");
  if (mod<0) SETERRQ(1,0,"Modification has to be non-negative");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->modification = mod;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSet11JacobiIterations"
int AMLSet11JacobiIterations(PC pc,int it)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"11Block iterations can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->it11 = it;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetSmootherChoice"
/*@
  AMLSetSmootherChoice - specify the smoother used in a multilevel method

  Parameters:
+ pc - the preconditioner
- smoother - choice of AMLSmoothNone, AMLPreSmooth, AMLPostSmooth, 
	      AMLPrePostSmooth

@*/
int AMLSetSmootherChoice(PC pc,AMLSmootherChoice smoother)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Smoothers can only be set for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->smoother_choice = smoother;
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetISFromStrong"
/*@
  AMLSetISFromStrong - specify that the independent set to be used
  in deriving the coarse grid is formed on basis of the matrix of
  strong connections.

  Parameter:
. pc - the preconditioner
@*/
int AMLSetISFromStrong(PC pc)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;

  PetscFunctionBegin;
  pc_data->mis_from_strong = 1;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetISFromOriginal"
/*@
  AMLSetISFromOriginal - specify that the independent set to be used
  in deriving the coarse grid is formed on basis of the original matrix.

  Parameter:
. pc - the preconditioner
@*/
int AMLSetISFromOriginal(PC pc)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;

  PetscFunctionBegin;
  pc_data->mis_from_strong = 0;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetTransfer"
int AMLSetTransfer(PC pc)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Transfer only defined for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->transfer = 1;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetNoTransfer"
int AMLSetNoTransfer(PC pc)
{
  int ierr,flag;

  PetscFunctionBegin;
  ierr = PCHasType(pc,PCMultiLevel,&flag); CHKERRQ(ierr);
  if (!flag) SETERRQ(1,0,"Transfer only defined for AML");
  {
    PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
    pc_data->transfer = 0;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetPivotRepair"
/* how to deal with negative pivots during factorisation:
   0 = leave alone
   1 = Kershaw repair
   2 = on-processor Manteuffel
   3 = globally synchronised Manteuffel */
int AMLSetPivotRepair(PC pc,int rep)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;

  PetscFunctionBegin;
  pc_data->pivot_repair = rep;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLSetTraceLevel"
int AMLSetTraceLevel(PC pc,int lev)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;

  PetscFunctionBegin;
  pc_data->trace_level = lev;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLorth"
int AMLorth(PC pc)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;

  PetscFunctionBegin;
  pc_data->orth = 1;
  PetscFunctionReturn(0);
}
