####################################
package RIB::Parser;
use strict;
use RIB::Class;
use XML::DOM;
use Data::Dumper;
use RIB::VocabularyTerm;

sub new {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $self  = {
    'CLASSES'  => [],
    'ERRORMSG' => undef
  };
  bless ($self, $class);
  return $self;
}

sub classes {
  my $self = shift;
  if (@_) { @{ $self->{'CLASSES'} } = @_ }
  return $self->{'CLASSES'};
}

sub getClass {
  my ($self,$class_name) = @_;
  my $tmp;
  foreach $tmp (@{$self->classes}) {
    if ($tmp->name eq $class_name) {
      return $tmp;
    }
  }
  $self->errormsg("Can't locate $class_name in configuration file");
  return undef;
}

sub push_class {
  my $self = shift;
  my $class = shift;
  push (@{$self->{'CLASSES'}}, $class);
}

sub errormsg {
  my $self = shift;
  if (@_) { $self->{'ERRORMSG'} = shift }
  return $self->{'ERRORMSG'};
}

sub parse_config_file {
  my ($self,$fn) = @_;
  my $cached = "$fn.dd";
  stat($cached);
  if (-r _ and -s _ and (-M _ < -M $fn)) {
    open(INPUT, $cached);
    my $t = $/; undef ($/); my $file=<INPUT>; $/=$t;
    close INPUT;
    no strict 'vars';
    my $tmp = eval "my $file";
    use strict 'vars';
    if ((!$@) and $tmp->{'CLASSES'}) {
      $self->{'CLASSES'} = $tmp->{'CLASSES'};
      return 1;
    }
  }
  unless (open (INPUT, $fn)) {
    $self->errormsg("Can't open $fn : $!");
    return 0;
  }
  my $tmp=$/; undef($/); my $file=<INPUT>; $/=$tmp;
  close INPUT;
  $self->parse_config($file) || return 0;
  open (OUTPUT, ">$cached");
  $Data::Dumper::Indent = 0;
  $Data::Dumper::Deepcopy = 1; # handle internal refs
  print OUTPUT Data::Dumper::Dumper($self);
  close OUTPUT;
  chmod(0644,$cached); # this file gets eval'd, so must be careful
  return 1;
}

sub parse_config {
  my ($self,$buf) = @_;
  my $parser = new XML::DOM::Parser;
  my $doc = $parser->parse($buf);
  my $classNodeList = $doc->getElementsByTagName("class");
  # CLASSES
  foreach my $i (0 .. $classNodeList->getLength-1) {
    my $classNode = $classNodeList->item($i);
    next unless $classNode->getNodeTypeName eq "ELEMENT_NODE";
    my $class = new RIB::Class;
    $class->name($classNode->getAttribute("name"));
    $class->extends($classNode->getAttribute("extends"));
    # ATTRIBUTES
    my $attrNodeList = $classNode->getElementsByTagName("attribute");
    foreach my $j (0 .. $attrNodeList->getLength()-1) {
      my $attrNode = $attrNodeList->item($j);
      next unless $attrNode->getNodeTypeName eq "ELEMENT_NODE";
      my $name = $attrNode->getAttribute("name");
      if (my $a = $class->get_attribute($name)) {
        $a->push_value($attrNode->getFirstChild()->getNodeValue());
      } else {
        my $attr = new RIB::Attribute;
        $attr->name($name);
        $attr->status($attrNode->getAttribute("status"));
        $attr->cardinality($attrNode->getAttribute("cardinality"));
        $attr->display($attrNode->getAttribute("display"));
        $attr->dtype($attrNode->getAttribute("dtype"));
        $attr->alt($attrNode->getAttribute("alt"));
        # build vocabulary if exists
        my $termNodeList = $attrNode->getElementsByTagName("term");
        if ($termNodeList->getLength() > 0) {
          my $rvt = new RIB::VocabularyTerm("root");
          my $nodeList = $attrNode->getChildNodes(); # assign forces scalar context
          &addTerms($self,$nodeList,$rvt);
          $attr->vocabulary($rvt);
        }
        elsif ($attrNode->getFirstChild()) {
          $attr->push_value($attrNode->getFirstChild->getNodeValue());
        }
        $class->push_attribute($attr);
      }
    }
    # RELATIONSHIPS
    my $relNodeList = $classNode->getElementsByTagName("relationship");
    foreach my $j (0 .. $relNodeList->getLength()-1) {
      my $relNode = $relNodeList->item($j);
      next unless $relNode->getNodeTypeName eq "ELEMENT_NODE";
      my $name = $relNode->getAttribute("name");
      if (my $r = $class->get_relationship($name)) {
        $r->push_value($relNode->getFirstChild()->getNodeValue());
      } else {
        my $rel = new RIB::Relationship;
        $rel->name($relNode->getAttribute("name"));
        if ($rel->name =~ /^\w+\.(\w+)/) {
          $rel->dest($1);
        } else {
          $self->errormsg("Bad relationship name : " . $rel->name);
          return 0;
        }
        $rel->status($relNode->getAttribute("status"));
        $rel->cardinality($relNode->getAttribute("cardinality"));
        $rel->display($relNode->getAttribute("display"));
        $rel->alt($relNode->getAttribute("alt"));
        if ($relNode->getFirstChild()) {
          $rel->push_value($relNode->getFirstChild->getNodeValue());
        }
        $class->push_relationship($rel);
      }
    }
    $self->push_class($class);
  }
  # Resolve all of the extends pointers. If one class extends another
  # then add all of the fields from the parent class to the child class.
  foreach my $rc (@{$self->classes()}) {
    my $myExtends = $rc->extends();
    my %alreadyAdded = ();
    while ($myExtends && !($alreadyAdded{$myExtends})) {
      $alreadyAdded{$myExtends} = 1;
      my $classToAdd = $self->getClass($myExtends);
      last unless $classToAdd;
      # copy attributes, unless overridden
      for my $myAttribute (@{$classToAdd->attributes()}) {
        unless ($rc->get_attribute($myAttribute->name())) {
          $rc->push_attribute($myAttribute);
        }
      }
      # copy relationships, unless overridden
      for my $myRelationship (@{$classToAdd->relationships()}) {
        unless ($rc->get_relationship($myRelationship->name())) {
          $rc->push_relationship($myRelationship);
        }
      }
      $myExtends = $classToAdd->extends();
    }
  }
  return 1;
}

sub parse_object {
  my ($self,$buf) = @_;
  my $parser = new XML::DOM::Parser;
  my $doc = $parser->parse($buf);
  my $classNode = $doc->getElementsByTagName("class")->item(0);;
  my $objectNode = $doc->getElementsByTagName("object")->item(0);;
  my $class = new RIB::Class;
  $class->name($classNode->getAttribute("name"));
  # ATTRIBUTES
  my $attrNodeList = $objectNode->getElementsByTagName("attribute");
  foreach my $j (0 .. $attrNodeList->getLength()-1) {
    my $attrNode = $attrNodeList->item($j);
    next unless $attrNode->getNodeType == 1;
    my $name = $attrNode->getAttribute("name");
    if (my $a = $class->get_attribute($name)) {
      $a->push_value($attrNode->getFirstChild()->getNodeValue());
    } else {
      my $attr = new RIB::Attribute;
      $attr->name($name);
      $attr->push_value($attrNode->getFirstChild->getNodeValue());
      $class->push_attribute($attr);
    }
  }
  # RELATIONSHIPS
  my $relNodeList = $objectNode->getElementsByTagName("relationship");
  foreach my $j (0 .. $relNodeList->getLength()-1) {
    my $relNode = $relNodeList->item($j);
    next unless $relNode->getNodeType == 1;
    my $name = $relNode->getAttribute("name");
    if (my $r = $class->get_relationship($name)) {
      $r->push_value($relNode->getFirstChild()->getNodeValue());
    } else {
      my $rel = new RIB::Relationship;
      $rel->name($relNode->getAttribute("name"));
      $rel->push_value($relNode->getFirstChild->getNodeValue());
      $class->push_relationship($rel);
    }
  }
  $self->push_class($class);
  return 1;
}


=pod
sub parse_object {
  my ($self,$buf) = @_;
  my $parser = new XML::DOM::Parser;
  my $doc = $parser->parse($buf);
  my $classesNode = $doc->getElementsByTagName("classes")->item(0);
  my $classNode = $classesNode->getFirstChild();
  while (defined($classNode->getNextSibling())
         and $classNode->getNodeTypeName ne "ELEMENT_NODE") {
    $classNode = $classNode->getNextSibling();
  }
  my $ribClass = new RIB::Class;
  $ribClass->name($classNode->getTagName());
  my $attributesNode = $classNode->getElementsByTagName("attributes")->item(0);
  my $attributeNode = $attributesNode->getFirstChild();
  while (defined $attributeNode) {
    if ($attributeNode->getNodeTypeName() eq "ELEMENT_NODE") {
      my $ribAttribute = new RIB::Attribute;
      $ribAttribute->name($attributeNode->getTagName());
      my $value = $attributeNode->getFirstChild()->getNodeValue();
      $value=~s/^\s+//s; $value=~s/\s+$//s;
      $ribAttribute->push_value($value);
      $ribClass->push_attribute($ribAttribute);
    }
    $attributeNode = $attributeNode->getNextSibling();
  }
  my $relationshipsNode = $classNode->getElementsByTagName("relationships")->item(0);
  my $relationshipNode = $relationshipsNode->getFirstChild();
  while (defined $relationshipNode) {
    if ($relationshipNode->getNodeTypeName() eq "ELEMENT_NODE") {
      my $ribRelationship = new RIB::Relationship;
      $ribRelationship->name($relationshipNode->getTagName());
      my $value = $relationshipNode->getFirstChild()->getNodeValue();
      $value=~s/^\s+//s; $value=~s/\s+$//s;
      $ribRelationship->push_value($value);
      $ribClass->push_relationship($ribRelationship);
    }
    $relationshipNode = $relationshipNode->getNextSibling();
  }
  $self->push_class($ribClass);
  return 1;
}
=cut

sub addTerms {
  my ($self,$nodeList,$rvt) = @_;
  for (my $i=0; $i<$nodeList->getLength(); $i++) {
    my $termNode = $nodeList->item($i);
    next unless $termNode->getNodeTypeName eq "ELEMENT_NODE";
    my $newTerm = new RIB::VocabularyTerm($termNode->getAttribute("value"));
    if ($termNode->getElementsByTagName("term")->getLength() > 0) {
        my $nl = $termNode->getChildNodes();
        &addTerms($self,$nl,$newTerm);
    }
    $rvt->addChild($newTerm);
  }
}

1;
