#BANGLINE

##############################################
use strict;
use RIB::Util;
use RIB::Parser;
use RIB::BIDMParser;  # this is for parsing objects from version < 2.0
use XML::DOM;
use HTML::Entities;
use HTTP::Date;
use Getopt::Std;
use LWP::UserAgent;

###########################################################
# get command line options
my %opts;
getopt('rin',\%opts);
my $repoHandle = $opts{'r'};
my $interopHandle = $opts{'i'};
my $doNotify = 0;
if ($repoHandle !~ /^\d+$/) {
  die "Properly formatted repository handle not supplied to $0";
}
if ($interopHandle !~ /^\d+$/) {
  die "Properly formatted interoperation handle not supplied to $0";
}
if (exists($opts{'n'})) {
  $doNotify = 1;
}

##############################################
my $util = new RIB::Util();
my $riburl = $util->RibUrl;
my $ribdir = $util->RibDir;
my $ua = new LWP::UserAgent;
$ua->agent("RIB/2.1.1 " . $ua->agent);

eval { $util->dbConnect(); };
if ($@) { &error("Can't connect to database : $@"); }


###########################################################
# get repository info
my $sth = $util->dbh->prepare("SELECT name,contact,primary_class,primary_attribute "
                            . "FROM repositories WHERE handle=$repoHandle");
my ($repoName,$repoContact,$primClass,$primAttr);
eval {
  $sth->execute();
  ($repoName,$repoContact,$primClass,$primAttr) = $sth->fetchrow_array();
};
if ($@ or (!($repoName and $primClass))) {
  die "Can't determine repository name, or primary class.";
}

###########################################################
# start the log
my $log = "Interoperation started at " . scalar(localtime()) . "\n"
        . "Name = $repoName\ncontact = $repoContact\n"
        . "Primary class = $primClass\nPrimary Attribute = $primAttr\n";
&checkpoint();

###########################################################
# start the interoperation
$log .= "Resetting registry timestamp.\n";
$sth = $util->dbh->prepare("UPDATE $repoHandle\_interop_registry "
                         . "SET log='', last_attempt=NOW() "
                         . "WHERE handle=$interopHandle");
eval { $sth->execute(); };
if ($@) {
  &error("Can't update interoperation registry. $@");
}


###########################################################
# parse repository configuration file. get vocab for prim attr
my $rp = new RIB::Parser;
my $class;
my @vocabTerms = ();
eval {
  $log .= "Parsing repository's data model file.\n";
  unless($rp->parse_config_file("$ribdir/docRoot/$repoHandle/config.xml")) {
    &error("Can't parse data model file : " . $rp->errormsg());
  }
  $class = $rp->getClass($primClass);
  unless ($class) {
    &error("Primary class $primClass not found in data model."); 
  }
  if ($primAttr) {
    my $attr = $class->get_attribute($primAttr);
    unless ($attr) {
      &error("Primary attribute $primAttr not found in data model."); 
    }
    my $rootNode = $attr->vocabulary();
    if ($rootNode) {
      &getLegalValues($rootNode,\@vocabTerms,"",1);
    }
  }
};
if ($@) {
  &error("Can't parse data model file : $@");
}


###########################################################
# get info for this interoperation
$log .= "Retrieving information from interoperation registry\n";
$sth = $util->dbh->prepare("SELECT name,url FROM $repoHandle\_interop_registry "
                         . "WHERE handle=$interopHandle");
my ($interopName,$interopUrl);
eval {
  $sth->execute();
  ($interopName,$interopUrl) = $sth->fetchrow_array();
};
if ($@) {
  &error("Can't retrieve information from interoperation registry. $@");
}
if (!($interopName and $interopUrl)) {
  &error("interoperation registry information is incomplete. $@");
}

###########################################################
# figure out what objects are already in the interop_objets
# table for this inteorperation
$log .= "Examining current objects from previous runs of this interoperation.\n";
$sth = $util->dbh->prepare("SELECT url,last_modified FROM "
                         . "$repoHandle\_interop_objects "
                         . "WHERE owner_handle=$interopHandle");
my %current_objects = ();
eval {
  $sth->execute();
  while (my $row = $sth->fetchrow_arrayref()) {
    $current_objects{$row->[0]} = $row->[1];
  }
};
if ($@) {
  &error("Can't access cached interoperation objects data. $@");
}

###########################################################
# contact repository
$log .= "Contacting remote repository.\n";
my $content = undef;
eval {
  my $req = HTTP::Request->new('GET' => $interopUrl);
  my $res = $ua->request($req);
  if (!($res->is_success)) {
    $log .= $res->status_line . "\n";
  } else {
    $content = $res->content();
  }
};
if ($@ or (!($content))) {
  &error ("Can't retreive $interopUrl. $@");
}

###########################################################
# examine the output. what version of RIB is it?
my $ribMajorVersion;
my $ribMinorVersion;
my @urls = ();

eval {
  # interoperating with RIB v1.x
  if ($content !~ /^\s*<\?xml /) {
    $log .= "Remote repository version is pre 2.0.\n";
    $log .= "Retrieving list of objects.\n";
    &checkpoint();
    $ribMajorVersion = 1;
    $ribMinorVersion = 0;
    while ($content =~ s/^\s*(\S+)\s*//s) {
      my $url = $1;
      next unless $url =~ /:\/\//;
      if ($url !~ m,/objects/$primClass/,) {
        $log .= "$url doesn't appear to be in primary class\n";
        next;
      }
      my $req = HTTP::Request->new('HEAD' => $url);
      my $res = $ua->request($req);
      if (!($res->is_success)) {
        $log .=  "HEAD request failed for $url : " . $res->status_line . "\n";
        next;
      }
      my $lm1 = $res->headers()->last_modified();
      if (defined $current_objects{$url}) {
        #                              year    mon    day  hour  min   sec
        $current_objects{$url} =~ /^(\d\d\d\d)(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)/;
        my $lm2 = HTTP::Date::str2time("$1-$2-$3 $4:$5:$6");
        if ($lm1 > $lm2) {
          $log .= "Changed : $url\n";
          push (@urls, $url);
        } else {
          $log .= "Unchanged : $url\n";
        }
        delete($current_objects{$url}); # anything left in the hash
                                        # will be deleted from the db
      } else {
        $log .= "New : $url\n";
        push (@urls, $url);
      }
    }
  }
  # interoperating with RIB v > 1.x
  else {
    my $parser = new XML::DOM::Parser;
    my $doc = $parser->parse($content);
    my $ribNode = undef;
    eval { $ribNode = $doc->getFirstChild(); };
    if ($@ or (!($ribNode))) {
      &error("Improperly formatted document at $interopUrl\n");
      exit(0);
    }
    my $ribVersion = $ribNode->getAttribute("version");
    $ribVersion =~ /^(\d+)\.?(\d*)/;
    $ribMajorVersion = $1;
    $ribMinorVersion = $2;
    $log .= "Remote repository version is $ribVersion.\n";
=pod
    if (int($ribMajorVersion)>2 or int($ribMinorVersion)>0) {
      &error("Cannot interoperate with repository version > 2.0\n");
    }
=cut
    $log .= "Retrieving list of objects.\n";
    &checkpoint();
    my $repositoryNode = $ribNode->getFirstChild();
    my $tmpName = $repositoryNode->getAttribute("name");
    my $classNodeList = $repositoryNode->getElementsByTagName("class");
    my $classNode = undef;
    foreach my $i (0 .. $classNodeList->getLength()-1) {
      my $tmpClassNode = $classNodeList->item($i);
      my $className = $tmpClassNode->getAttribute("name");
      if ($className eq $primClass) {
        $classNode = $tmpClassNode;
        last;
      }
    }
    unless ($classNode) {
      &error($interopName . " does not contain a $primClass class.");
    }
    my $objectsUrl = $classNode->getAttribute("objects");
    $objectsUrl = HTML::Entities::decode($objectsUrl);
    unless ($objectsUrl) {
      &error("Cannot determine url for retrieving remote object list\n");
    }
    my $req = HTTP::Request->new('GET' => $objectsUrl);
    my $res = $ua->request($req);
    if (!($res->is_success)) {
      &error("Cannot retrieve objects list at $objectsUrl : " . $res->status_line);
    }
    $content = $res->content();
    $parser = new XML::DOM::Parser;
    $doc = $parser->parse($content);
    my $objectNodeList = undef;
    eval {
      $objectNodeList = $doc->getElementsByTagName("object");
    };
    if ($@ or (!($objectNodeList))) {
      &error("Improperly formatted objects list at $objectsUrl");
    }
    foreach my $i (0 .. $objectNodeList->getLength()-1) {
      my $objectNode = $objectNodeList->item($i);
      my $lastModified = $objectNode->getAttribute("last_modified");
      my $objectUrl = $objectNode->getAttribute("url");
      if ($lastModified > $current_objects{$objectUrl}) {
        push(@urls,$objectUrl);
        $log .= "Changed : $objectUrl\n";
      } else {
        $log .= "Unchanged : $objectUrl\n";
      }
      delete($current_objects{$objectUrl}); # anything left in the hash
                                            # will be deleted from the db
    }
  }
};
if ($@) { &error($@); }

###########################################################
# now fetch the objects and put them in the database
unless (@urls) {
  $log .= "There are no objects to add from this interoperation.\n";
}
foreach my $url (@urls) {
  $log .= "Retrieving $url\n";
  &checkpoint();
  my %fields = ();
  my $nameValue;
  eval {
    my $req = HTTP::Request->new('GET' => $url);
    my $res = $ua->request($req);
    if (!($res->is_success)) {
      $log .=  "Cannot retrive object at $url : " . $res->status_line . "\n";
      next;
    }
    my $content = $res->content();
    if ($ribMajorVersion == 1) {
      my $bp = RIB::BIDMParser->new();
      $bp->parse($content);
      unless ($bp->valuesof("Name")) {
        $log .= "Object does not contain a Name attribute";
        next;
      }
      my @vals = $bp->valuesof("Name");
      $nameValue = $vals[0];
      foreach my $field (@{$class->attributes()},@{$class->relationships()}) {
        foreach my $val ($bp->valuesof($field->name)) {
          push (@{$fields{$field->name}}, $val);
        }
      }
    }
    elsif ($ribMajorVersion > 1) {
      $content =~ s/[^\w\-\~\`\!\@\#\$\%\^\&\*\(\)\_\+\-\=\{\}\|\[\]\\\;\'\:\"\,\.\/\<\>\?\\\s]//g;
      my $rp2 = new RIB::Parser;
      $rp2->parse_object($content);
      my $c = $rp2->getClass($primClass);
      unless ($c->get_attribute("Name")) {
        $log .= "Object does not contain a Name attribute";
        next;
      }
      $nameValue = $c->get_attribute("Name")->values()->[0];
      unless (defined $nameValue) {
        $log .= "Object does not contain a Name attribute";
        next;
      }
      foreach my $attr (@{$class->attributes}) {
        next unless $c->get_attribute($attr->name);
        foreach my $val (@{$c->get_attribute($attr->name)->values()}) {
          push (@{$fields{$attr->name}}, $val);
        }
      }
      foreach my $rel (@{$class->relationships}) {
        next unless $c->get_relationship($rel->name);
        foreach my $val (@{$c->get_relationship($rel->name)->values()}) {
          push (@{$fields{$rel->name}}, $val);
        }
      }
    }
  };
  if ($@) {
    $log .= "Parse error in $url. $@\n";
    next;
  }
  $log .= "Adding $nameValue to catalog.\n";
  &checkpoint();
  my $created = "NULL";
  eval {
    $sth = $util->dbh->prepare("SELECT created FROM $repoHandle\_interop_objects "
                             . " WHERE url="
                             . $util->dbh->quote($url));
    if ($sth->rows() >= 1) {
      $created = $sth->fetchrow_arrayref()->[0];
    }
  };
  if ($@) {
    $log.="Database error : $@\n";
    next;
  }
  eval {
    $util->dbh->do("DELETE FROM $repoHandle\_interop_objects "
                 . " WHERE url="
                 . $util->dbh->quote($url)
                 . " AND owner_handle=$interopHandle");
  };
  if ($@) {
    $log.="Database error : $@\n";
    next;
  }
  my %alreadyUsed = ();

  ###################################################################
  # create a prefix and suffix for any insert statements for this table
  my $prefix = "insert into $repoHandle\_interop_objects (";
  #foreach $field (@{$class->attributes}, @{$class->relationships}) {
  foreach my $field (keys %fields) {
    $field =~ s/\./_/;
    $prefix .= "$field\_, ";
  }
  $prefix .= "url, owner_handle, last_modified, created) VALUES (";
  my $suffix = $util->dbh->quote($url)
             . ", $interopHandle, NULL, $created)";

  my $i = 0;  # gets incremented for each insert
  my $do_another_insert = 1; # keeps track of whether or not another insert will
                             # need to be done after the first.
  my $do_this_insert = 0;    # sometimes an insert should not occur after the SQL
                             # has been assembled. This happens when the value from
                             # the primary attribute repeats itself in an addon row
                             # when no other fields are present.

  eval {
    while ($do_another_insert) {
      $do_another_insert = 0;
      $do_this_insert = 0;
      my $query = '';
      foreach my $field (keys %fields) {
        if ($field eq "Name") {
          $query .= $util->dbh->quote($nameValue).",";
          next;
        }
        my $value = $fields{$field}->[$i];
        my $f;
        unless ($f = $class->get_attribute($field)) {
          next unless $f = $class->get_relationship($field);
        }
        if ($field eq $primAttr and defined $value) {
          my $loops = 0;
          my $maxLoops = 100;
          while (!(grep(/^\Q$value\E$/, @vocabTerms))) {
            $value =~ s/\!?[^!]*$//;
            last if $value eq '';
            if (++$loops > $maxLoops) {
              $value = '';
              last;
            }
          }
          if ($alreadyUsed{$value}) {
            $value = undef;
          }
          $alreadyUsed{$value} = 1;
        }
        # check cardinality
        if ($i == 0 or $f->cardinality ne 'single') {
          if (defined $value) {
            $query .= $util->dbh->quote($value) . ",";
            $do_this_insert = 1;
          } else {
            $query .= "NULL,";
          }
        } else {
          $query .=  "NULL,";
        }
        if ($f->cardinality() ne 'single' and defined($fields{$field}->[$i+1])) {
          $do_another_insert = 1;
        }
      }
      if ($i == 0 or $do_this_insert) {
        $sth = $util->dbh->prepare($prefix . $query . $suffix);
        unless ($sth->execute) {
          $log .= "Error adding $url to catalog : " . $sth->errstr . "\n";
        }
      }
      $i++;
    }
  };
  if ($@) {
    $log .= "Error adding $url to catalog : $@\n";
    next;
  }
}

###########################################################
# remove any objects that have become stale
if (keys %current_objects) {
  $log .= "Removing expired interoperation objects from catalog\n";
  foreach my $url (keys %current_objects) {
    $log .= "  $url\n";
    $sth = $util->dbh->prepare("DELETE FROM $repoHandle\_interop_objects WHERE url="
                             . $util->dbh->quote($url)
                             . " AND owner_handle=$interopHandle");
    eval { $sth->execute(); };
    if ($@) { $log .= "Can't delete $url : $@"; }
  }
}

###########################################################
# update timestampes
$log .= "Resetting interop registry timestamps\n";
$sth = $util->dbh->prepare("UPDATE $repoHandle\_interop_registry SET checkpoint=0, "
                         . " last_success=NOW() WHERE handle=$interopHandle");
eval { $sth->execute(); };
if ($@) { $log .= "Can't reset interop registry timestamps: $@"; }

###########################################################
# close up
$log .= "Interoperation completed at " . scalar(localtime()) . "\n";
&checkpoint();
if ($doNotify) {
  my $req = HTTP::Request->new(POST => "mailto:$repoContact");
  $req->header(Subject => "RIB interoperation completed");
  my  $message = "Repository $repoName has completed an interoperation update with "
               . "$interopName\n\n"
               . "\n\nTo view the log please go to the url :\n\n"
               . "  $riburl/adminRepository.pl?rh=$repoHandle\n\n"
               . "and check the interoperation tab of the RIB interface.\n";
  $req->content($message);
  my $res = $ua->request($req);
  if (!($res->is_success)) {
      print STDERR $res->status_line, "\n";
  }
}
#print $log, "\n";
exit(0);

## here's a bug to go fix: when config gets vocab changed
## some of the entries might get changed without the main
## row last_modified field getting updated.
## 
## answer: clear out everything in interop when config changes
##         iff primary class (attr?) changes

sub error {
  my $message = shift;
  # first checkpoint and then send email to repo contact
  $log .= $message . "\n";
  &checkpoint();
  if ($doNotify) {
    my $req = HTTP::Request->new(POST => "mailto:$repoContact");
    $req->header('Subject' => "RIB interoperation failure");
    $message = "An interoperation update failed for repository $repoName for "
             . "the follwing reason :\n\n  "
             . $message
             . "\n\nTo investigate this problem please go to the url :\n\n"
             . "  $riburl/adminRepository.pl?rh=$repoHandle\n\n"
             . "and check the interoperation tab of the RIB interface.\n";
    $req->content($message);
    my $res = $ua->request($req);
    if (!($res->is_success)) {
        print STDERR $res->status_line, "\n";
    }
  }
  $sth = $util->dbh->prepare("UPDATE $repoHandle\_interop_registry SET "
                           . " last_failure=NOW() WHERE handle=$interopHandle");
  eval { $sth->execute(); };
  exit;
}

sub checkpoint {
  my $sth2 = $util->dbh->prepare("UPDATE $repoHandle\_interop_registry SET log="
                               . $util->dbh->quote($log)
                               . ", checkpoint=null WHERE handle=$interopHandle");
  eval { $sth2->execute(); };
  if ($@) { &error("Interoperation checkpoint failed. " . $@); }
}

sub getLegalValues {
  my ($vocabularyNode, $values, $prefix, $recursion) = @_;
  return if $recursion++ > 200;
  foreach my $child ($vocabularyNode->getChildren()) {
    push (@{$values}, $prefix.$child->term());
    &getLegalValues($child,$values,$prefix.$child->term()."!",$recursion);
  }
}
