package RIB::ObjectCreator;

use strict;
use RIB::Util;
use RIB::Parser;
use RIB::VocabularyTerm;
use HTML::Entities;

sub new {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $self  = { 'PARSER' => undef };
  if (@_) {
    $self->{'PARSER'} = shift;
  }
  bless ($self, $class);
  return $self;
}


sub createObject {
  my $self = shift;
  my $newObject = shift;
  my $repoHandle = shift;
  my $util = shift;

  # if this parameter is supplied then we won't get a new object
  # handle for the object, we'll use the one supplied instead.
  my $seq = shift;

  my $rp = new RIB::Parser;
  my $ribdir = $util->RibDir;
  my $riburl = $util->RibUrl;
  my $sth = undef;

  $newObject =~ s/[^\w\-\~\`\!\@\#\$\%\^\&\*\(\)\_\+\-\=\{\}\|\[\]\\\;\'\:\"\,\.\/\<\>\?\\\s]//g;

  unless ($rp->parse_object($newObject)) {
    die "Couldn't parse new object";
  }
  my $class = $rp->classes->[0];

  # create new parser unless supplied in class instance var
  my $config = $self->{'PARSER'};
  unless ($config) {
    $config = new RIB::Parser;
    $config->parse_config_file("$ribdir/docRoot/$repoHandle/config.xml");
  }

  my $config_class = $config->getClass($class->name());
  if (!$config_class) {
    return ($class->name()
          . " is no longer a valid class name. The class may have been deleted "
          . "from the repository data model.");
  }

  unless ($class->values_of("Name")) {
    return("Object doesn't contain a &quot;Name&quot; attribute");
  }

  my $class_handle = $repoHandle . "_" . $class->name;

  ###################################################################
  # create a prefix and suffix for any insert statements for this table
  my $prefix = "insert into $class_handle (";
  my $field;
  foreach $field (@{$class->attributes}, @{$class->relationships}) {
    if (!($config_class->get_attribute($field->name))) {
      next unless($config_class->get_relationship($field->name));
    }
    my $fieldname = $field->name;
    $fieldname =~ s/\./_/;
    $prefix .= $fieldname . "_, ";
  }
  $prefix .= "handle, extends, approved, created) values (";
  my $suffix = ",0,NULL)";

  ###################################################################
  # get a sequence number for this object
  unless (defined $seq) {
    $seq = $util->getSequence($class_handle);
  }

=pod
  ###################################################################
  Now build queries from the values object submitted to this script
  and submit them to the database.
  The code handles the cases where fields
  in the object have more than one value. In that case another row must
  be added to the table ain this case. Instead of specifying a new handle
  for this new row we point the extends row at the "parent" row's handle.
 
    example: here the 2nd and 3rd rows extends the first
 
             field1  field2  ... handle   extends
             -------------------------------------
              foo     goo    ...   1        NULL
              bar      -     ...   NULL     1
              yoo      -     ...   NULL     1
  ###################################################################
=cut
              
  my $i = 0;  # gets incremented for each insert
  my $do_another_insert = 1; # keeps track of whether or not another insert will
                             # need to be done after the first.

  while ($do_another_insert) {
    $do_another_insert = 0;
    my $query = '';

    foreach $field (@{$class->attributes}, @{$class->relationships}) {
      # make sure field name is valid
      my $f;
      unless ($f = $config_class->get_attribute($field->name)) {
        next unless $f = $config_class->get_relationship($field->name);
      }
      # if has vocab then ensure value is a valid term
      if (my $attr = $config_class->get_attribute($field->name)) {
        my $node = $attr->vocabulary;
        if ($node and defined($field->values->[$i])) {
          my $term = "";
          foreach my $s (split(/!/,$field->values->[$i])) {
            if ($node and my $child=$node->getChild($s)) {
              $term .= "$s!";
              $node = $child;
            } else {
              last;
            }
          }
          chop($term); # remove last '!'
          $field->values->[$i] = $term;
        }
      }
      # check cardinality
      if ($i == 0 or $f->cardinality ne 'single') {
        $query .= $util->dbh->quote($field->values->[$i]) . ",";
      } else {
        $query .=  "NULL,";
      }
      if ($f->cardinality() ne 'single' and defined($field->values->[$i+1])) {
        $do_another_insert = 1;
      }
    }
    if ($i) {
      $query .= "NULL,$seq";
    } else {
      $query .= "$seq,NULL";
    }
    $sth = $util->dbh->prepare($prefix . $query . $suffix);
    unless ($sth->execute) {
      if ($seq) {
        $util->dbh->do("delete from $class_handle where handle=$seq or "
                     . "extends = $seq");
      }
      return ($util->dbh->errstr);
    }
    $i++;
  }

  my $objectUrl = $riburl . "/object.pl?rh=$repoHandle&class="
                . $class->name()
                . "&oh=$seq";
  return $objectUrl;
}
1;
