C-------------------------------------------------------------------------
C  The routine generates the metric "measure" from a given function "u" 
C  and the mesh "tri,vrt,bnd".
C
C  Limitations to the mesh: each boundary node has to have at least
C  one adjacent interior node or one adjacent boundary node which is 
C  adjacent to an interior one.
C-------------------------------------------------------------------------
      Subroutine Metric( u, 
     &                   Vrt,Nvrt, Tri,Ntri, Bnd,Nbnd, measure, 
     &                   Nrmem,rmem, Nimem,imem )
C-------------------------------------------------------------------------
C  Input: u -    function, the basis for the metric
C         Nvrt - the number of nodes
C         Vrt  - coords of the nodes
C         Ntri - the number of triangles
C         Tri  - the connecticity table
C         Nbnd - the number of boundary edges
C         Bnd  - the list of boundary edges
C  Output: measure - metric to be defined. Is nothing else but the discrete 
C                    Hessian reduced to elliptic form
C  Work arrays: rmem - d.p., of length  Nrmem
C               imem - integer, of length Nimem
C                
C-------------------------------------------------------------------------
      implicit None

      include 'magic.fd'
C-------------------------------------------------------------------------
C    Function as the basis
C-------------------------------------------------------------------------     
      double precision u(*)

C-------------------------------------------------------------------------
C   The mesh
C-------------------------------------------------------------------------
      integer  Nvrt,Ntri,Nbnd
      double precision vrt(2,*)
      integer  Tri(3,*), Bnd(4,*)

C-------------------------------------------------------------------------
C    The metric to be defined
C-------------------------------------------------------------------------
      double precision  measure(3,*)

C-------------------------------------------------------------------------
C    Work arrays
C-------------------------------------------------------------------------
c      integer NhostTri(*), HostTri(MaxNeighb,*), BadNode(*)
c      double precision Mass(*)
      integer          Nrmem,Nimem
      integer          imem(*)
      double precision rmem(*)

C-------------------------------------------------------------------------
C   Local arrays for LAPACK
C-------------------------------------------------------------------------
      double precision A(2, 2), E(2), rW(10)
      integer          info

C-------------------------------------------------------------------------
C   Local variables
C-------------------------------------------------------------------------
      double precision DX12,DY12,DX13,DY13,DET,Dxu,Dyu,DxNi,DyNi
      integer i,j,k,m,it,iv,L1,L2,L3, Ni(3), mi,mj,ki,km 
      double precision meas(3),weight(1000)
      integer lbuf, kbuf(1000), kv
C  weight,kbuf: number of triangles sharing a node is a priori less than 1000
      logical flag

C-------------------------------------------------------------------------
C     Check bounds for work arrays
C-------------------------------------------------------------------------
      if (Nrmem.lt.Nvrt.or.Nimem.lt.2*Nvrt+3*Ntri) then
         write(*,*) 'Increase Nrmem to',Nvrt
         write(*,*) 'Increase Nimem to',2*Nvrt+3*Ntri
         stop
      end if


C-------------------------------------------------------------------------
C     Initialize BadNode,NhostTri,HostTri
C-------------------------------------------------------------------------
      do iv = 1, Nvrt
c        NhostTri(iv) = 0
         imem(iv+Nvrt) = 0
      end do

C-------------------------------------------------------------------------
C     Generate NhostTri
C-------------------------------------------------------------------------
      do it = 1, Ntri
         do i = 1, 3
c           NhostTri(tri(i,it)) = NhostTri(tri(i,it)) + 1
            imem(tri(i,it)+Nvrt) = imem(tri(i,it)+Nvrt) + 1
         end do
      end do


C-------------------------------------------------------------------------
c     Pointers to HostTri
C-------------------------------------------------------------------------
      imem( 1 ) = 2*Nvrt + 1
      do iv = 2, Nvrt
         imem( iv ) = imem( iv-1 ) + imem( Nvrt + iv-1 )
      end do
      do iv = 1, Nvrt
         imem( Nvrt + iv ) = 0
      end do
      do it = 1, Ntri
         do i = 1, 3
            iv = tri(i,it)
            k = imem( iv ) + imem( Nvrt + iv )
            imem( k ) = it
            imem( Nvrt + iv ) = imem( Nvrt + iv ) + 1     
         end do
      end do
c  Restore BadNode, NhostTri..
      do iv = 1, Nvrt
c        BadNode(iv) = 1
c        NhostTri(iv) = 0
         imem(iv) = 0
         imem(iv+Nvrt) = 0
      end do
      do it = 1, Ntri
         do i = 1, 3
            imem(tri(i,it)+Nvrt) = imem(tri(i,it)+Nvrt) + 1
         end do
      end do
C-------------------------------------------------------------------------
C     Mark boundary nodes
C-------------------------------------------------------------------------
      do i = 1, Nbnd
c        if (NhostTri(bnd(1,i)).gt.0) 
c           NhostTri(bnd(1,i)) = -NhostTri(bnd(1,i))
         imem(bnd(1,i)+Nvrt) = -abs(imem(bnd(1,i)+Nvrt))
         imem(bnd(1,i)) = 1
         imem(bnd(2,i)+Nvrt) = -abs(imem(bnd(2,i)+Nvrt))
         imem(bnd(2,i)) = 1
      end do

         
C-------------------------------------------------------------------------
C     Initialize Mass and measure
C-------------------------------------------------------------------------
      do iv = 1, Nvrt
c        Mass(iv) = 0d0
         rmem(iv) = 0d0
         measure(1,iv) = 0d0
         measure(2,iv) = 0d0
         measure(3,iv) = 0d0
      end do

C-------------------------------------------------------------------------
C     Recover the Hessian: H_ij  = M^{-1} integral_Om du/dx_i dNi/dx_j d Om
C-------------------------------------------------------------------------
      ki = 1
      do iv = 1, Nvrt
      do i  = 1, iabs(imem(Nvrt+iv))
c      do i  = 1, NhostTri(iv)
c        it = HostTri(i,iv)   
         it = imem(2*Nvrt+ki)
         ki = ki + 1
   
         L1 = tri(1,it)
         L2 = tri(2,it)
         L3 = tri(3,it)

         DX12 = VRT(1,L1)-VRT(1,L2)
         DX13 = VRT(1,L1)-VRT(1,L3)
         DY12 = VRT(2,L1)-VRT(2,L2)
         DY13 = VRT(2,L1)-VRT(2,L3)
         DET = DX12*DY13-DX13*DY12

        if (imem(Nvrt+iv).lt.0) then
          do j = 1, 3
           if (imem(tri(j,it)).eq.0) rmem(iv) = rmem(iv) + DABS(DET)/24
          end do
        else

C        Nodal basis function
         Ni(1) = 0
         Ni(2) = 0
         Ni(3) = 0
         if (iv.eq.L1) Ni(1) = 1d0
         if (iv.eq.L2) Ni(2) = 1d0
         if (iv.eq.L3) Ni(3) = 1d0

c        Mass(iv) = Mass(iv) + DABS(DET)/6d0
         rmem(iv) = rmem(iv) + DABS(DET)/6d0

         Dxu=((u(L1)-u(L2))*DY13 
     &      - (u(L1)-u(L3))*DY12)/DET
         Dyu=(-(u(L1)-u(L2))*DX13 
     &      + (u(L1)-u(L3))*DX12)/DET

         DxNi=((Ni(1)-Ni(2))*DY13 - (Ni(1)-Ni(3))*DY12)/DET
         DyNi=(-(Ni(1)-Ni(2))*DX13 + (Ni(1)-Ni(3))*DX12)/DET

         measure(1,iv) = measure(1,iv)-DABS(DET)*Dxu*DxNi/2d0
         measure(2,iv) = measure(2,iv)-DABS(DET)*Dyu*DyNi/2d0
         measure(3,iv) = measure(3,iv)-DABS(DET)*(Dxu*DyNi+Dyu*DxNi)/4d0

       end if
      end do
      end do

C     Inner nodes..
      do iv = 1, Nvrt
         if (imem(Nvrt+iv).gt.0) then
            measure(1,iv) = measure(1,iv)/rmem(iv)
            measure(2,iv) = measure(2,iv)/rmem(iv)
            measure(3,iv) = measure(3,iv)/rmem(iv)
         end if
      end do

C-------------------------------------------------------------------------
C  Extrapolate the measure from interior nodes to boundary
C  and avearage the extrapolations. Exlude respective nodes from 
C  the set of bad ones.
C-------------------------------------------------------------------------
      ki = 1
      do iv = 1, Nvrt
       if (imem(Nvrt+iv).gt.0) then
          do i = 1, imem(Nvrt+iv)
             ki = ki + 1
          end do
       else
          lbuf = 0
          do i  = 1, -imem(Nvrt+iv)
             it = imem(2*Nvrt+ki)
             ki = ki + 1
             L1 = tri(1,it)
             L2 = tri(2,it)
             L3 = tri(3,it)

             DX12 = VRT(1,L1)-VRT(1,L2)
             DX13 = VRT(1,L1)-VRT(1,L3)
             DY12 = VRT(2,L1)-VRT(2,L2)
             DY13 = VRT(2,L1)-VRT(2,L3)
             DET = DX12*DY13-DX13*DY12
             do j = 1, 3
                kv = tri(j,it)
                flag = .true.
                do k = 1, lbuf
                   if (kbuf(k).eq.kv) then
                      flag = .false.
                      weight(k) = weight(k) + DABS(DET)/24
                   end if
                end do
                if (flag.and.imem(kv).eq.0) then
                   lbuf = lbuf + 1
                   kbuf(lbuf) = kv
                   weight(lbuf) = DABS(DET)/24
                end if
             end do
          end do
          do k = 1, lbuf
             kv = kbuf(k)
             do j = 1, 3
                measure(j,iv) = measure(j,iv) 
     &               + measure(j,kv)*weight(k)/rmem(iv)
             end do
             imem(Nvrt+iv) = iabs(imem(Nvrt+iv))
          end do
       end if
      end do
     
      do iv = 1, Nvrt
         if (imem(iv).ne.0.and.imem(Nvrt+iv).gt.0) then
            imem(Nvrt+iv) = - imem(Nvrt+iv)
            imem(iv)      = 0
         end if
      end do
                
             

C-------------------------------------------------------------------------
C   Treating the BadNodes by secondary extrapolating
C-------------------------------------------------------------------------
      km = 1
      do iv = 1, Nvrt
c       if (BadNode(iv).eq.1) then
        if (imem(iv).eq.1) then
           i = 0
           do k = 1, 3
              meas(k) = 0d0
           end do
c          do m = 1, -NhostTri(iv)
           do m = 1, -imem(Nvrt+iv)
c             it = HostTri(m,iv)
              it = imem(2*Nvrt+km)
              km = km + 1
              do j = 1, 3
c                if (BadNode(tri(j,it)).eq.0) then
                 if (imem(tri(j,it)).eq.0) then
                    i = i + 1
                    do k = 1, 3
                       meas(k) = meas(k) + measure(k,tri(j,it))  
                    end do
                 end if
              end do
           end do
           if (i.eq.0) then
              write(*,*)'Sorry, I do not know what to do... '
              write(*,*)vrt(1,1),vrt(1,2),vrt(1,3)
              write(*,*)vrt(2,1),vrt(2,2),vrt(2,3)
              stop
           else
              do k = 1, 3
                 measure(k,iv) = meas(k)/i
              end do
           end if
        else
           do m = 1, iabs(imem(Nvrt+iv))
              km = km + 1
           end do
        end if
      end do


C-------------------------------------------------------------------------
C   Make the hessian to be elliptic
C-------------------------------------------------------------------------
      do iv = 1, Nvrt
        A(1, 1) = measure(1,iv)
        A(2, 2) = measure(2,iv)
        A(1, 2) = measure(3,iv)

        Call dsyev('V', 'U', 2, A, 2, E, rW, 10, info)
        If(info.NE.0) Call errMes(3011, 'metric2D',
     &                    'Error in Lapack routine dsyev')

        E(1) = dabs(E(1))
        E(2) = dabs(E(2))

        E(1) = max( E(1), E(2) * AniRatio )

        If(E(2).EQ.0D0) Then
           E(1) = AniEigenvalue
           E(2) = AniEigenvalue
        End if

        measure(1,iv) = E(1) * A(1, 1) ** 2 + E(2) * A(1, 2) ** 2
        measure(2,iv) = E(1) * A(2, 1) ** 2 + E(2) * A(2, 2) ** 2
        measure(3,iv) = E(1) * A(1, 1) * A(2, 1) 
     &                + E(2) * A(1, 2) * A(2, 2)
      end do

      return
      end 


