C ================================================================
      Program aniFEM
C ================================================================
      include 'fem2Dtri.fd'
      include 'assemble.fd'
C ================================================================
      Integer   MaxP,  MaxE,  MaxF, MaxPV, MaxEV, MaxFV
      Parameter(MaxP = 20000, MaxF = 10000, MaxE = 2 * MaxP)
      Parameter(MaxPV = MaxP, MaxFV = MaxF, MaxEV = MaxE)

      Integer   MaxIA, MaxA
      Parameter(MaxIA = 100 000, MaxA = 1 000 000)

      Integer   MaxWr, MaxWi
      Parameter(MaxWr = 300 000, MaxWi = 1 000 000)

      Integer   MaxSkipE, MaxQItr, nEStar
      Parameter(MaxSkipE = 100, MaxQItr = 15 000, nEStar = 2000)

      Real*8    Quality
      Parameter(Quality = 7D-1)

      Integer   nLOOPs
      Parameter(nLOOPs = 5)

C ================================================================
C group (M)
      Real*8  XYP(2, MaxP)
      Integer IPE(3, MaxE), IPF(4, MaxF)
      Integer lbE(MaxE)
      Integer IPV(MaxPV), IFV(MaxFV), IEV(MaxEV)

C group (Crv)
      Real*8  ParCrv(2, MaxF)
      Integer iFnc(MaxF)

C group (W)
      Real*8  rW(MaxWr)
      Integer iW(MaxWi)

C group (A)
      Integer IA(MaxIA), JA(MaxA)
      Real*8   A(MaxA), DA(MaxIA), U(MaxIA), F(MaxIA), G(MaxIA)

      Integer IB(MaxIA), JB(MaxA)
      Real*8   B(MaxA)

      Real*8  Aloc(MaxSize, MaxSize)

      Integer iSYS(2)
      Real*8  DATA
      Integer  Ddiff, Dbc, Drhs_scalar, Drhs_vector, Dexact
      EXTERNAL Ddiff, Dbc, Drhs_scalar, Drhs_vector, Dexact

C ================================================================
C Local variables
      Real*8  ECG1, ECG2, EWT2
      Real*8  eps, pi, su, se, sr

      Integer status
      Real*8  rQuality
      Logical flagAuto

      Real    ANItime, tmdata(2)
      Real*8  tm0, tmBF, tmLF, tmBC, tmAMG

      Data    pi/3.14159265358979D0/

C ================================================================
c ... load a mesh
      Call loadMone(
     &      nP,  MaxP,  nF,  MaxF,  nE,  MaxE,
     &      nPv, MaxPV, nFv, MaxFV, nEv, MaxEV,
     &      XYP, IPF, IPE, IPV, IFV, IEV, lbE,
     &      ParCrv, iFnc, "../data/wing")


c ... test for one element
      n = 4
      iP1 = IPE(1, n)
      iP2 = IPE(2, n)
      iP3 = IPE(3, n)

      label = 1
      Call FEM1Dedge(
     &     XYP(1, iP1), XYP(1, iP3),
     &     IDEN, FEM_P1, IDEN, FEM_P1,
     &     label, Drhs_scalar, DATA, 1, 
     &     MaxSize, Aloc, nRow, nCol)

      write(*,'(A,2I5)') 'local matrix:', nRow, nCol
      Do i = 1, nRow
         write(*,'(100F10.6)') (Aloc(i, j), j = 1, nCol)
      End do

      label = 1
      Call FEM2Dtri(
     &     XYP(1, iP1), XYP(1, iP2), XYP(1, iP3), 
     &     DUDX, FEM_P1, IDEN, FEM_P1,
     &     label, Ddiff, DATA, 1, 
     &     MaxSize, Aloc, nRow, nCol)

      write(*,'(A,2I5)') 'local matrix:', nRow, nCol
      Do i = 1, nRow
         write(*,'(100F10.6)') (Aloc(i, j), j = 1, nCol)
      End do
c ... you my stop here if you want to test only elemental matrices
c     Stop


c ... draw initial mesh
      Call draw(nP, nF, nE, XYP, iW, IPF, IPE, 'ini.ps')


c ... adaptive loop
      Do iLoop = 1, nLOOPs
         Write(*, 5000) iLoop


c  ...   start the clock
         tm0 = ANItime(tmdata)

         status = IOR(MATRIX_GENERAL, FORMAT_AMG)


c  ...   assemble a matrix
         Call BilinearFormVolume(
     &        nP, nE, XYP, IPE, lbE,
     &        GRAD, FEM_P1, GRAD, FEM_P1,
     &        Ddiff, DATA, 1,
     &        status, MaxIA, MaxA,
     &                IA, JA, DA, A, nRow, nCol,
     &        MaxWi, iW)

         Write(*,'(A,3I5)') 'Bilinear form:', nP, nRow, nCol

         tmBF = ANItime(tmdata)


c  ...   assemble a right-hand side
         Call LinearFormVolume(
     &        nP, nE, XYP, IPE, lbE,
     &        FEM_P1, 
     &        Drhs_scalar, DATA, 1,
     &        F, nRow,
     &        MaxWi, iW)

         Write(*,'(A,2I5)') 'Linear form:  ', nP, nRow

         tmLF = ANItime(tmdata)


c  ...   set up boundary conditions
         Call BoundaryConditions(
     &        nP, nF, nE, XYP, IPF, IPE,
     &        FEM_P1, 
     &        Dbc, DATA,
     &        IA, JA, DA, A, F, nRow,
     &        MaxWi, iW)

         tmBC = ANItime(tmdata)


c  ...   AMG set up
         NDA  = MaxA
         NDU  = MaxIA
         NDF  = MaxIA
         NDIA = MaxIA
         NDJA = MaxA
         NDIG = MaxWi

         MATRIX = 12
         ISWTCH = 4
         IFIRST = 10
         IOUT   = 10
         IPRINT = 11616

         LEVELX = 10
         NCYC   = 122400
         MADAPT = 27
         NRD    = 1131
         NSOLCO = 110
         NRU    = 1131

         ECG1   = 0.
         ECG2   = 0.25
         EWT2   = 0.35
         NWT    = 2
         NTR    = 0

         eps = 1D-10

c  ...   initial guess
         Do n = 1, nRow
            U(n) = 0D0
            G(n) = F(n)
         End do


c  ...   solve the problem
         If(.FALSE.) Then
            Call aux1R5(A, IA, JA, U, F, iW,
     &                  NDA, NDIA, NDJA, NDU, NDF, NDIG, nRow, MATRIX,
     &                  eps, IFIRST, ISWTCH, IOUT, IPRINT,
     &                  IERR)

         Else
            Call amg1r6(A, IA, JA, U, F, iW,
     &                  NDA, NDIA, NDJA, NDU, NDF, NDIG, nRow, MATRIX,
     &                  ISWTCH, IOUT, IPRINT,
     &                  LEVELX, IFIRST, NCYC, eps, MADAPT, NRD, 
     &                  NSOLCO, NRU,ECG1, ECG2, EWT2, NWT, NTR,
     &                  IERR)
         End if
         If(IERR.NE.0) Call errMes(IERR, 'main', 
     &                     'error in one of the AMG routines')

         tmAMG = ANItime(tmdata)


c  ...   compute errors
         iExact = Dexact(XYP(1,1), XYP(2,1), 1, DATA, iSYS, G(1))
         If(iExact.NE.TENSOR_NULL) Then
            Call mulAgen(nRow, IA, JA, A, U, F)

            su = 0D0
            sr = 0D0
            se = 0D0
            Do n = 1, nRow
               su = su + U(n) ** 2
               sr = sr + (F(n) - G(n)) ** 2

               G(n) = dsin(pi * XYP(1, n)) * dsin(pi * XYP(2, n))
               iExact = Dexact(XYP(1,n), XYP(2,n), 1, DATA, iSYS, G(n))
               se = se + (U(n) - G(n)) ** 2
            End do

            Write(*, 5001) dsqrt(sr), dsqrt(se), dsqrt(se / su)
            Write(*, 5002) tmBF - tm0, tmLF - tmBF, tmAMG - tmBC
         End if


c  ...   draw solution isolines
         Call draw_S(U, nP, XYP, nE, IPE, nF, IPF, 'sol', 20)


c  ...   adapt mesh
         if(iLoop.EQ.nLOOPs) Goto 1000

         flagAuto = .TRUE.
         status = 0
         iPrint = 1

         Call mesh_solution(
c group (M)
     &        nP, MaxP, nF, MaxF, nE, MaxE, nPv,
     &        XYP, IPF, IPE, IPV,
     &        ParCrv, iFnc,
     &        nEStar, 
c group (D)
     &        nFv, nEv, IFV, IEV, lbE,
     &        flagAuto, status,
c group (Q)
     &        MaxSkipE, MaxQItr,
     &        U, Quality, rQuality,
c group (W)
     &        MaxWr, MaxWi, rW, iW,
     &        iPrint, iERR)

          If(iERR.GT.1000) Call errMes(iERR, 'main', 
     &                      'unspecified error if aniSolutionShort')
      End do


c ... save solution in Gnuplot format
 1000 Open(10, file='sol.gnu')
      Do n = 1, nP
         write(10,'(2F7.3,E12.4)') XYP(1, n), XYP(2, n), U(n)
      End do
      Close(10)


c ... draw final mesh
      Do n = 1, nP
         iW(n) = 0
      End do
      Call draw(nP, nF, nE, XYP, iW, IPF, IPE, 'fin.ps')


      Stop

 5000 Format(/, '===== ', I2, 'th ADAPTIVE LOOP =====')

 5001 Format(/,'Errors: PCG residual    R^n-norm   (relative norm)',/,
     &         E20.3, E12.3, E18.3)

 5002 Format('Times:        Matrix         RHS        AMG solver',/,
     &         E20.3, E12.3, E18.3)
      End



C ================================================================
      Subroutine CTIME(time)
C ================================================================
      Real   time, ANItime, tmdata(2)
      Real*8 tm

      time = ANItime(tmdata)

      Return
      End



C ==========================================================
C  Three Fortran routines below create a metric field which
C  is 2x2 variable positive definite symmetric tensor HesP,
C                     | F(x,y)  H(x,y) |
C      Metric(x, y) = |                |
C                     | H(x,y)  G(x,y) |
C
C  *** Remarks:
C         1. The uniform metric is given below. It results 
C            in a quasi-uniform grid.
C ==========================================================
      Real*8 Function F(x, y)
C ==========================================================
      Real*8 x, y
      F = 1D0
      Return
      End


C ==========================================================
      Real*8 Function G(x, y)
C ==========================================================
      Real*8 x, y
      G = 1D0
      Return
      End


C ==========================================================
      Real*8 Function H(x, y)
C ==========================================================
      Real*8 x, y
      H = 0D0
      Return
      End


C ==========================================================
      Subroutine calCrv(tc, xyc, iFnc)
C ==========================================================
C  The routine computes the Cartesian coordinates of point
C  xyc from its parametric coordinate tc.
C
C  tc     - the given parametric coordinate of point
C  xyc(2) - the Cartesian coordinate of the same point
C  iFnc   - the function number for computing
C
C  On input :  tc, iFnc
C  On output:  xyc(2)
C
C  *** Remarks:
C         1. There are 2 different parametrizations with
C            positive identificators (called function
C            numbers) 2 and 5.
C
C ==========================================================
      Real*8  tc, xyc(2)

C group (Local variables)
      Real*8  CX2, tt, xx, yy, w, XB
      Real*8  X1, X2, XE, Y2, C1, C2, C3, C4, C5, wingAngle

C ==========================================================
c .. closed profile of the wing model
      If(iFnc.EQ.2) Then
         C1 = 0.17735D0
         C2 =-0.075597D0
         C3 =-0.212836D0
         C4 = 0.17363D0
         C5 =-0.062547D0

         wingAngle = 0D0

         X1 = 0.45D0
         X2 = 0.55D0
         XE = 1D0
         Y2 = 5D-1

         X1 = 0.40D0
         X2 = 0.60D0

         CX2 = 5D-1 * (X1 + X2)

         tt = 2 * tc
         if(tt.GT.1D0) Then
            tt = tt - 1D0
            isgn = -1
         Else
            tt = 1D0 - tt
            isgn = 1
         End if

         xx = X1 + tt * (X2 - X1)
         yy = Y2 + isgn * (X2 - X1) * (C1 * dsqrt(tt) + C2 * tt +
     &        C3 * tt * tt + C4 * tt * tt * tt +
     &        C5 * tt * tt * tt * tt)

         xyc(1) = CX2 + (xx - CX2) * dcos(wingAngle) +
     &                  (yy - Y2)  * dsin(wingAngle)

         xyc(2) = Y2 - (xx - CX2) * dsin(wingAngle) +
     &                 (yy - Y2)  * dcos(wingAngle)

c ... tail line after the wing
      Else If(iFnc.EQ.5) Then
         wingAngle = 0D0

         X1 = 0.45D0
         X2 = 0.55D0
         XE = 1D0
         Y2 = 5D-1

         X1 = 0.40D0
         X2 = 0.60D0

         w = 5D-1 * (X2 - X1)
         XB = X2 - w * (1D0 - dcos(wingAngle))

         xyc(1) = XB + (XE - XB) * tc
         xyc(2) = Y2 + w * dsin(-wingAngle)
      Else
         Write(*, '(A,I5)') 'Undefined function =', iFnc
         Stop
      End if

      Return
      End







